package org.apache.thrift;

import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TMemoryBuffer;

import thrift.test.ComparableUnion;
import thrift.test.Empty;
import thrift.test.SomeEnum;
import thrift.test.StructWithAUnion;
import thrift.test.TestUnion;
import thrift.test.TestUnionMinusStringField;
import junit.framework.TestCase;

public class TestTUnion extends TestCase {

  public void testBasic() throws Exception {
    TestUnion union = new TestUnion();

    assertFalse(union.isSet());
    assertNull(union.getFieldValue());

    union = new TestUnion(TestUnion._Fields.I32_FIELD, 25);

    assertEquals(Integer.valueOf(25), union.getFieldValue());
  
    assertEquals(Integer.valueOf(25), union.getFieldValue(TestUnion._Fields.I32_FIELD));
  
    try {
      union.getFieldValue(TestUnion._Fields.STRING_FIELD);
      fail("should have thrown an exception");
    } catch (IllegalArgumentException e) {
      // cool!
    }

    union = new TestUnion();

    // should not throw an exception here
    union.hashCode();

    union.setI32_field(1);
    assertEquals(1, union.getI32_field());
    union.hashCode();

    try {
      union.getString_field();
      fail("should have thrown an exception");
    } catch (Exception e) {
      // sweet
    }

    union = TestUnion.i32_field(1);

    assertFalse(union.equals((TestUnion)null));

    union = TestUnion.enum_field(SomeEnum.ONE);
    union.hashCode();

    union = new TestUnion();
    // should not throw an exception
    union.toString();
  }

  public void testCompareTo() throws Exception {
    ComparableUnion cu = ComparableUnion.string_field("a");
    ComparableUnion cu2 = ComparableUnion.string_field("b");

    assertTrue(cu.compareTo(cu) == 0);
    assertTrue(cu2.compareTo(cu2) == 0);

    assertTrue(cu.compareTo(cu2) < 0);
    assertTrue(cu2.compareTo(cu) > 0);

    cu2 = ComparableUnion.binary_field(new byte[]{2});

    assertTrue(cu.compareTo(cu2) < 0);
    assertTrue(cu2.compareTo(cu) > 0);

    cu = ComparableUnion.binary_field(new byte[]{1});

    assertTrue(cu.compareTo(cu2) < 0);
    assertTrue(cu2.compareTo(cu) > 0);
  }

  public void testEquality() throws Exception {
    TestUnion union = new TestUnion(TestUnion._Fields.I32_FIELD, 25);

    TestUnion otherUnion = new TestUnion(TestUnion._Fields.STRING_FIELD, "blah!!!");

    assertFalse(union.equals(otherUnion));

    otherUnion = new TestUnion(TestUnion._Fields.I32_FIELD, 400);

    assertFalse(union.equals(otherUnion));

    otherUnion = new TestUnion(TestUnion._Fields.OTHER_I32_FIELD, 25);

    assertFalse(union.equals(otherUnion));
  }

  public void testSerialization() throws Exception {
    TestUnion union = new TestUnion(TestUnion._Fields.I32_FIELD, 25);

    TMemoryBuffer buf = new TMemoryBuffer(0);
    TProtocol proto = new TBinaryProtocol(buf);

    union.write(proto);

    TestUnion u2 = new TestUnion();

    u2.read(proto);

    assertEquals(u2, union);

    StructWithAUnion swau = new StructWithAUnion(u2);

    buf = new TMemoryBuffer(0);
    proto = new TBinaryProtocol(buf);

    swau.write(proto);

    StructWithAUnion swau2 = new StructWithAUnion();
    assertFalse(swau2.equals(swau));
    swau2.read(proto);
    assertEquals(swau2, swau);

    // this should NOT throw an exception.
    buf = new TMemoryBuffer(0);
    proto = new TBinaryProtocol(buf);

    swau.write(proto);
    new Empty().read(proto);
  }

  public void testSkip() throws Exception {
    TestUnion tu = TestUnion.string_field("string");
    byte[] tuSerialized = new TSerializer().serialize(tu);
    TestUnionMinusStringField tums = new TestUnionMinusStringField();
    new TDeserializer().deserialize(tums, tuSerialized);
    assertNull(tums.getSetField());
    assertNull(tums.getFieldValue());
  }
}
