// Copyright (c) 2006- Facebook
// Distributed under the Thrift Software License
//
// See accompanying file LICENSE or visit the Thrift site at:
// http://developers.facebook.com/thrift/

#ifndef T_OOP_GENERATOR_H
#define T_OOP_GENERATOR_H

#include <sstream>
#include <string>
#include <fstream>
#include <iostream>

#include "globals.h"
#include "t_generator.h"

#include <algorithm>

/**
 * Class with utility methods shared across common object oriented languages.
 * Specifically, most of this stuff is for C++/Java.
 *
 */
class t_oop_generator : public t_generator {
 public:
  t_oop_generator(t_program* program) :
    t_generator(program) {}

  /**
   * Scoping, using curly braces!
   */

  void scope_up(std::ostream& out) {
    indent(out) << "{" << std::endl;
    indent_up();
  }

  void scope_down(std::ostream& out) {
    indent_down();
    indent(out) << "}" << std::endl;
  }

  std::string upcase_string(std::string original) {
    std::transform(original.begin(), original.end(), original.begin(), (int(*)(int)) toupper);
    return original;
  }

  void generate_docstring_comment(std::ofstream& out,
                                  std::string comment_start,
                                  std::string line_prefix,
                                  std::string contents,
                                  std::string comment_end) {
    if (comment_start != "") indent(out) << comment_start;
    std::stringstream docs(contents, std::ios_base::in);
    while (!docs.eof()) {
      char line[1024];
      docs.getline(line, 1024);
      if (strlen(line) > 0 || !docs.eof()) {  // skip the empty last line
        indent(out) << line_prefix << line << std::endl;
      }
    }
    if (comment_end != "") indent(out) << comment_end;
  }

  /**
   * Generates a comment about this code being autogenerated, using C++ style
   * comments, which are also fair game in Java / PHP, yay!
   *
   * @return C-style comment mentioning that this file is autogenerated.
   */
  virtual std::string autogen_comment() {
    return
      std::string("/**\n") +
      " * Autogenerated by Thrift\n" +
      " *\n" +
      " * DO NOT EDIT UNLESS YOU ARE SURE THAT YOU KNOW WHAT YOU ARE DOING\n" +
      " */\n";
  }
};

#endif

