package com.supwisdom.dlpay.system.service.impl;

import com.supwisdom.dlpay.framework.dao.OperLogDao;
import com.supwisdom.dlpay.framework.dao.OperRoleDao;
import com.supwisdom.dlpay.framework.dao.OperatorDao;
import com.supwisdom.dlpay.framework.domain.TOperLog;
import com.supwisdom.dlpay.framework.domain.TOperRole;
import com.supwisdom.dlpay.framework.domain.TOperator;
import com.supwisdom.dlpay.framework.util.*;
import com.supwisdom.dlpay.system.bean.LogBean;
import com.supwisdom.dlpay.system.bean.OperatorSearchBean;
import com.supwisdom.dlpay.system.service.OperatorService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Service
public class OperatorServiceImpl implements OperatorService {
    @Autowired
    private OperatorDao operatorDao;
    @Autowired
    private OperRoleDao operRoleDao;
    @Autowired
    private OperLogDao operLogDao;

    @Override
    public PageResult<TOperator> getOperatorsByPage(int pageNo, int pageSize, OperatorSearchBean searchBean) {
        Pageable pageable = PageRequest.of(pageNo - 1, pageSize, Sort.Direction.DESC, "opendate");
        Page<TOperator> page = operatorDao.findAll(new Specification<TOperator>() {
            @Override
            public Predicate toPredicate(Root<TOperator> root, CriteriaQuery<?> query, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<>();
                if (!StringUtil.isEmpty(searchBean.getOpercode())) {
                    predicates.add(criteriaBuilder.equal(root.get("opercode").as(String.class), searchBean.getOpercode().trim()));
                }
                if (!StringUtil.isEmpty(searchBean.getOpername())) {
                    predicates.add(criteriaBuilder.like(root.get("opername").as(String.class), "%" + searchBean.getOpername().trim() + "%"));
                }
                if (!StringUtil.isEmpty(searchBean.getMobile())) {
                    predicates.add(criteriaBuilder.like(root.get("mobile").as(String.class), "%" + searchBean.getMobile().trim() + "%"));
                }
                if (null != searchBean.getDeleteShow() && !searchBean.getDeleteShow()) {
                    predicates.add(criteriaBuilder.equal(root.get("status").as(String.class), TradeDict.STATUS_NORMAL)); //仅显示正常
                }
                return criteriaBuilder.and(predicates.toArray(new Predicate[0]));
            }
        }, pageable);

        return new PageResult<>(page);
    }

    @Override
    public boolean checkOpercodeExists(String opercode, String operid) {
        TOperator oper = operatorDao.findByOpercode(opercode.trim());
        if (null != oper && StringUtil.isEmpty(operid)) return true;
        if (null != oper && !StringUtil.isEmpty(operid) && !operid.trim().equals(oper.getOperid())) return true;
        return false;
    }

    @Override
    public boolean saveOperator(TOperator operator, String roles) {
        if (StringUtil.isEmpty(operator.getOperid())) {
            String[] roleids = roles.split(",");
            operator.setOpendate(DateUtil.getNow("yyyyMMdd"));
            operator.setStatus(TradeDict.STATUS_NORMAL);
            operator.setOpertype(WebConstant.OPERTYPE_DEFAULT);
            BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
            operator.setOperpwd(encoder.encode(WebConstant.OPERPWD_DEFAULT));
            operator = operatorDao.save(operator);
            for (String role : roleids) {
                TOperRole operRole = new TOperRole();
                operRole.setOperid(operator.getOperid());
                operRole.setRoleId(role);
                operRoleDao.save(operRole);
            }
        } else {
            Optional<TOperator> temp = operatorDao.findById(operator.getOperid());
            if (!temp.isPresent()) {
                return false;
            }
            TOperator op = temp.get();
            op.setSex(operator.getSex());
            op.setEmail(operator.getEmail());
            op.setMobile(operator.getMobile());
            op.setOpername(operator.getOpername());
            op.setOpercode(operator.getOpercode());
            operatorDao.save(op);
            operRoleDao.deleteByOperid(op.getOperid());
            String[] roleids = roles.split(",");
            for (String role : roleids) {
                TOperRole operRole = new TOperRole();
                operRole.setOperid(op.getOperid());
                operRole.setRoleId(role);
                operRoleDao.save(operRole);
            }
        }
        return true;
    }

    @Override
    public TOperator getOperatorByCode(String opercode) {
        return operatorDao.findByOpercode(opercode);
    }

    @Override
    public List<TOperRole> getOperRoleByOperId(String operid) {
        return operRoleDao.findAllByOperid(operid);
    }

    @Override
    public boolean resetPwd(String operid) {
        Optional<TOperator> temp = operatorDao.findById(operid);
        if (!temp.isPresent()) {
            return false;
        }
        TOperator op = temp.get();
        BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
        op.setOperpwd(encoder.encode(WebConstant.OPERPWD_DEFAULT));
        operatorDao.save(op);
        return true;
    }

    @Override
    public boolean updateState(String operid, String state) {
        Optional<TOperator> temp = operatorDao.findById(operid);
        if (!temp.isPresent()) {
            return false;
        }
        TOperator op = temp.get();
        op.setStatus(state);
        operatorDao.save(op);
        return true;
    }

    @Override
    public boolean saveOperLog(TOperLog log) {
        operLogDao.save(log);
        return true;
    }

    @Override
    public PageResult<TOperLog> getOperLogs(LogBean param) {
        Pageable pageable = PageRequest.of(param.getPageNo() - 1, param.getPageSize()
                , Sort.by(Sort.Direction.DESC,"logdate"));
        if (!StringUtil.isEmpty(param.getOpername())) {
            return new PageResult<>(operLogDao.findAllByOpernameContaining(param.getOpername(), pageable));
        }
        return new PageResult<>(operLogDao.findAll(pageable));
    }

    @Override
    public boolean saveOperator(TOperator operator) {
        Optional<TOperator> temp = operatorDao.findById(operator.getOperid());
        if (!temp.isPresent()) {
            return false;
        }
        operatorDao.save(operator);
        return true;
    }
}
