package com.supwisdom.dlpay.api

import com.supwisdom.dlpay.api.domain.TAccount
import com.supwisdom.dlpay.api.domain.TTransactionMain
import com.supwisdom.dlpay.api.service.TransactionService
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.framework.domain.TShopacc
import com.supwisdom.dlpay.framework.domain.TSubject
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.framework.util.TradeErrorCode

open class SubTransactionBuilder<T : SubTransactionBuilder<T>>(val parent: TransactionBuilder) {
    var payinfo: String = ""
        get() {
            return if (field.isEmpty()) parent.payinfo else field
        }

    var description: String = ""
        get() {
            return if (field.isEmpty()) parent.description else field
        }

    var remark: String = ""
        get() {
            return if (field.isEmpty()) parent.remark else field
        }

    lateinit var opposite: AccountProxy<*>

    fun hasOpposite(): Boolean {
        return this::opposite.isInitialized
    }

    fun and(): TransactionBuilder {
        return parent
    }

    // 收支方向, in / out
    var tradeFlag: String = ""
        private set

    var amount: Double = 0.0
        private set

    /**
     * amount : 交易金额， 正向交易金额必须 >= 0, 冲正交易金额必须 <= 0
     */
    fun setAmount(amount: Double, inOut: String): SubTransactionBuilder<T> {
        this.amount = amount
        this.tradeFlag = inOut
        return this
    }

    /**
     * amount : 交易金额， 正向交易金额必须 >= 0, 冲正交易金额必须 <= 0
     */
    fun addAmount(amount: Double): SubTransactionBuilder<T> {
        this.amount += amount
        return this
    }

}

class PersonTranactionBuilder(parent: TransactionBuilder, val person: TAccount)
    : SubTransactionBuilder<PersonTranactionBuilder>(parent) {
}

class ShopTransactionBuilder(parent: TransactionBuilder, val shopacc: TShopacc)
    : SubTransactionBuilder<ShopTransactionBuilder>(parent) {
}

class SubjectTransactionBuilder(parent: TransactionBuilder, val subject: TSubject)
    : SubTransactionBuilder<SubjectTransactionBuilder>(parent) {
}

class AccountProxy<T>(private val accountObj: T) {
    fun getAccountNo(): String {
        return when (accountObj) {
            is TAccount -> (accountObj as TAccount).accno
            is TShopacc -> (accountObj as TShopacc).shopaccno
            is TSubject -> (accountObj as TSubject).subjno
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "不支持的账户类型, $accountObj")
        }
    }

    fun getSubjectNo(): String {
        return when (accountObj) {
            is TAccount -> (accountObj as TAccount).subjno
            is TShopacc -> (accountObj as TShopacc).subjno
            is TSubject -> (accountObj as TSubject).subjno
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "不支持的账户类型, $accountObj")
        }
    }

    fun getAccountName(): String {
        return when (accountObj) {
            is TAccount -> (accountObj as TAccount).accname
            is TShopacc -> (accountObj as TShopacc).shopname
            is TSubject -> (accountObj as TSubject).subjname
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "不支持的账户类型, $accountObj")
        }
    }

    fun get(): T {
        return accountObj
    }
}

data class DebitCreditLine(val debit: AccountProxy<*>, val credit: AccountProxy<*>,
                           val amount: Double, val summary: String,
                           val seqno: Int)

// 在 constructor 中加入 service 接口
class TransactionBuilder {
    companion object {
        private val VALID_REVERSEFLAG = setOf(TradeDict.REVERSE_FLAG_CANCEL,
                TradeDict.REVERSE_FLAG_REVERSE,
                TradeDict.REVERSE_FLAG_NONE)

        private val VALID_OPERTYPE = setOf(TradeDict.OPERTYPE_OPER,
                TradeDict.OPERTYPE_SHOP, TradeDict.OPERTYPE_PERSON)

        private val VALID_TRADEFLAG = setOf(TradeDict.TRADE_FLAG_IN,
                TradeDict.TRADE_FLAG_OUT)
    }

    private lateinit var personBuilder: PersonTranactionBuilder
    private lateinit var shopBuilder: ShopTransactionBuilder
    private lateinit var subjectBuilder: SubjectTransactionBuilder

    private val debitCreditLines = mutableListOf<DebitCreditLine>()

    // 以下是交易主表属性
    var transDate: String = ""
        private set
    var transTime: String = ""
        private set
    var transCode: Int = 0 //交易码,各明细流水统一
        private set

    // 资金来源方案
    var sourceType: String = ""
        private set

    fun setTransInfo(date: String, time: String, code: Int, sourceType: String): TransactionBuilder {
        this.transDate = date
        this.transTime = time
        this.transCode = code
        this.sourceType = sourceType
        return this
    }

    var outtradeno: String = "" //第三方流水号
        private set
    var outId: String = ""
        private set

    fun setOutTransInfo(id: String, tradeno: String): TransactionBuilder {
        this.outId = id
        this.outtradeno = tradeno
        return this
    }

    var operId: String = ""
        private set
    var operType: String = ""
        private set

    fun operator(id: String, type: String): TransactionBuilder {
        this.operId = id
        this.operType = type
        return this
    }

    var tradeType: String = "none"
        private set

    fun reverse(): TransactionBuilder {
        this.tradeType = TradeDict.REVERSE_FLAG_REVERSE
        return this
    }

    fun cancel(): TransactionBuilder {
        this.tradeType = TradeDict.REVERSE_FLAG_CANCEL
        return this
    }


    // 以下属性可以在子表中不同
    var payinfo: String = ""
    var description: String = ""
    var remark: String = ""

    fun person(): PersonTranactionBuilder {
        return this.personBuilder
    }

    fun person(account: TAccount): PersonTranactionBuilder {
        return if (!this::personBuilder.isInitialized) {
            PersonTranactionBuilder(this, account).also {
                this.personBuilder = it
            }
        } else {
            this.personBuilder
        }
    }

    fun hasPerson(): Boolean {
        return this::personBuilder.isInitialized
    }

    fun shop(): ShopTransactionBuilder {
        return this.shopBuilder
    }

    fun shop(shopacc: TShopacc): ShopTransactionBuilder {
        return if (!this::shopBuilder.isInitialized) {
            ShopTransactionBuilder(this, shopacc).also {
                this.shopBuilder = it
            }
        } else {
            this.shopBuilder
        }
    }

    fun hasShop(): Boolean {
        return this::shopBuilder.isInitialized
    }

    fun subject(acc: TSubject): SubjectTransactionBuilder {
        return if (!this::subjectBuilder.isInitialized) {
            SubjectTransactionBuilder(this, acc).also {
                this.subjectBuilder = it
            }
        } else {
            this.subjectBuilder
        }
    }

    fun subject(): SubjectTransactionBuilder {
        return this.subjectBuilder
    }

    fun hasSubject(): Boolean {
        return this::subjectBuilder.isInitialized
    }

    fun <T, U> addDebitCreditRecord(debit: AccountProxy<T>, credit: AccountProxy<U>,
                                    amount: Double, summary: String): TransactionBuilder {
        debitCreditLines.add(DebitCreditLine(debit, credit, amount, summary,
                debitCreditLines.size + 1))
        return this
    }

    fun getAllDetails(): List<DebitCreditLine> {
        return debitCreditLines.toList()
    }

    private fun checkAmount(amount: Double): Boolean {
        return if (tradeType == TradeDict.REVERSE_FLAG_NONE) {
            amount >= 0.0
        } else {
            amount <= 0.0
        }
    }

    fun preCheck() {
        if (outId.isNotEmpty() && outtradeno.isEmpty()) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "第三方账号未指定第三方流水号")
        }

        if (transDate.length != 8 || transTime.length != 6) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "交易发生日期错误")
        }

        if (transCode <= 0) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "交易码错误")
        }

        if (tradeType !in VALID_REVERSEFLAG) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "冲正标志错误")
        }

        if ((operId.isNotEmpty() || operType.isNotEmpty())
                && operType !in VALID_OPERTYPE) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "操作员类型错误")
        }

        if (sourceType.isEmpty()) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "交易源类型错误")
        }
        if (hasPerson()) {
            person().also {
                if (it.tradeFlag !in VALID_TRADEFLAG) {
                    throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "个人交易收支方向错误")
                }
                if (!checkAmount(it.amount)) {
                    throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "个人交易金额不正确")
                }
                if (!it.hasOpposite()) {
                    throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "个人交易对方账户未设置")
                }
                if (it.payinfo.isEmpty()) {
                    throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "个人交易信息未设置")
                }
            }
        }

        if (hasShop()) {
            shop().also {
                if (it.tradeFlag !in VALID_TRADEFLAG) {
                    throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "商户交易收支方向错误")
                }
                if (!checkAmount(it.amount)) {
                    throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "商户交易金额不正确")
                }
                if (!it.hasOpposite()) {
                    throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "商户交易对方账户未设置")
                }
            }
        }
    }

    fun init(transactionService: TransactionService): TTransactionMain {
        return transactionService.init(this)
    }

    fun wip(transactionService: TransactionService): TTransactionMain {
        return transactionService.wip(this)
    }

    fun cancel(transactionService: TransactionService, originRefno: String): TTransactionMain {
        TODO("not implement")
    }

    fun reverse(transactionService: TransactionService, originRefno: String, amount: Double): TTransactionMain {
        TODO("not implement")
    }
}
