package com.supwisdom.dlpay.api

import com.supwisdom.dlpay.api.domain.TAccount
import com.supwisdom.dlpay.api.domain.TTransactionMain
import com.supwisdom.dlpay.api.service.AccountUtilServcie
import com.supwisdom.dlpay.api.service.TransactionService
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.framework.domain.TShopacc
import com.supwisdom.dlpay.framework.domain.TSubject
import com.supwisdom.dlpay.framework.util.Subject
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.framework.util.TradeErrorCode

open class SubTransactionBuilder(val parent: TransactionBuilder) {
    var paytype: String = ""
        get() {
            return if (field.isEmpty()) parent.paytype else field
        }

    var payinfo: String = ""
        get() {
            return if (field.isEmpty()) parent.payinfo else field
        }


    lateinit var opposite: AccountProxy<*>

    fun hasOpposite(): Boolean {
        return this::opposite.isInitialized
    }

    fun and(): TransactionBuilder {
        return parent
    }

    private var balanceInOut: String = ""

    fun tradeIn(): SubTransactionBuilder {
        this.balanceInOut = TradeDict.TRADE_FLAG_IN
        return this
    }

    fun tradeOut(): SubTransactionBuilder {
        this.balanceInOut = TradeDict.TRADE_FLAG_OUT
        return this
    }

    fun tradeFlag(): String {
        return this.balanceInOut
    }


}

class PersonTranactionBuilder(parent: TransactionBuilder, val person: TAccount)
    : SubTransactionBuilder(parent) {
    var amount: Double = 0.0
    var summary: String = ""
    var description: String = ""

}

class ShopTransactionBuilder(parent: TransactionBuilder, val shopacc: TShopacc)
    : SubTransactionBuilder(parent) {
    var amount: Double = 0.0
    var summary: String = ""
}

class SubjectTransactionBuilder(parent: TransactionBuilder, val subject: TSubject)
    : SubTransactionBuilder(parent) {
    var amount: Double = 0.0
    var summary: String = ""

}

class AccountProxy<T>(private val accountObj: T) {
    fun getAccountNo(): String {
        return when (accountObj) {
            is TAccount -> (accountObj as TAccount).accno
            is TShopacc -> (accountObj as TShopacc).shopaccno
            is TSubject -> (accountObj as TSubject).subjno
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "不支持的账户类型, $accountObj")
        }
    }

    fun getSubjectNo(): String {
        return when (accountObj) {
            is TAccount -> (accountObj as TAccount).subjno
            is TShopacc -> (accountObj as TShopacc).subjno
            is TSubject -> (accountObj as TSubject).subjno
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "不支持的账户类型, $accountObj")
        }
    }

    fun getAccountName(): String {
        return when (accountObj) {
            is TAccount -> (accountObj as TAccount).accname
            is TShopacc -> (accountObj as TShopacc).shopname
            is TSubject -> (accountObj as TSubject).subjname
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "不支持的账户类型, $accountObj")
        }
    }

    fun get(): T {
        return accountObj
    }
}

data class DebitCreditLine(val debit: AccountProxy<*>, val credit: AccountProxy<*>,
                           val amount: Double, val summary: String,
                           val seqno: Int)

// 在 constructor 中加入 service 接口
class TransactionBuilder {
    private lateinit var personBuilder: PersonTranactionBuilder
    private lateinit var shopBuilder: ShopTransactionBuilder
    private lateinit var subjectBuilder: SubjectTransactionBuilder

    private val debitCreditLines = mutableListOf<DebitCreditLine>()

    var transDate: String = ""
    var transTime: String = ""
    var transCode: Int = 0 //交易码,各明细流水统一
    var refno: String = ""
    var paytype: String = ""
    var payinfo: String = ""
    var outtradeno: String = "" //第三方流水号
    var outId: String = ""
    var reverseFlag: String = "none"

    fun person(): PersonTranactionBuilder {
        return this.personBuilder
    }

    fun person(account: TAccount): PersonTranactionBuilder {
        return if (!this::personBuilder.isInitialized) {
            PersonTranactionBuilder(this, account).also {
                this.personBuilder = it
            }
        } else {
            this.personBuilder
        }
    }

    fun hasPerson(): Boolean {
        return this::personBuilder.isInitialized
    }

    fun shop(): ShopTransactionBuilder {
        return this.shopBuilder
    }

    fun shop(shopacc: TShopacc): ShopTransactionBuilder {
        return if (!this::shopBuilder.isInitialized) {
            ShopTransactionBuilder(this, shopacc).also {
                this.shopBuilder = it
            }
        } else {
            this.shopBuilder
        }
    }

    fun hasShop(): Boolean {
        return this::shopBuilder.isInitialized
    }

    fun subject(acc: TSubject): SubjectTransactionBuilder {
        return if (!this::subjectBuilder.isInitialized) {
            SubjectTransactionBuilder(this, acc).also {
                this.subjectBuilder = it
            }
        } else {
            this.subjectBuilder
        }
    }

    fun subject(): SubjectTransactionBuilder {
        return this.subjectBuilder
    }

    fun hasSubject(): Boolean {
        return this::subjectBuilder.isInitialized
    }

    fun <T, U> addDebitCreditRecord(debit: AccountProxy<T>, credit: AccountProxy<U>,
                                    amount: Double, summary: String): TransactionBuilder {
        debitCreditLines.add(DebitCreditLine(debit, credit, amount, summary,
                debitCreditLines.size + 1))
        return this
    }

    fun getAllDetails(): List<DebitCreditLine> {
        return debitCreditLines.toList()
    }

    fun init(transactionService: TransactionService): TTransactionMain {
        return transactionService.init(this)
    }

    fun wip(transactionService: TransactionService): TTransactionMain {
        return transactionService.wip(this)
    }
}
