package com.supwisdom.dlpay.api.controller

import com.supwisdom.dlpay.api.AccountHolder
import com.supwisdom.dlpay.api.CallService
import com.supwisdom.dlpay.api.PersonTransBuilder
import com.supwisdom.dlpay.api.bean.CitizenCardPayfinishParam
import com.supwisdom.dlpay.api.bean.CitizenCardPayinitParam
import com.supwisdom.dlpay.api.bean.QueryDtlResultParam
import com.supwisdom.dlpay.api.service.*
import com.supwisdom.dlpay.exception.RequestParamCheckException
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.service.CommonService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.util.ConstantUtil
import com.supwisdom.dlpay.util.PaytypeUtil
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.*
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@RestController
@RequestMapping("/api/consume")
class ConsumeController {
    @Autowired
    lateinit var accountUtilServcie: AccountUtilServcie
    @Autowired
    lateinit var personBalancePayService: PersonBalancePayService
    @Autowired
    lateinit var paytypeService: PaytypeService
    @Autowired
    lateinit var userService: UserService
    @Autowired
    lateinit var systemUtilService: SystemUtilService
    @Autowired
    lateinit var consumePayService: ConsumePayService
    @Autowired
    lateinit var commonService: CommonService

    /**
     * 流水结果查询统一接口
     * */
    fun queryDtlResult(@RequestBody param: QueryDtlResultParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val dtl = if (StringUtil.isEmpty(param.refno)) {
                personBalancePayService.getUserdtlForUpdateNowait(param.refno!!)
            } else {
                personBalancePayService.getUserdtlByBillnoForUpdateNowait(param.billno!!, param.shopaccno!!)
            }
            val person = dtl.userid?.let { userService.findPersonByUserid(dtl.userid) }
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data("refno", dtl.refno)
                    .data("amount", dtl.amount)
                    .data("paytype", dtl.paytype)
                    .data("payinfo", dtl.payinfo)
                    .also {
                        if (null != person) {
                            it.data("name", person.name)
                        }
                    }
                    .success("查询成功"))

        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "查询错误"))
        }

    }

    /**
     * 市民卡交易初始化
     * */
    @PostMapping("/citizencard/payinit")
    fun citizencardPayinit(@RequestBody param: CitizenCardPayinitParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val person = userService.findPersonByIdentityCheckStatus(param.cardNo)
            if (consumePayService.checkShopPaytype(param.shopaccno, TradeDict.PAYTYPE_CITIZEN_CARD)) {
                val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                        .chooseTradetype(Tradetype.CONSUME) //消费
                        .setOwner(person) //记名
                        .setTransinfo(TradeCode.TRANSCODE_PAY, "支付交易")
                        .setTransDatetime(param.transdate, param.transtime) //交易时间
                        .selectPaytype(TradeDict.PAYTYPE_CITIZEN_CARD, param.cardNo)
                        .addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_CITIZEN_CARD),
                                AccountHolder.shop(param.shopaccno),
                                param.amount / 100.0, "市民卡代扣消费")
                        .also { builder ->
                            param.feelist?.forEach {
                                builder.addDetail(AccountHolder.feetype(it.feetype, TradeDict.PAYTYPE_CITIZEN_CARD)
                                        .with(AccountHolder.shop(param.shopaccno))
                                        .with(AccountHolder.subject(Subject.SUBJNO_PAY_CITIZEN_CARD))
                                        , it.amount / 100.0)
                            }
                        }
                        .init(personBalancePayService)

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("amount", dtl.amount)
                        .success("交易初始化成功"))
            }

            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_PAYTYPE_NOSUPPORT, "不支持支付方式<市民卡代扣>"))

        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

    /**
     * 市民卡交易确认
     * */
    @PostMapping("/citizencard/payfinish")
    fun citizencardPayinit(@RequestBody param: CitizenCardPayfinishParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            var dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .setRefno(param.refno)
                    .wip(personBalancePayService)
            val code = CallService.CallCitizenCardPay(consumePayService.getPaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, "shopaccno"), dtl)
            if (code.retcode == "0") {
                dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                        .setRefno(param.refno)
                        .addResult(code.data)
                        .success(personBalancePayService) //流水置成功

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("billno", dtl.outtradeno)
                        .success())
            } else {
                PersonTransBuilder.newBuilder(accountUtilServcie)
                        .setRefno(param.refno)
                        .addResult("errmsg", code.retmsg!!)
                        .finish(personBalancePayService, TradeDict.DTL_STATUS_FAIL) //流水置成功

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.BUSINESS_DEAL_ERROR, "交易扣费失败-${code.retmsg}"))
            }

        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

    /**
     * 账户余额支付
     * */
    @PostMapping("/balance/pay")
    fun balancePay(@RequestBody param: CitizenCardPayinitParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val person = userService.findPersonByIdentityCheckStatus(param.cardNo)
            if (consumePayService.checkShopPaytype(param.shopaccno, TradeDict.PAYTYPE_BALANCE)) {
                val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                        .chooseTradetype(Tradetype.CONSUME) //消费
                        .setOwner(person) //记名
                        .setTransinfo(TradeCode.TRANSCODE_PAY, "支付交易")
                        .setTransDatetime(param.transdate, param.transtime) //交易时间
                        .selectPaytype(TradeDict.PAYTYPE_BALANCE, param.cardNo)
                        .addDetail(AccountHolder.person(person.userid),
                                AccountHolder.shop(param.shopaccno),
                                param.amount / 100.0, "账户余额消费")
                        .also { builder ->
                            param.feelist?.forEach {
                                builder.addDetail(AccountHolder.feetype(it.feetype, TradeDict.PAYTYPE_BALANCE)
                                        .with(AccountHolder.shop(param.shopaccno))
                                        .with(AccountHolder.person(person.userid))
                                        , it.amount / 100.0)
                            }
                        }
                        .done(personBalancePayService)

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("amount", dtl.amount)
                        .success("交易成功"))
            }

            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_PAYTYPE_NOSUPPORT, "不支持支付方式<账户余额>"))

        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

    /**
     * 一卡通支付
     * */
    @PostMapping("/ykt/payinit")
    fun yktPayInit(@RequestBody param: CitizenCardPayinitParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            var person = userService.findByThirdUniqueIdenty(param.cardNo) //可能匿名?
            if (consumePayService.checkShopPaytype(param.shopaccno, TradeDict.PAYTYPE_YKT_CARD, person==null)) {
                val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                        .chooseTradetype(Tradetype.CONSUME) //消费
                        .also {
                            if(null!=person) it.setOwner(person)
                        }
                        .setTransinfo(TradeCode.TRANSCODE_PAY, "支付交易")
                        .setTransDatetime(param.transdate, param.transtime) //交易时间
                        .selectPaytype(TradeDict.PAYTYPE_YKT_CARD, param.cardNo)
                        .addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                AccountHolder.shop(param.shopaccno),
                                param.amount / 100.0, "一卡通支付")
                        .also { builder ->
                            param.feelist?.forEach {
                                builder.addDetail(AccountHolder.feetype(it.feetype, TradeDict.PAYTYPE_YKT_CARD)
                                        .with(AccountHolder.shop(param.shopaccno))
                                        .with(AccountHolder.subject(Subject.SUBJNO_PAY_YKT))
                                        , it.amount / 100.0)
                            }
                        }
                        .init(personBalancePayService)

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("amount", dtl.amount)
                        .success("交易初始化成功"))


            }

            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_PAYTYPE_NOSUPPORT, "不支持支付方式<一卡通支付>"))
        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

// ============================================== //
//
//    @GetMapping("/account/payinit")
//    fun accountPayInit(userid: String, amount: Int, manageFee: Int): ResponseEntity<Any> {
//        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
//                .setTransDatetime("20190411", "112311")
//                .enableOverdraft(false)
//                .addDetail(AccountHolder.person(userid),
//                        AccountHolder.shop("12323"),
//                        amount / 100.0, "")
//                .addDetail(AccountHolder.person(userid), AccountHolder.transType(301),
//                        manageFee / 100.0, "")
//                .done(personBalancePayService, false)
//
//        return ResponseEntity.ok(dtl)
//    }
//
//    @GetMapping("/account/payfinish")
//    fun accountPayFinish(refno: String): ResponseEntity<Any> {
//        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
//                .done(refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
//        return ResponseEntity.ok(dtl)
//    }


//
//    @PostMapping("/ykt/payfinish")
//    fun yktPayFinish(refno: String, yktshopid: String, devphyid: String?): ResponseEntity<Any> {
//        return try {
//            val dtl = personBalancePayService.wip(refno)
//            val person = userService.findPersonByUserid(dtl.userid)
//            val code = CallService.callYktPay(paytypeService.getPaytypeConfigByPaytype(PaytypeUtil.YKTPAY),
//                    dtl, DateUtil.getNow(), "", yktshopid, devphyid)
//            if (code.retcode == "0") {
//                PersonTransBuilder.newBuilder(accountUtilServcie)
//                        .done(dtl.refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .data("refno", dtl.refno)
//                        .success())
//            } else {
//                PersonTransBuilder.newBuilder(accountUtilServcie)
//                        .done(dtl.refno, TradeDict.DTL_STATUS_FAIL, personBalancePayService)
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .fail(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易请求失败-${code.retcode}"))
//            }
//        } catch (e: TransactionException) {
//            ResponseEntity.ok(ResponseBodyBuilder.create()
//                    .transException(e, "交易确认失败"))
//        }
//    }
//
//    /**
//     * 微信支付
//     * wechattype
//     * qrcode-扫微信二维码支付
//     * app-原生app微信支付
//     * mp-微信公众号支付
//     * h5-微信h5支付
//     *
//     * */
//    @PostMapping("/wechat/payinit")
//    fun wechatPayInit(userid: String, amount: Int, manageFee: Int,
//                      stuempno: String, shopid: String, transdate: String, transtime: String,
//                      outtradeno: String, payinfo: String, feetype: String,
//                      wechattype: String, realip: String?, qrcode: String?, openid: String?): ResponseEntity<Any> {
//        return try {
//            val paytype = paytypeService.getByPaytype(PaytypeUtil.WECHAT)
//            if (paytype == null || ConstantUtil.ENABLE_YES != paytype.enable) {
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .fail(1, "支付方式未开启"))
//            }
//            val person = userService.findByThirdUniqueIdenty(stuempno)
//            val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
//                    .setTransDatetime(transdate, transtime)
//                    .selectPaytype(PaytypeUtil.WECHAT, payinfo)
//                    .setOuttradeno(outtradeno)
//                    .also {
//                        if (null != person) it.setOwner(person)
//                    }
//                    .tryLock(true)
//                    .setTransinfo(TradeCode.TRANSCODE_WECHAT, "微信支付")
//                    .chooseTradetype(Tradetype.CONSUME)
//                    .also {
//                        when (feetype) {
//                            TradeDict.FEETYPE_CONSUME_MEALER -> {
//                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
//                                        AccountHolder.shop(shopid),
//                                        amount / 100.0, "微信支付")
//                                        .addDetail(AccountHolder.transType(TranstypeCode.TT_CONSUUME_MANAGE_FEE)
//                                                .with(AccountHolder.shop(shopid)),
//                                                manageFee / 100.0)
//                            }
//                            TradeDict.FEETYPE_CONSUME_DISCOUNT -> {
//                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
//                                        AccountHolder.shop(shopid),
//                                        (amount - manageFee) / 100.0, "微信支付")
//                                        .addDetail(AccountHolder.subject(Subject.SUBJNO_CONSUME_DISCOUNT),
//                                                AccountHolder.shop(shopid),
//                                                manageFee / 100.0, "优惠折扣")
//                            }
//                            else -> {
//                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
//                                        AccountHolder.shop(shopid),
//                                        amount / 100.0, "微信支付")
//                            }
//                        }
//                    }.done(personBalancePayService, false)
//            val code = CallService.callWechatPay(paytypeService.getPaytypeConfigByPaytype(PaytypeUtil.WECHAT),
//                    dtl, DateUtil.getNow(), wechattype, realip, qrcode, openid)
//            if (code.retcode == "0") {
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .data("refno", dtl.refno)
//                        .success())
//            } else {
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .fail(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易请求失败-${code.retcode}"))
//            }
//        } catch (e: TransactionException) {
//            ResponseEntity.ok(ResponseBodyBuilder.create()
//                    .transException(e, "交易初始化异常"))
//        }
//    }
}