package com.supwisdom.dlpay.api.controller

import com.supwisdom.dlpay.api.bean.OpenShopParam
import com.supwisdom.dlpay.api.bean.QueryShopParam
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.api.service.ShopService
import com.supwisdom.dlpay.exception.RequestParamCheckException
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.framework.service.CommonService
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.*
import java.net.URLDecoder
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@RestController
@RequestMapping("/api/shop")
class ShopAPIController {
    @Autowired
    private lateinit var shopService: ShopService
    @Autowired
    private lateinit var commonService: CommonService

    @PostMapping("/open")
    fun openAccount(@RequestBody param: OpenShopParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }
            shopService.findByThirdUniqueId(param.shopUniqueId)?.let {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.BUSINESS_SHOP_EXISTS, "该商户唯一号[${param.shopUniqueId}]已经存在"))
            }

            param.shopname = param.shopname?.let { URLDecoder.decode(param.shopname, Charsets.UTF_8.toString()) } //解码
            param.contactman = param.contactman?.let { URLDecoder.decode(param.contactman, Charsets.UTF_8.toString()) } //解码
            param.address = param.address?.let { URLDecoder.decode(param.address, Charsets.UTF_8.toString()) } //解码
            shopService.registerShop(param)?.let {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("shopid", it.shopid)
                        .data("shopaccno", it.shopaccno ?: "")
                        .success("注册成功"))
            }

            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_DEAL_ERROR, "商户注册失败"))
        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

    @GetMapping("/query")
    fun queryShop(@RequestBody param: QueryShopParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        return try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            shopService.findByShopBySearch(param.shopid, param.shopaccno, param.shopUniqueId)?.let {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("shop", it)
                        .success())
            }

            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.SHOP_NOT_EXISTS, "商户不存在"))
        } catch (ex: RequestParamCheckException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }
}