package com.supwisdom.dlpay.system.controller;

import com.supwisdom.dlpay.api.bean.JsonResult;
import com.supwisdom.dlpay.api.domain.TPaytype;
import com.supwisdom.dlpay.api.domain.TShopPaytype;
import com.supwisdom.dlpay.api.domain.TShopPaytypeConfig;
import com.supwisdom.dlpay.framework.domain.TShop;
import com.supwisdom.dlpay.framework.domain.TShopacc;
import com.supwisdom.dlpay.framework.service.SystemUtilService;
import com.supwisdom.dlpay.framework.util.PageResult;
import com.supwisdom.dlpay.framework.util.StringUtil;
import com.supwisdom.dlpay.framework.util.TradeDict;
import com.supwisdom.dlpay.framework.util.WebConstant;
import com.supwisdom.dlpay.system.bean.ShopConfigBean;
import com.supwisdom.dlpay.system.service.ParamService;
import com.supwisdom.dlpay.system.service.ShopDataService;
import com.supwisdom.dlpay.util.ConstantUtil;
import com.supwisdom.dlpay.util.WebCheckException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@Controller
public class ShopController {
  @Autowired
  private ShopDataService shopDataService;
  @Autowired
  private ParamService paramService;
  @Autowired
  private SystemUtilService systemUtilService;

  @GetMapping("/shop/index")
  public String shopView() {
    return "system/shop/index";
  }

  @GetMapping("/shop/shoptree")
  @PreAuthorize("hasPermission('/shop/index','')")
  @ResponseBody
  public JsonResult shopTreeData() {
    return JsonResult.ok("OK").put("data", shopDataService.getAllShopNodes());
  }

  @PostMapping("/shop/deleteshop")
  @PreAuthorize("hasPermission('/shop/deleteshop','')")
  @ResponseBody
  public JsonResult deleteShop(@RequestParam("shopid") Integer shopid) {
    TShop shop = shopDataService.getShopByShopid(shopid);
    if (null == shop) {
      return JsonResult.error("商户不存在，请重新查询");  //商户不存在，请重新查询
    }

    try {
      if (shopDataService.deleteShop(shop)) {
        return JsonResult.ok("删除成功");
      } else {
        return JsonResult.error("删除失败");
      }
    } catch (WebCheckException ex) {
      return JsonResult.error(ex.getMessage());
    } catch (Exception e) {
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @GetMapping("/shop/getshopinfo")
  @PreAuthorize("hasPermission('/shop/getshopinfo','')")
  @ResponseBody
  public JsonResult getShopInfo(@RequestParam("shopid") Integer shopid) {
    TShop shop = shopDataService.getShopByShopid(shopid);
    if (null == shop) {
      return JsonResult.error("商户不存在，请重新查询");  //商户不存在，请重新查询
    }
    return JsonResult.ok("查询成功").put("shop", shop);
  }

  @PostMapping("/shop/saveorupdate")
  @PreAuthorize("hasPermission('/shop/saveorupdate','')")
  @ResponseBody
  public JsonResult saveOrUpdateShop(@RequestParam("shopid") Integer shopid, @RequestParam("shopname") String shopname,
                                     @RequestParam("fshopid") Integer fshopid, @RequestParam("shoptype") Integer shoptype,
                                     @RequestParam(value = "contactman", required = false) String contactman,
                                     @RequestParam(value = "idtype", required = false) String idtype,
                                     @RequestParam(value = "idno", required = false) String idno,
                                     @RequestParam(value = "mobile", required = false) String mobile,
                                     @RequestParam(value = "tel", required = false) String tel,
                                     @RequestParam(value = "email", required = false) String email,
                                     @RequestParam(value = "addr", required = false) String addr,
                                     @RequestParam(value = "zipcode", required = false) String zipcode) {
    if (null == shopid || StringUtil.isEmpty(shopname) || null == fshopid || (shoptype != 0 && shoptype != 1)) {
      return JsonResult.error("参数传递错误");
    }

    TShop shop;
    boolean addflag = false;
    if (shopid == 0) {
      //新增
      addflag = true;
      shop = new TShop();
      shop.setFshopid(fshopid);
      shop.setShopname(shopname);
      shop.setShoptype(shoptype);
      shop.setStatus(TradeDict.STATUS_NORMAL);
    } else {
      //修改
      shop = shopDataService.getShopByShopid(shopid);
      if (null == shop) {
        return JsonResult.error("商户不存在,修改失败！请重新查询");
      }
      shop.setFshopid(fshopid);
      shop.setShopname(shopname);
//      shop.setShoptype(shoptype); //商户类型不能改
    }
    shop.setContactman(contactman == null ? null : contactman.trim());
    shop.setIdno(idno == null ? null : idno.trim());
    shop.setIdtype(idtype == null ? null : idtype.trim());
    if (!StringUtil.isEmpty(shop.getIdno()) && StringUtil.isEmpty(shop.getIdtype())) {
      return JsonResult.error("填写证件号时必须制定证件类型！");
    } else if (!StringUtil.isEmpty(shop.getIdno()) && "1".equals(shop.getIdtype()) && !StringUtil.isIdentity(shop.getIdno())) {
      return JsonResult.error("身份证格式错误！");
    }
    shop.setMobile(mobile == null ? null : mobile);
    if (!StringUtil.isEmpty(shop.getMobile()) && !StringUtil.isMobile(shop.getMobile())) {
      return JsonResult.error("请正确填写手机号！");
    }
    shop.setTel(tel == null ? null : tel.trim());
    shop.setEmail(email == null ? null : email.trim());
    if (!StringUtil.isEmpty(shop.getEmail()) && !StringUtil.isEmail(shop.getEmail())) {
      return JsonResult.error("请正确填写邮箱地址！");
    }
    shop.setAddr(addr == null ? null : addr.trim());
    shop.setZipcode(zipcode == null ? null : zipcode.trim());

    try {
      if (shopDataService.saveOrUpdateShop(shop)) {
        return JsonResult.ok(addflag ? "新增成功" : "修改成功").put("shop", shop);
      } else {
        return JsonResult.error(addflag ? "新增失败" : "修改失败");
      }
    } catch (WebCheckException ex) {
      return JsonResult.error(ex.getMessage());
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  /**
   * ====================================================
   * 商户支付能力配置
   * ====================================================
   */
  @GetMapping("/shop/config")
  public String shopConfigView(Model model) {
    model.addAttribute("paytypelist", shopDataService.getConsumePaytypes());
    return "system/shop/config";
  }

  @GetMapping("/shop/shopacctree")
  @PreAuthorize("hasPermission('/shop/config','')")
  @ResponseBody
  public JsonResult shopaccTreeData() {
    return JsonResult.ok("OK").put("data", shopDataService.getAllShopNodes());
  }

  @GetMapping("/shop/shoppaytypelist")
  @PreAuthorize("hasPermission('/shop/shoppaytypelist','')")
  @ResponseBody
  public PageResult<ShopConfigBean> getShopPaytypeList(@RequestParam("page") Integer pageNo,
                                                       @RequestParam("limit") Integer pageSize,
                                                       @RequestParam(value = "paytype", required = false) String paytype,
                                                       @RequestParam(value = "shopaccno", required = false) String shopaccno) {
    try {
      if (null == pageNo || pageNo < 1) pageNo = WebConstant.PAGENO_DEFAULT;
      if (null == pageSize || pageSize < 1) pageSize = WebConstant.PAGESIZE_DEFAULT;
      return shopDataService.getShopPaytypeInfos(shopaccno, paytype, pageNo, pageSize);
    } catch (Exception e) {
      e.printStackTrace();
      return new PageResult<>(99, "系统查询错误");
    }
  }

  @PostMapping("/shop/updatepaytypestat")
  @PreAuthorize("hasPermission('/shop/updatepaytypestat','')")
  @ResponseBody
  public JsonResult updateShopPaytypeStat(@RequestParam("shopaccno") String shopaccno, @RequestParam("paytype") String paytype,
                                          @RequestParam("state") String state, @RequestParam("optype") String optype) {
    if (StringUtil.isEmpty(shopaccno) || StringUtil.isEmpty(paytype)
        || (!ConstantUtil.ENABLE_NO.equals(state) && !ConstantUtil.ENABLE_YES.equals(state))
        || (!"consume".equals(optype) && !"anonymous".equals(optype) && !"reverse".equals(optype))) {
      return JsonResult.error("参数传递错误");
    }

    try {
      TShopPaytype shopPaytype = shopDataService.getShopPaytype(shopaccno.trim(), paytype.trim());
      if (null == shopPaytype) {
        return JsonResult.error("商户支付能力不存在！");
      }

      if ("consume".equals(optype)) {
        if (state.equals(shopPaytype.getConsumeEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        shopPaytype.setConsumeEnable(state);
      } else if ("anonymous".equals(optype)) {
        if (state.equals(shopPaytype.getAnonymousEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        shopPaytype.setAnonymousEnable(state);
      } else if ("reverse".equals(optype)) {
        if (state.equals(shopPaytype.getReverseEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        shopPaytype.setReverseEnable(state);
      }

      if (shopDataService.saveOrUpdateShopPaytype(shopPaytype)) {
        return JsonResult.ok(ConstantUtil.ENABLE_YES.equals(state) ? "启用成功" : "关闭成功");
      } else {
        return JsonResult.error(ConstantUtil.ENABLE_YES.equals(state) ? "启用失败" : "关闭失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @GetMapping("/shop/load4addpaytype")
  @PreAuthorize("hasPermission('/shop/load4addpaytype','')")
  public String load4AddShopPaytype(Model model){
    model.addAttribute("paytypelist", shopDataService.getConsumePaytypes());
    return "system/shop/configform";
  }

  @PostMapping("/shop/addshoppaytype")
  @PreAuthorize("hasPermission('/shop/addshoppaytype','')")
  @ResponseBody
  public JsonResult addShoppaytype(@RequestParam("shopaccno") String shopaccno,
                                   @RequestParam("paytype") String paytype,
                                   @RequestParam(value = "consumeEnable", required = false, defaultValue = "no") String consumeEnable,
                                   @RequestParam(value = "anonymousEnable", required = false, defaultValue = "no") String anonymousEnable,
                                   @RequestParam(value = "reverseEnable", required = false, defaultValue = "no") String reverseEnable) {
    if(StringUtil.isEmpty(shopaccno) || StringUtil.isEmpty(paytype)){
      return JsonResult.error("参数传递错误");
    }
    try {
      TShopacc shopacc = shopDataService.getShopaccByAccno(shopaccno.trim());
      TPaytype tPaytype = paramService.getPaytype(paytype.trim());
      if (null == shopacc) {
        return JsonResult.error("商户账户不存在！");
      }
      if (null == tPaytype) {
        return JsonResult.error("支付方式在系统中不存在！");
      }
      TShopPaytype shopPaytype = shopDataService.getShopPaytype(shopaccno.trim(), paytype.trim());
      if (null != shopPaytype) {
        return JsonResult.error("商户该支付能力已经存在！");
      }
      shopPaytype = new TShopPaytype();
      shopPaytype.setShopaccno(shopacc.getShopaccno());
      shopPaytype.setPaytype(tPaytype.getPaytype());
      shopPaytype.setConsumeEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(consumeEnable) ? ConstantUtil.ENABLE_YES : ConstantUtil.ENABLE_NO);
      shopPaytype.setAnonymousEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(anonymousEnable) ? ConstantUtil.ENABLE_YES : ConstantUtil.ENABLE_NO);
      shopPaytype.setReverseEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(reverseEnable) ? ConstantUtil.ENABLE_YES : ConstantUtil.ENABLE_NO);
      shopPaytype.setCreatetime(systemUtilService.getSysdatetime().getHostdatetime());
      if (shopDataService.saveOrUpdateShopPaytype(shopPaytype)) {
        return JsonResult.ok("新增成功");
      } else {
        return JsonResult.error("新增失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @GetMapping("/shop/load4paytypepara")
  @PreAuthorize("hasPermission('/shop/load4paytypepara','')")
  public String load4AddShopPaytypePara(@RequestParam("shopaccno") String shopaccno,
                                        @RequestParam("paytype") String paytype,
                                        Model model) {
    List<TShopPaytypeConfig> configList = shopDataService.getShopPaytypeConfigs(shopaccno, paytype);
    model.addAttribute("configlist", configList);
    model.addAttribute("shopaccno", shopaccno);
    model.addAttribute("paytype", paytype);
    return "system/shop/configpara";
  }

  @PostMapping("/shop/addpaytypepara")
  @PreAuthorize("hasPermission('/shop/addpaytypepara','')")
  @ResponseBody
  public JsonResult addShopPaytypePara(@RequestBody Map<String, String> param) {
    String shopaccnoHtmlKey = "hid_shopaccno"; //页面上传来shopaccno的KEY
    String paytypeHtmlKey = "hid_paytype"; //页面上传来paytype的KEY
    if (null == param || StringUtil.isEmpty(param.get(paytypeHtmlKey)) || StringUtil.isEmpty(param.get(shopaccnoHtmlKey))) {
      return JsonResult.error("参数传递错误");
    }
    try {
      String shopaccno = param.get(shopaccnoHtmlKey).trim();
      String paytype = param.get(paytypeHtmlKey).trim();
      param.remove(shopaccnoHtmlKey);
      param.remove(paytypeHtmlKey);
      if (shopDataService.saveOrUpdateShopPaytypeConfig(shopaccno, paytype, param)) {
        return JsonResult.ok("配置成功");
      } else {
        return JsonResult.error("配置失败");
      }
    } catch (WebCheckException ex) {
      return JsonResult.error(ex.getMessage());
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }



}
