package com.supwisdom.dlpay.system.service.impl;

import com.supwisdom.dlpay.api.bean.JsonResult;
import com.supwisdom.dlpay.framework.dao.*;
import com.supwisdom.dlpay.framework.data.NodeData;
import com.supwisdom.dlpay.framework.domain.TPermission;
import com.supwisdom.dlpay.framework.domain.TResource;
import com.supwisdom.dlpay.framework.domain.TRole;
import com.supwisdom.dlpay.framework.domain.TRoleFunction;
import com.supwisdom.dlpay.framework.util.DateUtil;
import com.supwisdom.dlpay.framework.util.PageResult;
import com.supwisdom.dlpay.framework.util.StringUtil;
import com.supwisdom.dlpay.system.bean.FunctionSearchBean;
import com.supwisdom.dlpay.system.bean.ZTreeNode;
import com.supwisdom.dlpay.system.service.RoleService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Service
public class RoleServiceImpl implements RoleService {
    @Autowired
    private RoleDao roleDao;

    @Autowired
    private ResourceDao resourceDao;

    @Autowired
    private RoleFunctionDao roleFunctionDao;

    @Autowired
    private PermissionDao permissionDao;
    @Autowired
    private OperRoleDao operRoleDao;


    @Override
    public List<TRole> findAllRoles() {
        List<TRole> list = roleDao.getAllRoles();
        if (!StringUtil.isEmpty(list)) return list;
        return new ArrayList<>(0);
    }

    @Override
    public List<TRole> findAllRolesByOperid(String operid) {
        List<TRole> list = roleDao.getAllByOperid(operid);
        if (!StringUtil.isEmpty(list)) return list;
        return new ArrayList<>(0);
    }

    @Override
    public TRole findRoleByRoleid(String roleid) {
        Optional<TRole> role = roleDao.findById(roleid);
        if(role.isPresent()){
            return role.get();
        }
        return null;
    }

    @Override
    public TResource findResourceByURI(String uri) {
        return resourceDao.findByUri(uri);
    }

    @Override
    public TRoleFunction findRoleFunctionByRoleIdAndFunctionId(String roleid, Integer functionid) {
        return roleFunctionDao.findByRoleIdAndFunctionId(roleid, functionid);
    }

    @Override
    public PageResult<TRole> getRolesByKey(FunctionSearchBean param,String operType,String operid) {
        Pageable pageable = PageRequest.of(param.getPageNo() - 1, param.getPageSize()
                , Sort.by("createtime"));
        if (operType.equals("P") || operType.equals("S")) {
            if (!StringUtil.isEmpty(param.getFunctioname())) {
                return new PageResult<>(roleDao.findAllByRoleNameContaining(param.getFunctioname(), pageable));
            }
            return new PageResult<>(roleDao.findAll(pageable));
        }else {
            if (!StringUtil.isEmpty(param.getFunctioname())) {
                return new PageResult<>(roleDao.findAllByOperidAndRoleNameContaining(operid,param.getFunctioname(), pageable));
            }
            return new PageResult<>(roleDao.findAllByOperid(operid,pageable));
        }
    }

    @Override
    public JsonResult saveRole(TRole role,String operid) {
        if (!StringUtil.isEmpty(role.getRoleId())) {
            TRole temp = roleDao.findByRoleNameAndRoleIdNot(role.getRoleName(), role.getRoleId());
            if (temp != null) {
                return JsonResult.error("角色名称已存在");
            }
            Optional<TRole> opt = roleDao.findById(role.getRoleId());
            if (opt == null || !opt.isPresent()) {
                return JsonResult.error("角色已被删除");
            }
            temp = opt.get();
            temp.setRoleName(role.getRoleName());
            temp.setRoleDesc(role.getRoleDesc());
            temp.setRoleCode("ROLE_ADMIN");
            temp.setLastsaved(DateUtil.getNow());
            roleDao.save(temp);
        } else {
            TRole temp = roleDao.findByRoleName(role.getRoleName());
            if (temp != null) {
                return JsonResult.error("角色名称已存在");
            }

            role.setCreatetime(DateUtil.getNow());
            role.setEditflag(1);
            role.setRoleCode("ROLE_ADMIN");
            role.setOperid(operid);
            roleDao.save(role);
        }
        return JsonResult.ok("成功");
    }

    @Override
    public List<TRoleFunction> getRoleFuncByRoleid(String roleId) {
        return roleFunctionDao.findByRoleId(roleId);
    }


    @Override
    public TPermission findByRoleIdAndResid(String roleFuncId, Integer resid) {
        return permissionDao.findByRoleIdAndResid(roleFuncId, resid);
    }


    private TPermission findOrNewPermission(List<TPermission> exists, String roleId, Integer resId) {
        for (TPermission item : exists) {
            if (resId.equals(item.getResid())) {
                return item;
            }
        }

        TPermission perm = new TPermission();
        perm.setRoleId(roleId);
        perm.setResid(resId);
        return perm;
    }

    private TRoleFunction findOrNewRoleFunction(List<TRoleFunction> exists, String roleId, Integer funcId) {
        for (TRoleFunction item : exists) {
            if (funcId.equals(item.getFunctionId())) {
                return item;
            }
        }
        TRoleFunction func = new TRoleFunction();
        func.setFunctionId(funcId);
        func.setRoleId(roleId);
        return func;
    }

    @Override
    public JsonResult saveRoleFuncId(String roleId, String funcs) {
        Optional<TRole> ret = roleDao.findById(roleId);
        if (!ret.isPresent()) {
            return JsonResult.error("角色不存在");
        }
        List<TRoleFunction> functions = roleFunctionDao.findByRoleId(roleId);
        List<TPermission> permissions = permissionDao.findByRoleId(roleId);
        ArrayList<TRoleFunction> retainFuncs = new ArrayList<>();
        ArrayList<TPermission> retainPerms = new ArrayList<>();
        String[] datas = funcs.split(",");
        for (String func : datas) {
            if (func.contains("_res")) {
                String id = func.replace("_res", "");
                retainPerms.add(findOrNewPermission(permissions, roleId, Integer.parseInt(id)));
            } else {
                if ("-1".equals(func)) continue; //全选排除
                retainFuncs.add(findOrNewRoleFunction(functions, roleId, Integer.parseInt(func)));
            }
        }
        // functions
        functions.removeAll(retainFuncs);
        if (functions.size() > 0)
            roleFunctionDao.deleteAll(functions);
        roleFunctionDao.saveAll(retainFuncs);
        // permissions
        permissions.removeAll(retainPerms);
        if (permissions.size() > 0)
            permissionDao.deleteAll(permissions);
        permissionDao.saveAll(retainPerms);
        return JsonResult.ok("成功");
    }

    @Override
    public List<ZTreeNode> findByRoleIdNative(String roleId,String operType) {
        List<NodeData> nodeData = null;
        if (operType.equals("P") || operType.equals("S")) {
            nodeData  = roleFunctionDao.findByRoleIdNative(roleId);
        }else if (operType.equals("H")){
            nodeData = roleFunctionDao.findByRoleIdAndGradeNative(roleId, 2);//区域管理员取等级大于2的
        }else if (operType.equals("L")){
            nodeData = roleFunctionDao.findByRoleIdAndGradeNative(roleId, 3);//楼栋管理员取等级大于3的---null
        }
        List<ZTreeNode> ret = new ArrayList<>();
        boolean allChecked = true;
        for (NodeData data : nodeData) {
            ZTreeNode zTreeNode = new ZTreeNode();
            zTreeNode.setpId(data.getPid());
            zTreeNode.setId(data.getId());
            zTreeNode.setName(data.getName());
            zTreeNode.setChecked(data.getChecked() != 0);
            if (allChecked && !zTreeNode.isChecked()) allChecked = false;
            zTreeNode.setOpen(data.getOpen() != 0);
            ret.add(zTreeNode);
        }
        ZTreeNode root = new ZTreeNode();
        root.setId("-1");
        root.setpId("-99");
        root.setName("选择全部功能");
        root.setChecked(allChecked);
        root.setOpen(true);
        ret.add(root);
        return ret;
    }

    @Override
    public JsonResult deleteRole(String roleid) {
        Optional<TRole> ret = roleDao.findById(roleid);
        if (ret == null || !ret.isPresent()) {
            return JsonResult.error("角色不存在");
        }
        if(ret.get().getEditflag()!=1){
            return JsonResult.error("该角色不能删除");
        }
        permissionDao.deleteByRoleId(roleid);
        roleFunctionDao.deleteByRoleId(roleid);
        operRoleDao.deleteByRoleId(roleid);
        roleDao.deleteById(roleid);
        return JsonResult.ok("成功");
    }
}
