package com.supwisdom.dlpay.api.repositories

import com.supwisdom.dlpay.api.domain.TAccount
import com.supwisdom.dlpay.api.domain.TPersondtl
import com.supwisdom.dlpay.api.domain.TShopdtl
import com.supwisdom.dlpay.exception.TransactionProcessException
import com.supwisdom.dlpay.framework.domain.TShopacc
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional
import javax.persistence.EntityManager
import javax.persistence.PersistenceContext

@Repository
@Transactional
class TAccountRepositoryImpl : TAccountRepository {

    @PersistenceContext
    private lateinit var entityManager: EntityManager

    override fun recalcAccountBalance(accno: String, avaibal: Double, amount: Double, overdraft: Boolean): Int {
        val account = entityManager.find(TAccount::class.java, accno) ?: return 0
        if (avaibal.compareTo(account.availbal!!) != 0) {
            throw TransactionProcessException(TradeErrorCode.ACCOUNT_TRADE_BUSY,
                    "账户余额已被更新")
        }

        //1. 从 redis 读取账户余额
        //2. 没有记录时，读取数据库余额并存入 redis ，如果返回已存在，读取redis 余额
        //3. 扣除 redis 中余额
        val query = entityManager.createQuery("update TAccount a set a.availbal=a.availbal+?1, a.balance=a.balance+?1 " +
                "where a.accno=?2 and a.availbal=?3")
                ?: throw TransactionProcessException(TradeErrorCode.BUSINESS_DEAL_ERROR, "更新个人账户余额错误")

        query.setParameter(1, amount)
                .setParameter(2, accno)
                .setParameter(3, avaibal)
        return query.executeUpdate()
    }

    override fun recalcAccountBalance(account: TAccount, amount: Double, overdraft: Boolean): Int {
        return recalcAccountBalance(account.accno, account.availbal, amount, overdraft)
    }

    override fun recalcAccountBalance(dtl: TPersondtl, amount: Double, overdraft: Boolean): Int {
        return entityManager.find(TAccount::class.java, dtl.accountNo)?.let {
            recalcAccountBalance(it, amount, overdraft)
        } ?: throw TransactionProcessException(TradeErrorCode.BUSINESS_DEAL_ERROR,
                "交易流水<${dtl.refno}>未找到指定个人账户")
    }
}

@Repository
@Transactional
class ShopaccRepositoryImpl : TShopaccRepository {

    @PersistenceContext
    private lateinit var entityManager: EntityManager

    override fun recalcShopBalance(dtl: TShopdtl, amount: Double, overdraft: Boolean): Int {
        val shopacc = entityManager.find(TShopacc::class.java, dtl.shopaccno)
                ?: throw TransactionProcessException(TradeErrorCode.ACCOUNT_NOT_EXISTS,
                        "交易流水<${dtl.refno}>商户账户不存在")

        val query = entityManager.createQuery("update TShopacc c set c.balance=c.balance-?1" +
                " where c.shopaccno=?2 and c.balance=?3")
        query.setParameter(1, amount)
                .setParameter(2, dtl.shopaccno)
                .setParameter(3, shopacc.balance)
        return query.executeUpdate()
    }
}