package com.supwisdom.dlpay.consume.controller

import com.supwisdom.dlpay.consume.AccountHolder
import com.supwisdom.dlpay.consume.CallService
import com.supwisdom.dlpay.consume.PersonTransBuilder
import com.supwisdom.dlpay.consume.domain.TPaytype
import com.supwisdom.dlpay.consume.service.AccountUtilServcie
import com.supwisdom.dlpay.consume.service.PaytypeService
import com.supwisdom.dlpay.consume.service.PersonBalancePayService
import com.supwisdom.dlpay.consume.service.PersonService
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.util.ConstUtil
import com.supwisdom.dlpay.util.PaytypeUtil
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController

@RestController
@RequestMapping("/api")
class ConsumeController {
    @Autowired
    lateinit var accountUtilServcie: AccountUtilServcie

    @Autowired
    lateinit var personBalancePayService: PersonBalancePayService

    @Autowired
    lateinit var paytypeService: PaytypeService
    @Autowired
    lateinit var personService: PersonService

    @PostMapping("/account/pay")
    fun accountPay(userid: String, amount: Int, manageFee: Int): ResponseEntity<Any> {
        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                .setTransDatetime("20190411", "112311")
                .enableOverdraft(false)
                .addDetail(AccountHolder.person(userid),
                        AccountHolder.shop("12323"),
                        amount / 100.0, "")
                .addDetail(AccountHolder.person(userid), AccountHolder.transType(301),
                        manageFee / 100.0, "")
                .done(personBalancePayService, true)
        return ResponseEntity.ok(dtl)
    }

    @GetMapping("/account/payinit")
    fun accountPayInit(userid: String, amount: Int, manageFee: Int): ResponseEntity<Any> {
        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                .setTransDatetime("20190411", "112311")
                .enableOverdraft(false)
                .addDetail(AccountHolder.person(userid),
                        AccountHolder.shop("12323"),
                        amount / 100.0, "")
                .addDetail(AccountHolder.person(userid), AccountHolder.transType(301),
                        manageFee / 100.0, "")
                .done(personBalancePayService, false)

        return ResponseEntity.ok(dtl)
    }

    @GetMapping("/account/payfinish")
    fun accountPayFinish(refno: String): ResponseEntity<Any> {
        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                .done(refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
        return ResponseEntity.ok(dtl)
    }

    @PostMapping("/ykt/payinit")
    fun yktPayInit(userid: String, amount: Int, manageFee: Int,
                   stuempno: String, shopid: String, transdate: String, transtime: String,
                   outtradeno: String, payinfo: String, feetype: String): ResponseEntity<Any> {
        //一卡通支付款 112240
        return try {
            val paytype = paytypeService.getByPaytype(PaytypeUtil.YKTPAY)
            if (paytype == null || ConstUtil.ENABLE_YES != paytype.enable) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(1, "支付方式未开启"))
            }
            val person = personService.getPersonByThirdUniqueIdenty(stuempno)
            val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .setTransDatetime(transdate, transtime)
                    .selectPaytype(PaytypeUtil.YKTPAY, payinfo)
                    .setOuttradeno(outtradeno)
                    .setOwner(person)
                    .tryLock(true)
                    .setTransinfo(TradeCode.TRANSCODE_YKTPAY, "一卡通支付")
                    .chooseTradetype(Tradetype.CONSUME)
                    .also {
                        when (feetype) {
                            TradeDict.FEETYPE_CONSUME_MEALER -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                        AccountHolder.shop(shopid),
                                        amount / 100.0, "一卡通支付")
                                        .addDetail(AccountHolder.transType(TranstypeCode.TT_CONSUUME_MANAGE_FEE)
                                                .with(AccountHolder.shop(shopid)),
                                                manageFee / 100.0)
                            }
                            TradeDict.FEETYPE_CONSUME_DISCOUNT -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                        AccountHolder.shop(shopid),
                                        (amount - manageFee) / 100.0, "一卡通支付")
                                        .addDetail(AccountHolder.subject(Subject.SUBJNO_CONSUME_DISCOUNT),
                                                AccountHolder.shop(shopid),
                                                manageFee / 100.0, "优惠折扣")
                            }
                            else -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                        AccountHolder.shop(shopid),
                                        amount / 100.0, "一卡通支付")
                            }
                        }
                    }.done(personBalancePayService, false)
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data("refno", dtl.refno)
                    .success())
        } catch (e: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(e, "交易初始化异常"))
        }
    }

    @PostMapping("/ykt/payfinish")
    fun yktPayFinish(refno: String, yktshopid: String, devphyid: String?): ResponseEntity<Any> {
        return try {
            val dtl = personBalancePayService.wip(refno)
            val person = personService.getPersonByUserid(dtl.userid)
            val code = CallService.callYktPay(paytypeService.getPaytypeConfigByPaytype(PaytypeUtil.YKTPAY),
                    dtl, DateUtil.getNow(), person.thirdUniqueIdenty, yktshopid, devphyid)
            if (code.retcode == "0") {
                PersonTransBuilder.newBuilder(accountUtilServcie)
                        .done(dtl.refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .success())
            } else {
                PersonTransBuilder.newBuilder(accountUtilServcie)
                        .done(dtl.refno, TradeDict.DTL_STATUS_FAIL, personBalancePayService)
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易请求失败-${code.retcode}"))
            }
        } catch (e: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(e, "交易确认失败"))
        }
    }

    /**
     * 微信支付
     * wechattype
     * qrcode-扫微信二维码支付
     * app-原生app微信支付
     * mp-微信公众号支付
     * h5-微信h5支付
     *
     * */
    @PostMapping("/wechat/payinit")
    fun wechatPayInit(userid: String, amount: Int, manageFee: Int,
                      stuempno: String, shopid: String, transdate: String, transtime: String,
                      outtradeno: String, payinfo: String, feetype: String,
                      wechattype: String, realip: String?, qrcode: String?, openid: String?): ResponseEntity<Any> {
        return try {
            val paytype = paytypeService.getByPaytype(PaytypeUtil.WECHAT)
            if (paytype == null || ConstUtil.ENABLE_YES != paytype.enable) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(1, "支付方式未开启"))
            }
            val person = personService.getPersonByThirdUniqueIdenty(stuempno)
            val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .setTransDatetime(transdate, transtime)
                    .selectPaytype(PaytypeUtil.WECHAT, payinfo)
                    .setOuttradeno(outtradeno)
                    .setOwner(person)
                    .tryLock(true)
                    .setTransinfo(TradeCode.TRANSCODE_WECHAT, "微信支付")
                    .chooseTradetype(Tradetype.CONSUME)
                    .also {
                        when (feetype) {
                            TradeDict.FEETYPE_CONSUME_MEALER -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
                                        AccountHolder.shop(shopid),
                                        amount / 100.0, "微信支付")
                                        .addDetail(AccountHolder.transType(TranstypeCode.TT_CONSUUME_MANAGE_FEE)
                                                .with(AccountHolder.shop(shopid)),
                                                manageFee / 100.0)
                            }
                            TradeDict.FEETYPE_CONSUME_DISCOUNT -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
                                        AccountHolder.shop(shopid),
                                        (amount - manageFee) / 100.0, "微信支付")
                                        .addDetail(AccountHolder.subject(Subject.SUBJNO_CONSUME_DISCOUNT),
                                                AccountHolder.shop(shopid),
                                                manageFee / 100.0, "优惠折扣")
                            }
                            else -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
                                        AccountHolder.shop(shopid),
                                        amount / 100.0, "微信支付")
                            }
                        }
                    }.done(personBalancePayService, false)
            val code = CallService.callWechatPay(paytypeService.getPaytypeConfigByPaytype(PaytypeUtil.WECHAT),
                    dtl, DateUtil.getNow(), wechattype, realip, qrcode, openid)
            if (code.retcode == "0") {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .success())
            } else {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易请求失败-${code.retcode}"))
            }
        } catch (e: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(e, "交易初始化异常"))
        }
    }
}