package com.supwisdom.dlpay.consume.service.impl

import com.supwisdom.dlpay.consume.PersonTransBuilder
import com.supwisdom.dlpay.consume.dao.AccountDao
import com.supwisdom.dlpay.consume.dao.DebitCreditDtlDao
import com.supwisdom.dlpay.consume.dao.UserdtlDao
import com.supwisdom.dlpay.consume.domain.TAccount
import com.supwisdom.dlpay.consume.domain.TDebitCreditDtl
import com.supwisdom.dlpay.consume.domain.TUserdtl
import com.supwisdom.dlpay.consume.service.AccountUtilServcie
import com.supwisdom.dlpay.consume.service.PersonBalancePayService
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.exception.TransactionProcessException
import com.supwisdom.dlpay.framework.dao.ShopaccDao
import com.supwisdom.dlpay.framework.dao.SubjectDao
import com.supwisdom.dlpay.framework.dao.TranstypeDao
import com.supwisdom.dlpay.framework.domain.TShopacc
import com.supwisdom.dlpay.framework.domain.TSubject
import com.supwisdom.dlpay.framework.domain.TTranstype
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import org.hibernate.exception.LockTimeoutException
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.dao.CannotAcquireLockException
import org.springframework.stereotype.Service
import javax.persistence.EntityManager
import javax.persistence.PersistenceContext


@Service
class AccountUtilServcieImpl : AccountUtilServcie {

    @Autowired
    lateinit var accountDao: AccountDao

    @Autowired
    lateinit var shopaccDao: ShopaccDao

    @Autowired
    lateinit var subjectDao: SubjectDao

    @Autowired
    lateinit var transtypeDao: TranstypeDao


    override fun readAccountForUpdateNowait(userid: String): TAccount {
        return try {
            accountDao.getByUseridForUpdateNowait(userid)
                    ?: throw TransactionProcessException(TradeErrorCode.ACCOUNT_NOT_EXISTS, "账户<$userid>不存在")
        } catch (ex: Exception) {
            when (ex) {
                is CannotAcquireLockException, is LockTimeoutException -> throw TransactionException(TradeErrorCode.ACCOUNT_TRADE_BUSY, "账户<$userid>交易繁忙，请稍后再试")
                else -> throw ex
            }
        }
    }

    override fun readAccount(userid: String): TAccount {
        return accountDao.findByUserid(userid)
                ?: throw TransactionProcessException(TradeErrorCode.ACCOUNT_NOT_EXISTS, "账户<$userid>不存在")
    }

    override fun readShopAcc(shopId: Int): TShopacc {
        return shopaccDao.findByShopid(shopId)
                ?: throw TransactionProcessException(TradeErrorCode.SHOP_NOT_EXISTS, "商户<$shopId>不存在")
    }

    override fun readSubject(subjno: String): TSubject {
        return subjectDao.findBySubjno(subjno)
                ?: throw TransactionProcessException(TradeErrorCode.SUBJECT_NOT_EXISTS, "科目<$subjno>不存在")
    }

    override fun readTranstype(transtype: Int): TTranstype {
        return transtypeDao.findByTranstype(transtype)
                ?: throw TransactionProcessException(TradeErrorCode.SUBJECT_NOT_EXISTS, "交易类型<$transtype>未配置")
    }
}

@Service
class PersonBalancePayServiceImpl : PersonBalancePayService {
    @Autowired
    lateinit var userdtlDao: UserdtlDao
    @Autowired
    lateinit var debitCreditDtlDao: DebitCreditDtlDao

    @Autowired
    lateinit var accountDao: AccountDao

    @PersistenceContext
    lateinit var em: EntityManager

    @Autowired
    lateinit var systemUtilService: SystemUtilService


    private fun getlockAccount(accno: String): TAccount {
        return accountDao.getByAccnoForUpdate(accno)
                ?: throw throw TransactionProcessException(TradeErrorCode.ACCOUNT_NOT_EXISTS, "账号<$accno>不存在")
    }

    private fun getlockAccountNowait(accno: String): TAccount {
        return try {
            accountDao.getByAccnoForUpdateNowait(accno)
                    ?: throw throw TransactionProcessException(TradeErrorCode.ACCOUNT_NOT_EXISTS, "账号<$accno>不存在")
        } catch (ex: Exception) {
            when (ex) {
                is CannotAcquireLockException, is LockTimeoutException -> throw TransactionException(TradeErrorCode.ACCOUNT_TRADE_BUSY, "账号<$accno>交易繁忙，请稍后再试")
                else -> throw ex
            }
        }
    }

    private fun getLockUserdtlNowait(refno: String): TUserdtl {
        return try {
            userdtlDao.findByRefnoForUpdateNowait(refno)
                    ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易参考号<$refno>不存在")
        } catch (ex: Exception) {
            when (ex) {
                is CannotAcquireLockException, is LockTimeoutException -> throw TransactionException(TradeErrorCode.LOCK_READ_TIMEOUT, "交易参考号<$refno>流水被锁定，请稍后再试")
                else -> throw ex
            }
        }
    }

    private fun getLockUserdtl(refno: String): TUserdtl {
        return userdtlDao.findByRefnoForUpdate(refno)
                ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易参考号<$refno>不存在")
    }

    private fun checkOuttradenoExist(outtradeno: String): Boolean {
//        TODO("判断 outtradeno 重复发起")
        return false
    }

    private fun doDealAccount(accno: String, amount: Double, overdraft: Boolean): TAccount {
        val account = getlockAccount(accno)
        if (account.tacCheck())
            throw TransactionProcessException(TradeErrorCode.ACCOUNT_TAC_ERROR, "账户<$accno>tac校验异常")

        account.addAmount(amount) //入账
        if (account.checkOverflow())
            throw TransactionProcessException(TradeErrorCode.OVERFLOW_BALANCE_ERROR, "账户<$accno>已超最大余额限制")

        if (!overdraft && account.checkOverdraft())
            throw TransactionProcessException(TradeErrorCode.SHORT_BALANCE_ERROR, "账户<$accno>余额不足")

        return accountDao.save(account) //入库更新
    }

    private fun doDealShopacc(shopaccno: String, amount: Double) {
        return
    }

    override fun process(builder: PersonTransBuilder): TUserdtl {
        return finish(init(builder), TradeDict.DTL_STATUS_SUCCESS, null)
    }

    override fun init(builder: PersonTransBuilder): TUserdtl {
        val userdtl = TUserdtl()
        userdtl.refno = systemUtilService.refno
        userdtl.accdate = systemUtilService.accdate
        userdtl.userid = builder.person.userid
        if (StringUtil.isEmpty(builder.transDate)) {
            userdtl.transdate = systemUtilService.sysdatetime.hostdate
        } else {
            userdtl.transdate = builder.transDate
        }
        if (StringUtil.isEmpty(builder.transTime)) {
            userdtl.transdate = systemUtilService.sysdatetime.hosttime
        } else {
            userdtl.transtime = builder.transTime
        }
        userdtl.paytype = builder.paytype
        userdtl.payinfo = builder.payinfo
        userdtl.transcode = builder.transcode
        if (StringUtil.isEmpty(builder.description)) {
            userdtl.transdesc = systemUtilService.getTranscodeName(builder.transcode, null);
        } else {
            userdtl.transdesc = builder.description
        }
        userdtl.outtradeno = builder.outtradeno
//        userdtl.operid =
        when (builder.tradetype) {
            Tradetype.RECHARGE -> userdtl.tradeflag = 1
            Tradetype.CONSUME -> userdtl.tradeflag = 2
        }
        userdtl.createtime = systemUtilService.sysdatetime.hostdatetime
        if (checkOuttradenoExist(userdtl.outtradeno)) {
            throw TransactionProcessException(TradeErrorCode.OUTTRADENO_ALREADY_EXISTS, "外部流水号重复")
        }

        userdtl.amount = builder.amount
        userdtl.status = TradeDict.DTL_STATUS_INIT
        userdtlDao.save(userdtl)

        builder.details.forEach {
            TDebitCreditDtl().apply {
                refno = userdtl.refno
                seqno = it.rowno
                drsubjno = it.debitSubjNo
                draccno = it.debitAccNo
                crsubjno = it.creditSubjNo
                craccno = it.creditAccNo
                amount = it.amount
                summary = it.summary
                debitCreditDtlDao.save(this)
            }
        }
        return userdtl
    }

    override fun finish(paydtl: TUserdtl, status: String, businessData: Map<String, String>?): TUserdtl {
        return finish(paydtl.refno, status, businessData)
    }

    private fun fail(userdtl: TUserdtl, businessData: Map<String, String>?) {
        //失败
        if (TradeDict.DTL_STATUS_SUCCESS == userdtl.status)
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水已经交易成功")
        userdtl.status = TradeDict.DTL_STATUS_FAIL
        userdtl.endtime = systemUtilService.sysdatetime.hostdatetime
        userdtl.remark = businessData?.get("errmsg")
        userdtlDao.save(userdtl)
    }

    private fun success(userdtl: TUserdtl, businessData: Map<String, String>?) {
        if (TradeDict.DTL_STATUS_SUCCESS == userdtl.status) {
            return
        }
        //TODO 校验已经成功的流水，不重复入账
        debitCreditDtlDao.findByRefno(userdtl.refno).forEach { detail ->
            //个人账户入账
            if (Subject.SUBJNO_PERSONAL_DEPOSIT == detail.drsubjno) {
                doDealAccount(detail.draccno, -1 * detail.amount, false) //借方消费
            }
            if (Subject.SUBJNO_PERSONAL_DEPOSIT == detail.crsubjno) {
                doDealAccount(detail.craccno, detail.amount, false) //贷方充值
            }

            //商户入账
            if (Subject.SUBJNO_MACHANT_INCOME == detail.drsubjno) {
                doDealShopacc(detail.draccno, -1 * detail.amount)
            }
            if (Subject.SUBJNO_MACHANT_INCOME == detail.crsubjno) {
                doDealShopacc(detail.craccno, detail.amount)
            }
        }

        userdtl.status = TradeDict.DTL_STATUS_SUCCESS
        userdtl.accdate = systemUtilService.accdate //入账成功时更新
        userdtl.endtime = systemUtilService.sysdatetime.hostdatetime
        //TODO 存储一些业务参数
        userdtlDao.save(userdtl)
    }

    override fun finish(refno: String, status: String, businessData: Map<String, String>?): TUserdtl {
        val userdtl = getLockUserdtl(refno)
        return when (status) {
            TradeDict.DTL_STATUS_FAIL -> {
                fail(userdtl, businessData)
                userdtl
            }
            TradeDict.DTL_STATUS_SUCCESS -> {
                //成功
                success(userdtl, businessData)
                userdtl
            }
            else -> throw TransactionProcessException(TradeErrorCode.TRANSDTL_STATUS_ERROR, "未指定明确的交易结束状态")
        }
    }

    override fun wip(paydtl: TUserdtl): TUserdtl {
        return wip(paydtl.refno)
    }

    override fun wip(refno: String): TUserdtl {
        val userdtl = getLockUserdtlNowait(refno)
        if (TradeDict.DTL_STATUS_WIP == userdtl.status) {
            return userdtl
        }

        if (TradeDict.DTL_STATUS_INIT != userdtl.status) {
            throw TransactionProcessException(TradeErrorCode.TRANSDTL_STATUS_NOT_INIT,
                    "交易参考号<$refno>非初始化流水")
        }
        userdtl.status = TradeDict.DTL_STATUS_WIP  //待支付
        return userdtlDao.save(userdtl)
    }
}