package com.supwisdom.dlpay

import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.core.JwtTokenUtil
import com.supwisdom.dlpay.framework.core.PasswordBCryptConfig
import com.supwisdom.dlpay.framework.redisrepo.ApiJwtRepository
import com.supwisdom.dlpay.framework.security.ValidateCodeSecurityConfig
import com.supwisdom.dlpay.framework.service.OperatorDetailService
import com.supwisdom.dlpay.framework.util.TradeDict
import org.jose4j.jwt.consumer.InvalidJwtException
import org.jose4j.lang.JoseException
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.core.annotation.Order
import org.springframework.http.HttpStatus
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken
import org.springframework.security.authentication.dao.DaoAuthenticationProvider
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder
import org.springframework.security.config.annotation.web.builders.HttpSecurity
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
import org.springframework.security.config.http.SessionCreationPolicy
import org.springframework.security.core.authority.SimpleGrantedAuthority
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder
import org.springframework.security.web.authentication.AuthenticationFailureHandler
import org.springframework.security.web.authentication.AuthenticationSuccessHandler
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter
import org.springframework.security.web.authentication.rememberme.JdbcTokenRepositoryImpl
import org.springframework.security.web.util.matcher.AntPathRequestMatcher
import org.springframework.stereotype.Component
import org.springframework.web.filter.OncePerRequestFilter
import java.security.SecureRandom
import javax.servlet.FilterChain
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse
import javax.sql.DataSource


@Component
class ApiJwtAuthenticationFilter : OncePerRequestFilter() {
    @Autowired
    lateinit var jwtConfig: JwtConfig

    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository

    private var jwtUtil: JwtTokenUtil? = null


    private fun getUtil(): JwtTokenUtil {
        if (jwtUtil == null) {
            jwtUtil = JwtTokenUtil((jwtConfig))
        }
        return jwtUtil as JwtTokenUtil
    }

    override fun doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain) {
        request.getHeader(jwtConfig.header)?.let { authHeader ->
            try {
                val jwt = if (authHeader.startsWith(jwtConfig.tokenHeader)) {
                    authHeader.substring(jwtConfig.tokenHeader.length)
                } else {
                    throw JoseException("JWT Header error")
                }
                val claims = getUtil().verifyToken(jwt)
                apiJwtRepository.findById(claims["jti"].toString()).let {
                    if (!it.isPresent) {
                        throw JoseException("JWT has not been register")
                    }
                    // token 已被设为黑名单
                    if (it.get().status != TradeDict.JWT_STATUS_NORMAL) {
                        throw JoseException("JWT status error : ${it.get().status}")
                    }
                }
                val auth = UsernamePasswordAuthenticationToken(claims["uid"], null,
                        (claims["authorities"] as ArrayList<*>)
                                .map { SimpleGrantedAuthority(it as String) })
                SecurityContextHolder.getContext().authentication = auth
            } catch (e: InvalidJwtException) {
                SecurityContextHolder.clearContext()
                if (e.hasExpired()) {
                    // jwt 过期后返回 401
                    apiJwtRepository.deleteById(e.jwtContext.jwtClaims.jwtId)
                    response.sendError(HttpStatus.UNAUTHORIZED.value(), e.message)
                } else {
                    response.sendError(HttpStatus.BAD_REQUEST.value(), e.message)
                }
            } catch (e: JoseException) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.sendError(HttpStatus.UNAUTHORIZED.value(), e.message)
            }
        }
        filterChain.doFilter(request, response)
    }
}

@EnableWebSecurity
class WebSecurityConfig {

    companion object {
        @Configuration
        @Order(1)
        class ApiWebSecurityConfigurationAdapter : WebSecurityConfigurerAdapter() {

            @Autowired
            lateinit var apiFilter: ApiJwtAuthenticationFilter

            override fun configure(http: HttpSecurity) {
                // 设置 API 访问权限管理
                http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                        .and()
                        .addFilterAfter(apiFilter,
                                UsernamePasswordAuthenticationFilter::class.java)
                        .antMatcher("/api/**")
                        .authorizeRequests()
                        .antMatchers("/api/auth/**").permitAll()
                        .antMatchers("/api/notify/**").permitAll()
                        .antMatchers("/api/common/**").hasAnyRole("THIRD_COMMON", "THIRD_ADMIN")
                        .antMatchers("/api/consume/**").hasRole("THIRD_CONSUME")
                        .antMatchers("/api/deposit/**").hasRole("THIRD_DEPOSIT")
                        .antMatchers("/api/user/**").hasRole("THIRD_ADMIN")
                        .antMatchers("/api/shop/**").hasRole("THIRD_SHOP")
                        .anyRequest().hasRole("THIRD_COMMON")
                        .and()
                        .csrf().ignoringAntMatchers("/api/**")
            }
        }

        @Configuration
        class MvcWebSecurityConfigurationAdapter : WebSecurityConfigurerAdapter() {
            @Autowired
            lateinit var dataSource: DataSource
            @Autowired
            lateinit var validateCodeSecurityConfig: ValidateCodeSecurityConfig
            @Autowired
            lateinit var authenticationFailureHandler: AuthenticationFailureHandler
            @Autowired
            lateinit var authenticationSuccessHandler: AuthenticationSuccessHandler
            @Autowired
            lateinit var passwordBCryptConfig: PasswordBCryptConfig


            @Autowired
            lateinit var userDetailsService: OperatorDetailService

            override fun configure(auth: AuthenticationManagerBuilder) {
                auth.authenticationProvider(authenticationProvider())
            }

            @Bean
            fun authenticationProvider(): DaoAuthenticationProvider {
                return DaoAuthenticationProvider().apply {
                    setUserDetailsService(userDetailsService)
                    setPasswordEncoder(passwordEncoder())
                }
            }

            @Bean
            fun passwordEncoder(): BCryptPasswordEncoder {
                return if (passwordBCryptConfig.seed.isBlank()) {
                    BCryptPasswordEncoder()
                } else {
                    BCryptPasswordEncoder(passwordBCryptConfig.length,
                            SecureRandom(passwordBCryptConfig.seed.toByteArray()))
                }
            }

            @Bean
            fun jdbcTokenImplement(): JdbcTokenRepositoryImpl {
                return JdbcTokenRepositoryImpl().also {
                    it.setDataSource(dataSource)
                }
            }

            override fun configure(http: HttpSecurity) {
                // 设置 Web MVC 应用权限
                http.apply(validateCodeSecurityConfig)
                        .and().csrf()
                        .and()
                        .authorizeRequests()
                        .antMatchers("/login", "/login/form").permitAll()
                        .antMatchers("/static/**").permitAll()
                        .antMatchers("/code/image").permitAll()
                        .antMatchers("/**").hasAnyRole("USER", "ADMIN")
                        .anyRequest().authenticated()
                        .and()
                        .sessionManagement()
                        .sessionCreationPolicy(SessionCreationPolicy.ALWAYS)
                        .and()
                        .formLogin()
                        .loginPage("/login")
                        .loginProcessingUrl("/login/form")
                        .successHandler(authenticationSuccessHandler)
                        .failureHandler(authenticationFailureHandler)
                        .and()
                        .logout()
                        .logoutRequestMatcher(AntPathRequestMatcher("/logout"))
                        .logoutSuccessUrl("/login")
                        .deleteCookies("JSESSIONID")
                        .invalidateHttpSession(true)
                // 设置 Web MVC 应用权限
//                http.apply(validateCodeSecurityConfig)
//                        .and()
//                        .formLogin()
//                        .loginPage("/login")
//                        .loginProcessingUrl("/login/form")
//                        .successHandler(zyAuthenticationSuccessHandler)
//                        .failureHandler(zyAuthenticationFailureHandler)
//                        .and()
//                        .logout()
//                        .logoutRequestMatcher(AntPathRequestMatcher("/logout"))
//                        .logoutSuccessUrl("/login")
//                        .deleteCookies("JSESSIONID")
//                        .invalidateHttpSession(true)
//                        .and()
//                        .userDetailsService(userDetailsService)
//                        .authorizeRequests()
//                        .antMatchers("/login").permitAll()
//                        .antMatchers("/static/**").permitAll()
//                        .antMatchers("/code/image").permitAll()
//                        .anyRequest().authenticated()
//                        .and()
//                        .sessionManagement()
//                        .invalidSessionStrategy(myInvalidSessionStrategy)
//                        .maximumSessions(1)
//                        .sessionRegistry(SessionRegistryImpl())
//                        .maxSessionsPreventsLogin(true)
//                        .and()
//                        .and()
//                        .headers().frameOptions().disable()
            }
        }
    }
}
