package com.supwisdom.dlpay.system.controller;

import com.supwisdom.dlpay.api.bean.JsonResult;
import com.supwisdom.dlpay.api.domain.TSourceType;
import com.supwisdom.dlpay.api.domain.TPaytypeConfig;
import com.supwisdom.dlpay.framework.domain.TApiClient;
import com.supwisdom.dlpay.framework.domain.TBusinesspara;
import com.supwisdom.dlpay.framework.domain.TSyspara;
import com.supwisdom.dlpay.framework.service.SystemUtilService;
import com.supwisdom.dlpay.framework.util.*;
import com.supwisdom.dlpay.system.service.ParamService;
import com.supwisdom.dlpay.util.ConstantUtil;
import com.supwisdom.dlpay.util.WebCheckException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Controller
public class ParamController {
  @Autowired
  private ParamService paramService;
  @Autowired
  private SystemUtilService systemUtilService;

  @GetMapping("/param/syspara")
  public String sysparaView() {
    return "system/param/syspara";
  }

  @GetMapping("/param/sysparalist")
  @PreAuthorize("hasPermission('/param/syspara','')")
  @ResponseBody
  public PageResult<TSyspara> getDataList(@RequestParam("page") Integer pageNo,
                                          @RequestParam("limit") Integer pageSize,
                                          @RequestParam(value = "paraid", required = false) Integer paraid,
                                          @RequestParam(value = "paraname", required = false) String paraname) {
    try {
      if (null == pageNo || pageNo < 1) pageNo = WebConstant.PAGENO_DEFAULT;
      if (null == pageSize || pageSize < 1) pageSize = WebConstant.PAGESIZE_DEFAULT;
      return paramService.getSysparaPage(paraid, paraname, pageNo, pageSize);
    } catch (Exception e) {
      e.printStackTrace();
      return new PageResult<>(99, "系统查询错误");
    }
  }

  @PostMapping("/param/sysparaupdate")
  @PreAuthorize("hasPermission('/param/sysparaupdate','')")
  @ResponseBody
  public JsonResult updateSyspara(@RequestParam("paraid") Integer paraid,
                                  @RequestParam("paraval") String paraval) {
    if (null == paraid || null == paraval) {
      return JsonResult.error("参数传递错误");
    } else if (paraval.length() > 30) {
      return JsonResult.error("参数值过长");
    }
    TSyspara syspara = paramService.getSysparaByParaid(paraid);
    if (null == syspara) {
      return JsonResult.error("全局参数不存在");
    }
    syspara.setParaval(paraval);
    syspara.setLastsaved(systemUtilService.getSysdatetime().getHostdatetime());
    if (syspara.checkValueInvalid()) {
      return JsonResult.error("参数值格式错误");
    }
    try {
      if (paramService.updateSyspara(syspara)) {
        return JsonResult.ok("修改成功");
      } else {
        return JsonResult.error("修改失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }


  /**
   * ====================================================
   * 业务参数配置
   * ====================================================
   */
  @GetMapping("/param/businesspara")
  public String busiparaView() {
    return "system/param/businesspara";
  }

  @GetMapping("/param/businessparalist")
  @PreAuthorize("hasPermission('/param/businesspara','')")
  @ResponseBody
  public PageResult<TBusinesspara> getBusinessDataList(@RequestParam("page") Integer pageNo,
                                                       @RequestParam("limit") Integer pageSize,
                                                       @RequestParam(value = "paraname", required = false) String paraname) {
    try {
      if (null == pageNo || pageNo < 1) pageNo = WebConstant.PAGENO_DEFAULT;
      if (null == pageSize || pageSize < 1) pageSize = WebConstant.PAGESIZE_DEFAULT;
      return paramService.getBusinessparaPage(paraname, pageNo, pageSize);
    } catch (Exception e) {
      e.printStackTrace();
      return new PageResult<>(99, "系统查询错误");
    }
  }

  @PostMapping("/param/deletebusinesspara")
  @PreAuthorize("hasPermission('/param/deletebusinesspara','')")
  @ResponseBody
  public JsonResult deleteBusinesspara(@RequestParam("parakey") String parakey) {
    try {
      if (StringUtil.isEmpty(parakey)) {
        return JsonResult.error("参数名为空");
      }
      TBusinesspara businesspara = paramService.getBusinessparaByParakey(parakey);
      if (null == businesspara) {
        return JsonResult.error("业务参数[" + parakey + "]不存在！");
      }
      if (paramService.deleteBusinesspara(businesspara)) {
        return JsonResult.ok("删除成功！");
      } else {
        return JsonResult.error("删除失败！");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/businessparaupdate")
  @PreAuthorize("hasPermission('/param/businessparaupdate','')")
  @ResponseBody
  public JsonResult updateBusinesspara(@RequestParam("parakey") String parakey, @RequestParam("paraval") String paraval) {
    try {
      if (StringUtil.isEmpty(parakey) || null == paraval) {
        return JsonResult.error("参数传递错误");
      }
      TBusinesspara businesspara = paramService.getBusinessparaByParakey(parakey);
      if (null == businesspara) {
        return JsonResult.error("业务参数[" + parakey + "]不存在！");
      }
      businesspara.setParaval(paraval);
      if (paramService.saveOrUpdateBusinesspara(businesspara)) {
        return JsonResult.ok("修改成功！");
      } else {
        return JsonResult.error("修改失败！");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @GetMapping("/param/load4addbuspara")
  @PreAuthorize("hasPermission('/param/load4addbuspara','')")
  public String loadadd() {
    return "system/param/busparaform";
  }

  @GetMapping("/param/checkbusname")
  @ResponseBody
  public JsonResult checkname(@RequestParam("parakey") String parakey) {
    TBusinesspara businesspara = paramService.getBusinessparaByParakey(parakey);
    if (businesspara == null) {
      return JsonResult.ok();
    } else {
      return JsonResult.error("参数名重复");
    }
  }

  @PostMapping("/param/addbusinesspara")
  @PreAuthorize("hasPermission('/param/addbusinesspara','')")
  @ResponseBody
  public JsonResult addBusinesspara(@RequestParam("parakey") String parakey, @RequestParam("paraval") String paraval) {
    if (StringUtil.isEmpty(parakey) || null == paraval) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TBusinesspara businesspara = paramService.getBusinessparaByParakey(parakey);
      if (null != businesspara) {
        return JsonResult.error("参数名重复");
      }
      businesspara = new TBusinesspara();
      businesspara.setParakey(parakey.trim());
      businesspara.setParaval(paraval.trim());
      if (paramService.saveOrUpdateBusinesspara(businesspara)) {
        return JsonResult.ok("新增成功！");
      } else {
        return JsonResult.error("新增失败！");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }


  /**
   * ====================================================
   * APPID应用权限参数管理
   * ====================================================
   */
  @GetMapping("/param/apiclientpara")
  public String appidParaView() {
    return "system/param/apiclientpara";
  }

  @GetMapping("/param/apiclientparalist")
  @PreAuthorize("hasPermission('/param/apiclientpara','')")
  @ResponseBody
  public PageResult<TApiClient> getApiclientDataList(@RequestParam("page") Integer pageNo,
                                                     @RequestParam("limit") Integer pageSize,
                                                     @RequestParam(value = "appid", required = false) String appid) {
    try {
      if (null == pageNo || pageNo < 1) pageNo = WebConstant.PAGENO_DEFAULT;
      if (null == pageSize || pageSize < 1) pageSize = WebConstant.PAGESIZE_DEFAULT;
      return paramService.getApiClientparaPage(appid, pageNo, pageSize);
    } catch (Exception e) {
      e.printStackTrace();
      return new PageResult<>(99, "系统查询错误");
    }
  }

  @GetMapping("/param/load4addapiclient")
  @PreAuthorize("hasPermission('/param/load4addapiclient','')")
  public String load4AddApiclientPara(Model model) {
    Map<String, String> roles = new HashMap<>(0);
    roles.put("ROLE_THIRD_COMMON", "ROLE_THIRD_COMMON");
    roles.put("ROLE_THIRD_CONSUME", "ROLE_THIRD_CONSUME");
    roles.put("ROLE_THIRD_DEPOSIT", "ROLE_THIRD_DEPOSIT");
    roles.put("ROLE_THIRD_SHOP", "ROLE_THIRD_SHOP");
    roles.put("ROLE_THIRD_ADMIN", "ROLE_THIRD_ADMIN");
    model.addAttribute("rolelist", roles);
    return "system/param/apiclientform";
  }

  @PostMapping("/param/updateapiclientstate")
  @PreAuthorize("hasPermission('/param/updateapiclientstate','')")
  @ResponseBody
  public JsonResult updateApiclientState(@RequestParam("appid") String appid, @RequestParam("state") String state) {
    if (StringUtil.isEmpty(appid) || (!TradeDict.STATUS_NORMAL.equals(state) && !TradeDict.STATUS_CLOSED.equals(state))) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TApiClient apiClient = paramService.getApiClientByAppid(appid);
      if (null == apiClient) {
        return JsonResult.error("应用参数不存在");
      } else if (state.equals(apiClient.getStatus())) {
        return JsonResult.error("状态修改错误，请刷新查询");
      }
      apiClient.setStatus(state);
      if (paramService.saveOrUpdateApiClient(apiClient)) {
        return JsonResult.ok(TradeDict.STATUS_NORMAL.equals(state) ? "启用成功" : "关闭成功");
      } else {
        return JsonResult.error(TradeDict.STATUS_NORMAL.equals(state) ? "启用失败" : "关闭失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/deleteapiclient")
  @PreAuthorize("hasPermission('/param/deleteapiclient','')")
  @ResponseBody
  public JsonResult deleteApiclientParam(@RequestParam("appid") String appid) {
    if (StringUtil.isEmpty(appid)) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TApiClient apiClient = paramService.getApiClientByAppid(appid);
      if (null == apiClient) {
        return JsonResult.error("应用参数不存在");
      }

      if (paramService.deleteApiClient(apiClient)) {
        return JsonResult.ok("删除成功");
      } else {
        return JsonResult.error("删除失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/updateapiclientpara")
  @PreAuthorize("hasPermission('/param/updateapiclientpara','')")
  @ResponseBody
  public JsonResult updateApiclientSecret(@RequestParam("appid") String appid,
                                          @RequestParam(value = "secret", required = false) String secret,
                                          @RequestParam(value = "roles", required = false) String roles) {
    if (StringUtil.isEmpty(appid) || (StringUtil.isEmpty(secret) && StringUtil.isEmpty(roles))) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TApiClient apiClient = paramService.getApiClientByAppid(appid);
      if (null == apiClient) {
        return JsonResult.error("应用参数不存在");
      }
      if (!StringUtil.isEmpty(secret)) {
        apiClient.setSecret(secret);
      }
      if (!StringUtil.isEmpty(roles)) {
        apiClient.setRoles(roles);
      }
      if (paramService.saveOrUpdateApiClient(apiClient)) {
        return JsonResult.ok("修改成功");
      } else {
        return JsonResult.error("修改失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/addapiclientpara")
  @PreAuthorize("hasPermission('/param/addapiclientpara','')")
  @ResponseBody
  public JsonResult addApiclientPara(@RequestParam("appid") String appid,
                                     @RequestParam("roles") String roles) {
    if (StringUtil.isEmpty(appid) || StringUtil.isEmpty(roles)) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TApiClient apiClient = paramService.getApiClientByAppid(appid);
      if (null != apiClient) {
        return JsonResult.error("APPID重复");
      }
      apiClient = new TApiClient();
      apiClient.setAppid(appid.trim());
      apiClient.setSecret(RandomUtils.getUUIDStr());
      apiClient.setStatus(TradeDict.STATUS_NORMAL);
      apiClient.setRoles(roles);
      if (paramService.saveOrUpdateApiClient(apiClient)) {
        return JsonResult.ok("新增成功");
      } else {
        return JsonResult.error("新增失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  /**
   * ====================================================
   * 支付能力配置
   * ====================================================
   */
  @GetMapping("/param/paytype")
  public String paytypeView() {
    return "system/param/paytype";
  }

  @GetMapping("/param/paytypelist")
  @PreAuthorize("hasPermission('/param/paytype','')")
  @ResponseBody
  public PageResult<TSourceType> getPaytypeDataList(@RequestParam("page") Integer pageNo,
                                                    @RequestParam("limit") Integer pageSize,
                                                    @RequestParam(value = "paytype", required = false) String paytype) {
    try {
      if (null == pageNo || pageNo < 1) pageNo = WebConstant.PAGENO_DEFAULT;
      if (null == pageSize || pageSize < 1) pageSize = WebConstant.PAGESIZE_DEFAULT;
      return paramService.getPaytypePage(paytype, pageNo, pageSize);
    } catch (Exception e) {
      e.printStackTrace();
      return new PageResult<>(99, "系统查询错误");
    }
  }

  @GetMapping("/param/load4addpaytype")
  @PreAuthorize("hasPermission('/param/load4addpaytype','')")
  public String load4AddPaytype(Model model) {
    return "system/param/paytypeform";
  }

  @PostMapping("/param/updatepaytypestate")
  @PreAuthorize("hasPermission('/param/updatepaytypestate','')")
  @ResponseBody
  public JsonResult updatePaytypeState(@RequestParam("paytype") String paytype,
                                       @RequestParam("state") Boolean state,
                                       @RequestParam(value = "optype", required = false) String optype) {
    if (StringUtil.isEmpty(paytype)
        || (!StringUtil.isEmpty(optype) && !"charge".equals(optype) && !"consume".equals(optype) && !"anonymous".equals(optype))) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TSourceType tPaytype = paramService.getPaytype(paytype);
      if (null == tPaytype) {
        return JsonResult.error("支付方式不存在！");
      }
      if ("charge".equals(optype)) {
        if (state.equals(tPaytype.isChargeEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        tPaytype.setChargeEnable(state);
      } else if ("consume".equals(optype)) {
        if (state.equals(tPaytype.getConsumeEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        tPaytype.setConsumeEnable(state);
      } else if ("anonymous".equals(optype)) {
        if (state.equals(tPaytype.getAnonymousEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        tPaytype.setAnonymousEnable(state);
      } else {
        if (state.equals(tPaytype.getEnable())) {
          return JsonResult.error("状态错误，请重新查询后操作");
        }
        tPaytype.setEnable(state); //默认切换主状态
      }

      if (paramService.saveOrUpdatePaytype(tPaytype)) {
        return JsonResult.ok(state ? "启用成功" : "关闭成功");
      } else {
        return JsonResult.error(!state ? "启用失败" : "关闭失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/updatepaytypename")
  @PreAuthorize("hasPermission('/param/updatepaytypename','')")
  @ResponseBody
  public JsonResult updatePaytypeName(@RequestParam("paytype") String paytype,
                                      @RequestParam("paydesc") String paydesc) {
    if (StringUtil.isEmpty(paytype) || StringUtil.isEmpty(paydesc)) {
      return JsonResult.error("参数传递错误");
    }
    try {
      TSourceType tPaytype = paramService.getPaytype(paytype);
      if (null == tPaytype) {
        return JsonResult.error("支付方式不存在！");
      }
      tPaytype.setPaydesc(paydesc.trim());
      if (paramService.saveOrUpdatePaytype(tPaytype)) {
        return JsonResult.ok("修改成功");
      } else {
        return JsonResult.error("修改失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/deletepaytype")
  @PreAuthorize("hasPermission('/param/deletepaytype','')")
  @ResponseBody
  public JsonResult deletePaytype(@RequestParam("paytype") String paytype) {
    try {
      TSourceType tPaytype = paramService.getPaytype(paytype);
      if (null == tPaytype) {
        return JsonResult.error("支付方式不存在！");
      }
      if (paramService.deletePaytype(tPaytype)) {
        return JsonResult.ok("删除成功");
      } else {
        return JsonResult.error("删除失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @PostMapping("/param/addpaytype")
  @PreAuthorize("hasPermission('/param/addpaytype','')")
  @ResponseBody
  public JsonResult addPaytype(@RequestParam("paytype") String paytype,
                               @RequestParam(value = "enable", required = false, defaultValue = "no") String enable,
                               @RequestParam(value = "chargeEnable", required = false, defaultValue = "no") String chargeEnable,
                               @RequestParam(value = "consumeEnable", required = false, defaultValue = "no") String consumeEnable,
                               @RequestParam(value = "anonymousEnable", required = false, defaultValue = "no") String anonymousEnable,
                               @RequestParam("paydesc") String paydesc) {
    try {
      TSourceType tPaytype = paramService.getPaytype(paytype);
      if (null != tPaytype) {
        return JsonResult.error("支付方式已经存在");
      }
      tPaytype = new TSourceType();
      tPaytype.setSourceType(paytype.trim());
      if (StringUtil.isEmpty(paydesc)) return JsonResult.error("支付名称不能为空！");
      tPaytype.setPaydesc(paydesc.trim());
      tPaytype.setEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(enable));
      tPaytype.setChargeEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(chargeEnable));
      tPaytype.setConsumeEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(consumeEnable));
      tPaytype.setAnonymousEnable(ConstantUtil.ENABLE_YES.equalsIgnoreCase(anonymousEnable));
      if (paramService.saveOrUpdatePaytype(tPaytype)) {
        return JsonResult.ok("新增成功");
      } else {
        return JsonResult.error("新增失败");
      }
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }

  @GetMapping("/param/checkpaytype")
  @ResponseBody
  public JsonResult checkPaytype(@RequestParam("paytype") String paytype) {
    TSourceType tPaytype = paramService.getPaytype(paytype);
    if (null != tPaytype) {
      return JsonResult.error("支付方式已经存在");
    } else {
      return JsonResult.ok("可以使用");
    }
  }

  @GetMapping("/param/load4paytypeconfig")
  @PreAuthorize("hasPermission('/param/load4paytypeconfig','')")
  public String load4PaytypeConfig(@RequestParam("paytype") String paytype, Model model) {
    List<TPaytypeConfig> configList = paramService.getPaytypeConfigList(paytype);
    model.addAttribute("configlist", configList);
    model.addAttribute("paytype", paytype);
    return "system/param/paytypeconfig";
  }

  @PostMapping("/param/addpaytypeconfig")
  @PreAuthorize("hasPermission('/param/addpaytypeconfig','')")
  @ResponseBody
  public JsonResult addPaytypeConfig(@RequestBody Map<String, String> param) {
    String paytypeHtmlKey = "hid_paytype"; //页面上传来paytype的KEY
    if (null == param || StringUtil.isEmpty(param.get(paytypeHtmlKey))) {
      return JsonResult.error("参数传递错误");
    }

    try {
      String paytype = param.get(paytypeHtmlKey).trim();
      param.remove(paytypeHtmlKey);
      if (paramService.savePaytypeConfig(paytype, param)) {
        return JsonResult.ok("配置成功");
      } else {
        return JsonResult.error("配置失败");
      }
    } catch (WebCheckException ex) {
      return JsonResult.error(ex.getMessage());
    } catch (Exception e) {
      e.printStackTrace();
      return JsonResult.error("系统处理异常").put("exception", e);
    }
  }


}
