package com.supwisdom.dlpay.system.service.impl;

import com.supwisdom.dlpay.api.dao.PaytypeConfigDao;
import com.supwisdom.dlpay.api.dao.SourceTypeDao;
import com.supwisdom.dlpay.api.dao.ShopPaytypeConfigDao;
import com.supwisdom.dlpay.api.dao.ShopPaytypeDao;
import com.supwisdom.dlpay.api.domain.TSourceType;
import com.supwisdom.dlpay.api.domain.TPaytypeConfig;
import com.supwisdom.dlpay.api.domain.TShopPaytype;
import com.supwisdom.dlpay.api.domain.TShopPaytypeConfig;
import com.supwisdom.dlpay.framework.dao.ShopDao;
import com.supwisdom.dlpay.framework.dao.ShopaccDao;
import com.supwisdom.dlpay.framework.data.SystemDateTime;
import com.supwisdom.dlpay.framework.domain.TShop;
import com.supwisdom.dlpay.framework.domain.TShopacc;
import com.supwisdom.dlpay.framework.service.SystemUtilService;
import com.supwisdom.dlpay.framework.util.PageResult;
import com.supwisdom.dlpay.framework.util.StringUtil;
import com.supwisdom.dlpay.framework.util.Subject;
import com.supwisdom.dlpay.framework.util.TradeDict;
import com.supwisdom.dlpay.system.bean.ShopConfigBean;
import com.supwisdom.dlpay.system.bean.ZTreeNode;
import com.supwisdom.dlpay.system.service.ShopDataService;
import com.supwisdom.dlpay.util.ConstantUtil;
import com.supwisdom.dlpay.util.WebCheckException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.data.jpa.domain.Specification;

import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Service
public class ShopDataServiceImpl implements ShopDataService {
  @Autowired
  private ShopDao shopDao;
  @Autowired
  private ShopaccDao shopaccDao;
  @Autowired
  private ShopPaytypeDao shopPaytypeDao;
  @Autowired
  private SystemUtilService systemUtilService;
  @Autowired
  private SourceTypeDao paytypeDao;
  @Autowired
  private PaytypeConfigDao paytypeConfigDao;
  @Autowired
  private ShopPaytypeConfigDao shopPaytypeConfigDao;

  @Override
  public List<ZTreeNode> getAllShopNodes() {
    List<ZTreeNode> result = new ArrayList<>(0);
    List<TShop> shoplist = shopDao.getNormalShops();
    if (!StringUtil.isEmpty(shoplist)) {
      for (TShop shop : shoplist) {
        if (null == shop || !TradeDict.STATUS_NORMAL.equals(shop.getStatus())) continue; //跳过注销商户
        ZTreeNode node = new ZTreeNode();
        node.setId(shop.getShopid().toString());
        node.setName(shop.getShopid() + "_" + shop.getShopname());
        node.setpId(shop.getFshopid() == null ? "" : shop.getFshopid().toString());
        node.setChecked(false);
        node.setOpen(true);
        node.setShoptype(shop.getShoptype());
        node.setShopaccno(StringUtil.isEmpty(shop.getShopaccno()) ? "" : shop.getShopaccno());
        node.setIconSkin(shop.getShoptype() == 0 ? "pIcon01" : "pIcon02");
        result.add(node);
      }
    }
    return result;
  }

  @Override
  public TShop getShopByShopid(Integer shopid) {
    if (null != shopid) {
      return shopDao.getTShopByShopid(shopid);
    }
    return null;
  }

  @Override
  public boolean deleteShop(TShop shop) throws WebCheckException {
    if (null != shop) {
      List<TShop> childShops = shopDao.getChildShopsByShopid(shop.getShopid());
      if (!StringUtil.isEmpty(childShops))
        throw new WebCheckException("请先删除下级商户");
      shop.setStatus(TradeDict.STATUS_CLOSED);
      shop.setClosedate(systemUtilService.getSysdatetime().getHostdate());
      shopDao.save(shop);
      if (!StringUtil.isEmpty(shop.getShopaccno())) {
        TShopacc shopacc = shopaccDao.getByShopaccno(shop.getShopaccno());
        if (null == shopacc) throw new WebCheckException("数据异常！对应的商户账户不存在！");
        shopacc.setStatus(TradeDict.STATUS_CLOSED);
        shopacc.setClosedate(shop.getClosedate());
        shopaccDao.save(shopacc);
      }
      return true;
    }
    return false;
  }

  @Override
  public boolean saveOrUpdateShop(TShop shop) throws WebCheckException {
    if (shop.getFshopid() != 0) {
      TShop fshop = shopDao.getTShopByShopid(shop.getFshopid());
      if (null == fshop) {
        throw new WebCheckException("上级商户不存在！");
      } else if (fshop.getShoptype() == 1) {
        throw new WebCheckException("上级商户不是商户组！");
      }
    }
    if (shopDao.checkShopnameExist(shop.getShopname(), shop.getFshopid(), shop.getShopid() == null ? 0 : shop.getShopid()) > 0) {
      throw new WebCheckException("商户名称在同级下已经存在！");
    }
    if (null != shop.getShopid() && shop.getShopid() > 0) {
      //修改
      shopDao.save(shop);
      if (!StringUtil.isEmpty(shop.getShopaccno())) {
        shopaccDao.updateShopnameByShopaccno(shop.getShopname(), shop.getShopaccno()); //更新商户名
      }
      return true;
    } else {
      //新增
      SystemDateTime dt = systemUtilService.getSysdatetime();
      shop.setOpendate(dt.getHostdate());
      shopDao.save(shop);
      if (shop.getShoptype() == 1) {
        TShopacc shopacc = new TShopacc();
        shopacc.setShopaccno(String.format("2%09d", shop.getShopid()));
        shopacc.setShopid(shop.getShopid());
        shopacc.setShopname(shop.getShopname());
        shopacc.setSubjno(Subject.SUBJNO_MACHANT_INCOME);
        shopacc.setStatus(TradeDict.STATUS_NORMAL);
        shopacc.setOpendate(dt.getHostdate());
        shopacc.setBalance(0D);
        shopaccDao.save(shopacc);

        shop.setShopaccno(shopacc.getShopaccno());
        shopDao.save(shop);

        TShopPaytype shopPaytype = new TShopPaytype();
        shopPaytype.setShopaccno(shopacc.getShopaccno());
        shopPaytype.setPaytype(TradeDict.PAYTYPE_BALANCE);
        shopPaytype.setConsumeEnable(true);
        shopPaytype.setAnonymousEnable(false);
        shopPaytype.setReverseEnable(false);
        shopPaytype.setCreatetime(dt.getHostdatetime());
        shopPaytypeDao.save(shopPaytype); //默认增加余额支付方式
      }
      return true;
    }
  }

  @Override
  public List<TSourceType> getConsumePaytypes() {
    List<TSourceType> list = paytypeDao.getConsumeSourceTypes();
    if (!StringUtil.isEmpty(list)) {
      return list;
    }
    return new ArrayList<TSourceType>(0);
  }

  @Override
  public PageResult<ShopConfigBean> getShopPaytypeInfos(String shopaccno, String paytype, int pageNo, int pageSize) {
    Pageable pageable = PageRequest.of(pageNo - 1, pageSize, Sort.by("shopaccno", "paytype"));

    Page<TShopPaytype> page = shopPaytypeDao.findAll(new Specification<TShopPaytype>() {
      @Override
      public Predicate toPredicate(Root<TShopPaytype> root, CriteriaQuery<?> query, CriteriaBuilder criteriaBuilder) {
        List<Predicate> predicates = new ArrayList<>();
        if (!StringUtil.isEmpty(shopaccno)) {
          predicates.add(criteriaBuilder.equal(root.get("shopaccno").as(String.class), shopaccno.trim()));
        }
        if (!StringUtil.isEmpty(paytype)) {
          predicates.add(criteriaBuilder.equal(root.get("paytype").as(String.class), paytype.trim()));
        }
        return criteriaBuilder.and(predicates.toArray(new Predicate[0]));
      }
    }, pageable);

    if (null == page || page.getTotalElements() == 0) {
      return new PageResult<>(99, "无数据");
    }
    List<ShopConfigBean> list = new ArrayList<>(0);
    for (TShopPaytype pt : page.getContent()) {
      ShopConfigBean bean = new ShopConfigBean();
      bean.setShopaccno(pt.getShopaccno());
      TShopacc shopacc = shopaccDao.getByShopaccno(pt.getShopaccno());
      bean.setShopname(shopacc == null ? "" : shopacc.getShopname());
      bean.setPaytype(pt.getPaytype());
      TSourceType tPaytype = paytypeDao.getBySourceType(pt.getPaytype());
      bean.setPaydesc(tPaytype == null ? "" : tPaytype.getPaydesc());
      bean.setConsumeEnable(pt.getConsumeEnable());
      bean.setAnonymousEnable(pt.getAnonymousEnable());
      bean.setReverseEnable(pt.getReverseEnable());
      list.add(bean);
    }
    return new PageResult<>(page.getTotalElements(), list);
  }

  @Override
  public TShopPaytype getShopPaytype(String shopaccno, String paytype) {
    return shopPaytypeDao.getById(paytype, shopaccno);
  }

  @Override
  public TShopacc getShopaccByAccno(String shopaccno) {
    if (!StringUtil.isEmpty(shopaccno)) {
      return shopaccDao.getByShopaccno(shopaccno.trim());
    }
    return null;
  }

  @Override
  public boolean saveOrUpdateShopPaytype(TShopPaytype shopPaytype) {
    if (null != shopPaytype) {
      shopPaytypeDao.save(shopPaytype);
      return true;
    }
    return false;
  }

  @Override
  public List<TShopPaytypeConfig> getShopPaytypeConfigs(String shopaccno, String paytype) {
    List<TShopPaytypeConfig> result = new ArrayList<>(0);
    if (!StringUtil.isEmpty(paytype)) {
      List<TPaytypeConfig> list = paytypeConfigDao.getByPaytypeOrderByConfigid(paytype.trim());
      if (!StringUtil.isEmpty(list)) {
        for (TPaytypeConfig pt : list) {
          TShopPaytypeConfig spc = shopPaytypeConfigDao.getShopPaytypeConfigById(shopaccno, pt.getPaytype(), pt.getConfigid());
          if (null == spc) {
            spc = new TShopPaytypeConfig();
            spc.setShopaccno(shopaccno);
            spc.setPaytype(pt.getPaytype());
            spc.setConfigid(pt.getConfigid());
            spc.setConfigName(pt.getConfigName());
            spc.setConfigValue(null);
          }
          result.add(spc);
        }
      }
    }
    return result;
  }

  @Override
  public boolean saveOrUpdateShopPaytypeConfig(String shopaccno, String paytype, Map<String, String> param) throws WebCheckException {
    TShopPaytype shopPaytype = getShopPaytype(shopaccno, paytype);
    if (null == shopPaytype) {
      throw new WebCheckException("此商户[" + shopaccno + "]还不具有该支付能力[" + paytype + "]");
    }
    for (String key : param.keySet()) {
      String value = param.get(key);
      TShopPaytypeConfig spc = shopPaytypeConfigDao.getShopPaytypeConfigById(shopPaytype.getShopaccno(), shopPaytype.getPaytype(), key);
      if (null != spc) {
        spc.setConfigValue(StringUtil.isEmpty(value) ? null : value.trim());
      } else {
        spc = new TShopPaytypeConfig();
        spc.setShopaccno(shopPaytype.getShopaccno());
        spc.setPaytype(shopPaytype.getPaytype());
        spc.setConfigid(key);
        spc.setConfigValue(StringUtil.isEmpty(value) ? null : value.trim());
        TPaytypeConfig paytypeConfig = paytypeConfigDao.getByPaytypeAndAndConfigid(shopPaytype.getPaytype(), key);
        if (null != paytypeConfig) {
          spc.setConfigName(paytypeConfig.getConfigName());
        }
      }
      shopPaytypeConfigDao.save(spc);
    }
    return true;
  }

}
