package com.supwisdom.dlpay.consume.controller

import com.supwisdom.dlpay.consume.AccountHolder
import com.supwisdom.dlpay.consume.CallService
import com.supwisdom.dlpay.consume.PersonTransBuilder
import com.supwisdom.dlpay.consume.service.AccountUtilServcie
import com.supwisdom.dlpay.consume.service.PaytypeService
import com.supwisdom.dlpay.consume.service.PersonBalancePayService
import com.supwisdom.dlpay.consume.service.PersonService
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.util.PaytypeUtil
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RestController

@RestController
class ConsumeController {
    @Autowired
    lateinit var accountUtilServcie: AccountUtilServcie

    @Autowired
    lateinit var personBalancePayService: PersonBalancePayService

    @Autowired
    lateinit var paytypeService: PaytypeService
    @Autowired
    lateinit var personService: PersonService

    @PostMapping("/account/pay")
    fun accountPay(userid: String, amount: Int, manageFee: Int): ResponseEntity<Any> {
        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                .setTransDatetime("20190411", "112311")
                .enableOverdraft(false)
                .addDetail(AccountHolder.person(userid),
                        AccountHolder.shop("12323"),
                        amount / 100.0, "")
                .addDetail(AccountHolder.person(userid), AccountHolder.transType(301),
                        manageFee / 100.0, "")
                .done(personBalancePayService, true)
        return ResponseEntity.ok(dtl)
    }

    @GetMapping("/account/payinit")
    fun accountPayInit(userid: String, amount: Int, manageFee: Int): ResponseEntity<Any> {
        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                .setTransDatetime("20190411", "112311")
                .enableOverdraft(false)
                .addDetail(AccountHolder.person(userid),
                        AccountHolder.shop("12323"),
                        amount / 100.0, "")
                .addDetail(AccountHolder.person(userid), AccountHolder.transType(301),
                        manageFee / 100.0, "")
                .done(personBalancePayService, false)

        return ResponseEntity.ok(dtl)
    }

    @GetMapping("/account/payfinish")
    fun accountPayFinish(refno: String): ResponseEntity<Any> {
        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                .done(refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
        return ResponseEntity.ok(dtl)
    }

    @PostMapping("/ykt/payinit")
    fun yktPayInit(userid: String, amount: Int, manageFee: Int,
                   stuempno: String, shopid: String, transdate: String, transtime: String,
                   outtradeno: String, payinfo: String, feetype: String): ResponseEntity<Any> {
        //一卡通支付款 112240
        return try {
            val person = personService.getPersonByThirdUniqueIdenty(stuempno)
            val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .setTransDatetime(transdate, transtime)
                    .selectPaytype(PaytypeUtil.YKTPAY, payinfo)
                    .setOuttradeno(outtradeno)
                    .setOwner(person)
                    .tryLock(true)
                    .setTransinfo(TradeCode.TRANSCODE_YKTPAY, "一卡通支付")
                    .chooseTradetype(Tradetype.CONSUME)
                    .also {
                        when (feetype) {
                            TradeDict.FEETYPE_CONSUME_MEALER -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                        AccountHolder.shop(shopid),
                                        amount / 100.0, "一卡通支付")
                                        .addDetail(AccountHolder.transType(TranstypeCode.TT_CONSUUME_MANAGE_FEE)
                                                .with(AccountHolder.shop(shopid)),
                                                manageFee / 100.0)
                            }
                            TradeDict.FEETYPE_CONSUME_DISCOUNT -> {
                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                        AccountHolder.shop(shopid),
                                        (amount - manageFee) / 100.0, "一卡通支付")
                                        .addDetail(AccountHolder.subject(Subject.SUBJNO_CONSUME_DISCOUNT),
                                                AccountHolder.shop(shopid),
                                                manageFee / 100.0, "优惠折扣")
                            }
                        }
                    }.done(personBalancePayService, false)
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .success()
                    .data("refno", dtl.refno)
                    .build())
        } catch (e: TransactionException) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(99, e, "查询异常")
                    .build())
        }
    }

    @PostMapping("/ykt/payfinish")
    fun yktPayFinish(refno: String, yktshopid: String, devphyid: String?): ResponseEntity<Any> {
        val dtl = personBalancePayService.wip(refno)
        val person = personService.getPersonByUserid(dtl.userid)
        val code = CallService.callYktPay(paytypeService.getPaytypeConfigByPaytype(PaytypeUtil.YKTPAY),
                dtl, DateUtil.getNow(), person.thirdUniqueIdenty, yktshopid, devphyid)
        return if (code.retcode == "0") {
            val suc = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .done(dtl.refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
            ResponseEntity.ok(suc)
        } else {
            val fail = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .done(dtl.refno, TradeDict.DTL_STATUS_FAIL, personBalancePayService)
            ResponseEntity.ok(fail)
        }
    }
}