package com.supwisdom.dlpay.api.controller

import com.supwisdom.dlpay.api.*
import com.supwisdom.dlpay.api.bean.*
import com.supwisdom.dlpay.api.domain.TAccount
import com.supwisdom.dlpay.api.service.*
import com.supwisdom.dlpay.exception.RequestParamCheckException
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.exception.TransactionProcessException
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.domain.TShopacc
import com.supwisdom.dlpay.framework.service.CommonService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.util.ConstantUtil
import com.supwisdom.dlpay.util.PaytypeUtil
import org.apache.catalina.authenticator.SpnegoAuthenticator
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.web.bind.annotation.*
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@RestController
@RequestMapping("/api/consume")
class ConsumeController {
    @Autowired
    lateinit var accountUtilServcie: AccountUtilServcie
    @Autowired
    lateinit var personBalancePayService: PersonBalancePayService
    @Autowired
    lateinit var paytypeService: PaytypeService
    @Autowired
    lateinit var userService: UserService
    @Autowired
    lateinit var systemUtilService: SystemUtilService
    @Autowired
    lateinit var consumePayService: ConsumePayService
    @Autowired
    lateinit var commonService: CommonService

    @Autowired
    lateinit var transactionService: TransactionService

    /**
     * 流水结果查询统一接口
     * */
    fun queryDtlResult(@RequestBody param: QueryDtlResultParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val dtl = if (StringUtil.isEmpty(param.refno)) {
                personBalancePayService.getUserdtlForUpdateNowait(param.refno!!)
            } else {
                personBalancePayService.getUserdtlByBillnoForUpdateNowait(param.billno!!, param.shopaccno!!)
            }
            val person = dtl.userid?.let { userService.findPersonByUserid(dtl.userid) }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data("refno", dtl.refno)
                    .data("amount", dtl.amount)
                    .data("paytype", dtl.paytype)
                    .data("payinfo", dtl.payinfo)
                    .also {
                        if (null != person) {
                            it.data("name", person.name)
                        }
                    }
                    .success("查询成功"))

        } catch (ex: RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "查询错误"))
        }

    }

    private fun consumeMealer(param: CitizenCardPayinitParam, builder: TransactionBuilder,
                              feetype: ConsumeFeetype, person: TAccount, shop: TShopacc) {
        val feetypeConfig = accountUtilServcie.readFeetype(TradeDict.FEETYPE_CONSUME_MEALER,
                TradeDict.PAYTYPE_BALANCE)
        val amount = feetype.amount / 100.0
        when (feetypeConfig.crsubjno) {
            shop.subjno -> {
                builder.addDebitCreditRecord(AccountProxy(person), AccountProxy(shop),
                        amount, feetypeConfig.summary)
                        .shop().also {
                            it.amount += amount
                        }
            }
            else -> {
                val subject = accountUtilServcie.readSubject(feetypeConfig.crsubjno)
                builder.addDebitCreditRecord(AccountProxy(person), AccountProxy(subject),
                        amount, feetypeConfig.summary)
            }
        }

    }

    private fun consumeDiscount(param: CitizenCardPayinitParam, builder: TransactionBuilder,
                                feetype: ConsumeFeetype, person: TAccount, shop: TShopacc) {
        val feetypeConfig = accountUtilServcie.readFeetype(TradeDict.FEETYPE_CONSUME_MEALER,
                TradeDict.PAYTYPE_BALANCE)
        val amount = feetype.amount / 100.0
        when (feetypeConfig.drsubjno) {
            shop.subjno -> {
                builder.addDebitCreditRecord(AccountProxy(shop), AccountProxy(person),
                        amount, feetypeConfig.summary)
            }
            else -> {
                val subject = accountUtilServcie.readSubject(feetypeConfig.drsubjno)
                builder.addDebitCreditRecord(AccountProxy(subject), AccountProxy(shop),
                        amount, feetypeConfig.summary)
            }
        }
    }

    /**
     * ============================================================================
     *                           账户【余额支付】
     * ============================================================================
     * */
    @PostMapping("/balance/pay")
    fun balancePay(@RequestBody param: CitizenCardPayinitParam, authentication: Authentication): ResponseEntity<Any> {
        try {
            if (param.checkParam() && param.checkSign(commonService.getSecretByAppid(authentication.name))) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val person = userService.findPersonByIdentityCheckStatus(param.cardNo)
            if (consumePayService.checkShopPaytype(param.shopaccno, TradeDict.PAYTYPE_BALANCE)) {
                val account = accountUtilServcie.readAccount(person.userid)
                val shopacc = accountUtilServcie.readShopbyAccno(param.shopaccno)

                val builder = TransactionBuilder().apply {
                    transDate = param.transdate
                    transTime = param.transtime
                    paytype = TradeDict.PAYTYPE_BALANCE
                    transCode = TradeCode.TRANSCODE_BALANCE_PAY
                }.person(account).apply {
                    description = "账户余额支付"
                    summary = "账户余额消费"
                    amount = param.amount / 100.0
                    opposite = AccountProxy(shopacc)
                }.and().shop(shopacc).apply {
                    summary = "账户余额消费"
                    amount = param.amount / 100.0
                    opposite = AccountProxy(account)
                }.and()

                param.feelist?.forEach {
                    when (it.feetype) {
                        TradeDict.FEETYPE_CONSUME_MEALER -> {
                            consumeMealer(param, builder, it, account, shopacc)
                        }
                        TradeDict.FEETYPE_CONSUME_DISCOUNT -> {
                            consumeDiscount(param, builder, it, account, shopacc)
                        }
                        else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                "支付费率清单不支持feetype<${it.feetype}>")
                    }
                }

                val dtl = builder.init(transactionService)
                transactionService.success(dtl.refno)

                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("amount", dtl.personDtl.amount)
                        .success("交易成功"))
            }

            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_PAYTYPE_NOSUPPORT, "不支持支付方式<账户余额>"))

        } catch (ex: RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }


    /**
     * ============================================================================
     *                           市民卡【交易初始化】
     * ============================================================================
     * */
    @PostMapping("/citizencard/payinit")
    fun citizencardPayinit(@RequestBody param: CitizenCardPayinitParam, authenticateAction: Authentication): ResponseEntity<Any> {
        try {
            if (param.checkParam() && param.checkSign(commonService.getSecretByAppid(authenticateAction.name))) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val person = userService.findPersonByIdentityCheckStatus(param.cardNo)
            if (consumePayService.checkShopPaytype(param.shopaccno, TradeDict.PAYTYPE_CITIZEN_CARD)) {

                val account = accountUtilServcie.readAccount(person.userid)
                val shopacc = accountUtilServcie.readShopbyAccno(param.shopaccno)
                val subject = accountUtilServcie.readSubject(Subject.SUBJNO_PAY_CITIZEN_CARD)
                val transaction = TransactionBuilder().apply {
                    transCode = TradeCode.TRANSCODE_CITIZENCARD_PAY
                    transDate = param.transdate
                    transTime = param.transtime
                    paytype = TradeDict.PAYTYPE_CITIZEN_CARD
                    payinfo = "市民卡代扣消费"
                }.person(account).apply {
                    amount = param.amount / 100.0
                    opposite = AccountProxy(shopacc)
                    summary = "市民卡代扣消费"
                }.and().shop(shopacc).apply {
                    amount = param.amount / 100.0
                    opposite = AccountProxy(account)
                    summary = "市民卡代扣消费"
                }.and().addDebitCreditRecord(AccountProxy(subject), AccountProxy(shopacc),
                        param.amount / 100.0, "市民卡代扣消费")
                        .also { builder ->
                            param.feelist?.forEach {
                                //fixme: 科目 -> 商户 与个人无关
                                val feeconfig = accountUtilServcie.readFeetype(it.feetype,
                                        TradeDict.PAYTYPE_CITIZEN_CARD)
                                val subject = accountUtilServcie.readSubject(feeconfig.drsubjno)
                                builder.addDebitCreditRecord(AccountProxy(subject), AccountProxy(shopacc),
                                        it.amount / 100.0, feeconfig.summary)

                            }
                        }.init(transactionService)

                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", transaction.refno)
                        .data("amount", transaction.shopDtl.amount)
                        .success("交易初始化成功"))
            }

            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_PAYTYPE_NOSUPPORT, "不支持支付方式<市民卡代扣>"))

        } catch (ex: RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

    /**
     * ============================================================================
     *                           市民卡【交易确认】
     * ============================================================================
     * */
    @PostMapping("/citizencard/payfinish")
    fun citizencardPayinit(@RequestBody param: CitizenCardPayfinishParam, authentication: Authentication): ResponseEntity<Any> {
        try {
            if (param.checkParam() && param.checkSign(commonService.getSecretByAppid(authentication.name))) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val code = transactionService.wip(param.refno).let {
                CallService.CallCitizenCardPay(
                        consumePayService.getPaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, it.shopDtl.shopaccno),
                        it.shopDtl)
            }
            if (code.retcode == "0") {
                transactionService.success(param.refno).let {
                    return ResponseEntity.ok(ResponseBodyBuilder.create()
                            .data("refno", it.refno)
                            .data("billno", it.outTradeNo)
                            .success("交易确认成功"))
                }
            } else {
                transactionService.fail(param.refno).let {
                    return ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(TradeErrorCode.BUSINESS_DEAL_ERROR, "交易扣费失败-${code.retmsg}"))
                }
            }
        } catch (ex: RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }


    /**
     * ============================================================================
     *                           一卡通支付【交易初始化】
     * ============================================================================
     * */
    @PostMapping("/ykt/payinit")
    fun yktPayInit(@RequestBody param: YktCardPayinitParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            val person = param.uid?.let { userService.findByThirdUniqueIdenty(it) } //没注册，可能匿名?
            if (consumePayService.checkShopPaytype(param.shopaccno, TradeDict.PAYTYPE_YKT_CARD, person == null)) {
                val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                        .chooseTradetype(Tradetype.CONSUME) //消费
                        .also {
                            if (null != person) it.setOwner(person)
                        }
                        .setTransinfo(TradeCode.TRANSCODE_YKTCARD_PAY, "一卡通支付")
                        .setTransDatetime(param.transdate, param.transtime) //交易时间
                        .selectPaytype(TradeDict.PAYTYPE_YKT_CARD, param.stuempno)
                        .addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_YKT),
                                AccountHolder.shop(param.shopaccno),
                                param.amount / 100.0, "一卡通支付")
                        .also { builder ->
                            param.feelist?.forEach {
                                //fixme: 科目 -> 商户 与个人无关
                                builder.addDetail(AccountHolder.feetype(it.feetype, TradeDict.PAYTYPE_YKT_CARD)
                                        .with(AccountHolder.shop(param.shopaccno))
                                        .with(AccountHolder.subject(Subject.SUBJNO_PAY_YKT))
                                        , it.amount / 100.0)
                            }
                        }
                        .addExtendParam("stuempno", param.stuempno)
                        .addExtendParam("yktshopid", param.yktshopid ?: "")
                        .addExtendParam("devphyid", param.devphyid ?: "")
                        //.addExtendParam(param.extendmap)  //fixme: 保存调一卡通附加参数 (是否直接传附加参数map)
                        .init(personBalancePayService)

                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("amount", dtl.amount)
                        .success("交易初始化成功"))

            }

            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(TradeErrorCode.BUSINESS_PAYTYPE_NOSUPPORT, "不支持支付方式<一卡通支付>"))
        } catch (ex: RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }


    /**
     * ============================================================================
     *                           一卡通支付【交易确认】
     * ============================================================================
     * */
    @PostMapping("/ykt/payfinish")
    fun yktPayFinish(@RequestBody param: CitizenCardPayfinishParam, request: HttpServletRequest, response: HttpServletResponse): ResponseEntity<Any> {
        try {
            if (param.checkParam() && param.checkSign(commonService.getAppidSecretByRequest(request))) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误"))
            }

            var dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                    .setRefno(param.refno)
                    .wip(personBalancePayService)
            val extendMap = consumePayService.getUserdtlExtendParamMap(dtl.refno)
            val code = CallService.callYktPay(consumePayService.getPaytypeConfig(TradeDict.PAYTYPE_YKT_CARD, dtl.shopaccno, dtl.userid == null),
                    dtl, DateUtil.getNow(), extendMap["stuempno"]!!, extendMap["yktshopid"]!!, extendMap["devphyid"])
            if (code.retcode == "0") {
                dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
                        .setRefno(param.refno)
                        .addResult(code.data)
                        .success(personBalancePayService) //流水置成功

                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("refno", dtl.refno)
                        .data("billno", dtl.outtradeno)
                        .success())
            } else {
                PersonTransBuilder.newBuilder(accountUtilServcie)
                        .setRefno(param.refno)
                        .addResult("errmsg", code.retmsg!!)
                        .finish(personBalancePayService, TradeDict.DTL_STATUS_FAIL) //流水置成功

                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.BUSINESS_DEAL_ERROR, "交易扣费失败-${code.retmsg}"))
            }

        } catch (ex: RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(ex, "请求参数错误"))
        } catch (et: TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .transException(et, "业务处理错误"))
        }
    }

// ============================================== //
//
//    @GetMapping("/account/payinit")
//    fun accountPayInit(userid: String, amount: Int, manageFee: Int): ResponseEntity<Any> {
//        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
//                .setTransDatetime("20190411", "112311")
//                .enableOverdraft(false)
//                .addDetail(AccountHolder.person(userid),
//                        AccountHolder.shop("12323"),
//                        amount / 100.0, "")
//                .addDetail(AccountHolder.person(userid), AccountHolder.transType(301),
//                        manageFee / 100.0, "")
//                .done(personBalancePayService, false)
//
//        return ResponseEntity.ok(dtl)
//    }
//
//    @GetMapping("/account/payfinish")
//    fun accountPayFinish(refno: String): ResponseEntity<Any> {
//        val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
//                .done(refno, TradeDict.DTL_STATUS_SUCCESS, personBalancePayService)
//        return ResponseEntity.ok(dtl)
//    }


//
//    /**
//     * 微信支付
//     * wechattype
//     * qrcode-扫微信二维码支付
//     * app-原生app微信支付
//     * mp-微信公众号支付
//     * h5-微信h5支付
//     *
//     * */
//    @PostMapping("/wechat/payinit")
//    fun wechatPayInit(userid: String, amount: Int, manageFee: Int,
//                      stuempno: String, shopid: String, transdate: String, transtime: String,
//                      outtradeno: String, payinfo: String, feetype: String,
//                      wechattype: String, realip: String?, qrcode: String?, openid: String?): ResponseEntity<Any> {
//        return try {
//            val paytype = paytypeService.getByPaytype(PaytypeUtil.WECHAT)
//            if (paytype == null || ConstantUtil.ENABLE_YES != paytype.enable) {
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .fail(1, "支付方式未开启"))
//            }
//            val person = userService.findByThirdUniqueIdenty(stuempno)
//            val dtl = PersonTransBuilder.newBuilder(accountUtilServcie)
//                    .setTransDatetime(transdate, transtime)
//                    .selectPaytype(PaytypeUtil.WECHAT, payinfo)
//                    .setOuttradeno(outtradeno)
//                    .also {
//                        if (null != person) it.setOwner(person)
//                    }
//                    .tryLock(true)
//                    .setTransinfo(TradeCode.TRANSCODE_WECHAT, "微信支付")
//                    .chooseTradetype(Tradetype.CONSUME)
//                    .also {
//                        when (feetype) {
//                            TradeDict.FEETYPE_CONSUME_MEALER -> {
//                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
//                                        AccountHolder.shop(shopid),
//                                        amount / 100.0, "微信支付")
//                                        .addDetail(AccountHolder.transType(TranstypeCode.TT_CONSUUME_MANAGE_FEE)
//                                                .with(AccountHolder.shop(shopid)),
//                                                manageFee / 100.0)
//                            }
//                            TradeDict.FEETYPE_CONSUME_DISCOUNT -> {
//                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
//                                        AccountHolder.shop(shopid),
//                                        (amount - manageFee) / 100.0, "微信支付")
//                                        .addDetail(AccountHolder.subject(Subject.SUBJNO_CONSUME_DISCOUNT),
//                                                AccountHolder.shop(shopid),
//                                                manageFee / 100.0, "优惠折扣")
//                            }
//                            else -> {
//                                it.addDetail(AccountHolder.subject(Subject.SUBJNO_PAY_WECHAT),
//                                        AccountHolder.shop(shopid),
//                                        amount / 100.0, "微信支付")
//                            }
//                        }
//                    }.done(personBalancePayService, false)
//            val code = CallService.callWechatPay(paytypeService.getPaytypeConfigByPaytype(PaytypeUtil.WECHAT),
//                    dtl, DateUtil.getNow(), wechattype, realip, qrcode, openid)
//            if (code.retcode == "0") {
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .data("refno", dtl.refno)
//                        .success())
//            } else {
//                ResponseEntity.ok(ResponseBodyBuilder.create()
//                        .fail(TradeErrorCode.TRANSACTION_NOT_EXISTS, "交易请求失败-${code.retcode}"))
//            }
//        } catch (e: TransactionException) {
//            ResponseEntity.ok(ResponseBodyBuilder.create()
//                    .transException(e, "交易初始化异常"))
//        }
//    }
}