package com.supwisdom.dlpay.api.service.impl

import com.supwisdom.dlpay.api.TransactionBuilder
import com.supwisdom.dlpay.api.dao.*
import com.supwisdom.dlpay.api.domain.*
import com.supwisdom.dlpay.api.repositories.ShopaccRepositoryImpl
import com.supwisdom.dlpay.api.repositories.TAccountRepositoryImpl
import com.supwisdom.dlpay.api.service.PersonAccountService
import com.supwisdom.dlpay.api.service.TransactionService
import com.supwisdom.dlpay.exception.TransactionProcessException
import com.supwisdom.dlpay.framework.util.Subject
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import java.sql.SQLException
import javax.transaction.Transactional


open class PersonAccountServiceImpl : PersonAccountService {
    @Transactional
    override fun recalcBalance(account: TAccount, amount: Double) {
        TODO("not implemented") //To change body of created functions use File | Settings | File Templates.
    }
}

@Service
class TransactionServiceImpl : TransactionService {

    companion object {
        val INIT_DTL_STATUS = setOf(TradeDict.DTL_STATUS_INIT, TradeDict.DTL_STATUS_WIP)
    }

    @Autowired
    lateinit var transactionMainDao: TransactionMainDao

    @Autowired
    lateinit var accoutDao: AccountDao

    @Autowired
    lateinit var persondtlDao: PersondtlDao

    @Autowired
    lateinit var shopdtlDao: ShopdtlDao

    @Autowired
    lateinit var subjectdtlDao: SubjectdtlDao

    @Autowired
    lateinit var debitCreditDtlDao: DebitCreditDtlDao

    @Autowired
    lateinit var accountRepository: TAccountRepositoryImpl

    @Autowired
    lateinit var shopaccRepository: ShopaccRepositoryImpl

    private fun builderRecords(builder: TransactionBuilder, status: String): TTransactionMain {
        // 记录三方的交易流水（个人，商户，科目)
        try {
            val transaction = TTransactionMain().apply {
                refno = "20000000000001"
                accdate = "20190520"
            }


            if (builder.hasPerson()) {
                TPersondtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    userid = builder.person().person.userid
                    userName = builder.person().person.accname
                    transdate = builder.transDate
                    transtime = builder.transTime
                    befbal = builder.person().person.availbal
                    amount = builder.person().amount
                    paytype = builder.person().paytype
                    payinfo = builder.person().payinfo
                    transcode = builder.transCode
                    oppositeAccNo = builder.person().opposite.getAccountNo()
                    oppositeAccName = builder.person().opposite.getAccountName()
                    this.status = status
                }.also {
                    // save persondtl
                    persondtlDao.save(it)
                    transaction.personDtl = it
                    transaction.person = true
                }
            }
            if (builder.hasShop()) {
                TShopdtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    this.amount = builder.shop().amount
                    this.payInfo = builder.shop().payinfo
                    this.payType = builder.shop().paytype
                    this.transDate = builder.transDate
                    this.transTime = builder.transTime
                    this.tradeCode = builder.transCode
                    this.shopaccno = builder.shop().shopacc.shopaccno
                    this.shopname = builder.shop().shopacc.shopname
                    this.oppositeAccNo = builder.shop().opposite.getAccountNo()
                    this.oppositeAccName = builder.shop().opposite.getAccountName()
                    this.status = status
                }.also {
                    // save shopdtl
                    shopdtlDao.save(it)
                    transaction.shopDtl = it
                    transaction.shop = true
                }
            }
            if (builder.hasSubject()) {
                // save subjectdtl
                TSubjectdtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    this.amount = builder.subject().amount
                    this.payInfo = builder.subject().payinfo
                    this.payType = builder.subject().paytype
                    this.tradeCode = builder.transCode
                    this.transDate = builder.transDate
                    this.transTime = builder.transTime
                    this.oppositeAccNo = builder.subject().opposite.getAccountNo()
                    this.oppositeAccName = builder.subject().opposite.getAccountName()
                    this.status = status
                }.also {
                    subjectdtlDao.save(it)
                    transaction.subjectDtl = it
                    transaction.subject = true
                }
            }

            builder.getAllDetails().map { line ->
                TDebitCreditDtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    this.seqno = line.seqno
                    this.draccno = line.debit.getAccountNo()
                    this.drsubjno = line.debit.getSubjectNo()
                    this.craccno = line.credit.getAccountNo()
                    this.crsubjno = line.credit.getSubjectNo()
                    this.amount = line.amount
                    this.summary = line.summary
                }.also {
                    debitCreditDtlDao.save(it)
                }
            }.apply {
                transaction.details = this
            }

            val details = debitCreditDtlDao.findByRefno(transaction.refno)

            if (builder.hasPerson()) {
                sumBalanceFromDetails(details, builder.person().person.accno,
                        Subject.SUBJNO_PERSONAL_DEPOSIT).also {
                    if (transaction.personDtl.amount != it) {
                        throw TransactionProcessException(TradeErrorCode.INPUT_DATA_ERROR,
                                "输入金额错误，个人余额不符<${transaction.personDtl.amount}>")
                    }
                }
            }

            if (builder.hasShop()) {
                sumBalanceFromDetails(details, builder.shop().shopacc.shopaccno,
                        Subject.SUBJNO_MACHANT_INCOME).also {
                    if (transaction.shopDtl.amount != it) {
                        throw TransactionProcessException(TradeErrorCode.INPUT_DATA_ERROR,
                                "输入金额错误，商户余额不符<${transaction.shopDtl.amount}>")
                    }
                }
            }

            if (builder.hasSubject()) {
                sumBalanceFromDetails(details, builder.subject().subject.subjno,
                        transaction.subjectDtl.subjectno).also {
                    if (transaction.subjectDtl.amount != it) {
                        throw TransactionProcessException(TradeErrorCode.INPUT_DATA_ERROR,
                                "输入金额错误，科目余额不符<${transaction.subjectDtl.amount}>")
                    }
                }
            }

            transactionMainDao.save(transaction)
            return transaction
        } catch (ex: SQLException) {
            throw TransactionProcessException(TradeErrorCode.BUSINESS_DEAL_ERROR, ex.message)
        }
    }

    override fun init(builder: TransactionBuilder): TTransactionMain {
        return builderRecords(builder, TradeDict.DTL_STATUS_INIT)
    }

    override fun wip(builder: TransactionBuilder): TTransactionMain {
        return builderRecords(builder, TradeDict.DTL_STATUS_WIP)
    }

    private fun sumBalanceFromDetails(details: List<TDebitCreditDtl>,
                                      accno: String, subjno: String): Double {
        return details.sumByDouble { line ->
            if (line.crsubjno == subjno && line.craccno == accno) {
                line.amount
            } else if (line.crsubjno == subjno && line.craccno == accno) {
                -line.amount
            } else {
                0.0
            }
        }
    }

    override fun wip(refno: String): TTransactionMain {
        val transaction = transactionMainDao.findById(refno).let {
            if (it.isPresent) it.get() else null
        } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>参考号错误")

        if (transaction.status != TradeDict.DTL_STATUS_INIT) {
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>状态错误")
        }
        if (transaction.person) {
            persondtlDao.findById(refno).let {
                if (!it.isPresent) it.get() else null
            }?.also {
                if (it.status != TradeDict.DTL_STATUS_INIT) {
                    throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "个人流水<$refno>状态错误")
                }
                it.status = TradeDict.DTL_STATUS_WIP
                persondtlDao.save(it)
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS, "个人流水<$refno>不存在")
        }

        if (transaction.shop) {
            shopdtlDao.findById(refno).let {
                if (it.isPresent) it.get() else null
            }?.also {
                if (it.status != TradeDict.DTL_STATUS_INIT) {
                    throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "商户流水<$refno>状态错误")
                }
                it.status = TradeDict.DTL_STATUS_WIP
                shopdtlDao.save(it)
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS, "商户流水<$refno>不存在")
        }

        if (transaction.subject) {
            subjectdtlDao.findById(refno).let {
                if (it.isPresent) it.get() else null
            }?.also {
                if (it.status != TradeDict.DTL_STATUS_INIT) {
                    throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "科目流水<$refno>状态错误")
                }
                it.status = TradeDict.DTL_STATUS_WIP
                subjectdtlDao.save(it)
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS, "科目流水<$refno>不存在")
        }
        transactionMainDao.save(transaction)
        return transaction
    }

    override fun fail(refno: String): TTransactionMain {
        val transaction = transactionMainDao.findByRefnoForUpdate(refno)
                ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>参考号错误")

        val errorStatus = setOf(TradeDict.DTL_STATUS_SUCCESS, TradeDict.DTL_STATUS_CANCEL)
        if (transaction.status in errorStatus) {
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>状态错误")
        }

        transaction.status = TradeDict.DTL_STATUS_FAIL
        transactionMainDao.save(transaction)
        return transaction
    }


    override fun success(refno: String): TTransactionMain {
        val transaction = transactionMainDao.findByRefnoForUpdate(refno)
                ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>参考号错误")

        val errorStatus = setOf(TradeDict.DTL_STATUS_SUCCESS, TradeDict.DTL_STATUS_CANCEL)
        if (transaction.status in errorStatus) {
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>状态错误")
        }
        transaction.status = TradeDict.DTL_STATUS_FAIL
        if (transaction.person) {
            // update account balance
            transaction.personDtl?.let {
                accountRepository.recalcAccountBalance(it, it.amount, false)
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                    "个人流水<${transaction.refno}>不存在")
        }
        if (transaction.shop) {
            // update shop balance
            transaction.shopDtl?.let {
                shopaccRepository.recalcShopBalance(it, it.amount, false)
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                    "商户流水<${transaction.refno}>不存在")
        }

        if (transaction.subject) {
            // update subject balance
        }
        transactionMainDao.save(transaction)
        return transaction
    }
}