package com.supwisdom.dlpay.framework.service.impl;

import com.supwisdom.dlpay.framework.core.DatabaseConfig;
import com.supwisdom.dlpay.framework.dao.*;
import com.supwisdom.dlpay.framework.data.SystemDateTime;
import com.supwisdom.dlpay.framework.domain.*;
import com.supwisdom.dlpay.framework.service.SystemUtilService;
import com.supwisdom.dlpay.framework.util.DateUtil;
import com.supwisdom.dlpay.framework.util.NumberUtil;
import com.supwisdom.dlpay.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.Date;

@Service
public class SystemUtilServiceImpl implements SystemUtilService {
  @Autowired
  private DatabaseConfig databaseConfig;

  @Autowired
  private TaskLockDao taskLockDao;
  @Autowired
  private SettleCtlDao settleCtlDao;
  @Autowired
  private TranscodeDao transcodeDao;
  @Autowired
  private SysparaDao sysparaDao;
  @Autowired
  private BusinessparaDao businessparaDao;

  private static final Logger logger = LoggerFactory.getLogger(SystemUtilServiceImpl.class);

  public static class SystemDateTimeImpl implements SystemDateTime {
    private Date now;
    private static final SimpleDateFormat sdft = new SimpleDateFormat("yyyyMMddHHmmss");

    public SystemDateTimeImpl(Date now) {
      this.now = now;
    }

    @Override
    public String getHostdate() {
      return sdft.format(this.now).substring(0, 8);
    }

    @Override
    public String getHosttime() {
      return sdft.format(this.now).substring(8, 14);
    }

    @Override
    public String getHostdatetime() {
      return sdft.format(this.now);
    }

    @Override
    public Date getSysdate() {
      return this.now;
    }
  }

  /**
   * 获取oracle数据库时间
   */
  private SystemDateTime getOracleDatetime() {
    return taskLockDao.getOracleDatetime();
  }

  private SystemDateTime getPGDatetime() {
    return taskLockDao.getPGDatetime();
  }

  @Override
  public SystemDateTime getSysdatetime() {
    switch (databaseConfig.getPlatform()) {
      case "postgresql":
        return getPGDatetime();
      default:
        return getOracleDatetime();
    }
  }

  @Override
  public TTaskLock updateTaskLock(TTaskLock lock) {
    return taskLockDao.save(lock);
  }

  @Override
  public TTaskLock doLockTask(String taskcode, Integer minRecover, String remark) {
    if (null == minRecover) minRecover = 10; //默认10分钟
    String hostdatetime = getSysdatetime().getHostdatetime(); //yyyyMMddHHmmss
    TTaskLock lock = taskLockDao.getTaskLockWithLock(taskcode);
    if (lock != null) {
      if (lock.getTaskstatus() == 1 && DateUtil.compareDatetime(DateUtil.getNewTime(hostdatetime, -60 * minRecover), lock.getTasktime()) < 0) {
        // 被锁，正在执行操作
        return null;
      } else {
        lock.setTaskstatus(1);
        lock.setTasktime(hostdatetime);
        taskLockDao.save(lock);
        return lock;
      }
    } else {
      lock = new TTaskLock();
      lock.setTaskcode(taskcode);
      lock.setTaskstatus(1);
      lock.setTasktime(hostdatetime);
      lock.setRemark(remark);
      taskLockDao.save(lock);
    }
    return lock;
  }

  @Override
  public String getAccdate() {
    String hostdate = getSysdatetime().getHostdate();
    TSettlectl settlectl = settleCtlDao.getOne(1);
    if (null != settlectl && null != settlectl.getSettledate()) {
      if (Integer.valueOf(hostdate) < settlectl.getSettledate().intValue()) {
        return settlectl.getSettledate().toString();
      }
    }
    return hostdate;
  }

  private String getOracleRefno() {
    return taskLockDao.getOracleRefno();
  }

  @Override
  public String getRefno() {
    switch (databaseConfig.getPlatform()) {
      case "postgresql":
        return taskLockDao.getPgRefno();
      default:
        return taskLockDao.getOracleRefno();
    }
  }

  @Override
  public String getTranscodeName(int transocde, String defaultValue) {
    TTranscode tTranscode = transcodeDao.getOne(transocde);
    if (null != tTranscode && !StringUtil.isEmpty(tTranscode.getTransname())) {
      return tTranscode.getTransname();
    }
    return defaultValue;
  }

  @Override
  public String getSysparaValue(int paraid) {
    TSyspara syspara = sysparaDao.findByParaid(paraid);
    if (null != syspara) return syspara.getParaval();
    return null;
  }

  @Override
  public String getSysparaValue(int paraid, String defaultValue) {
    String paraval = getSysparaValue(paraid);
    if (null != paraval) return paraval;
    return defaultValue;
  }

  @Override
  public boolean getSysparaValueAsBoolean(int paraid) {
    TSyspara syspara = sysparaDao.findByParaid(paraid);
    if (null != syspara && "1".equals(syspara.getParaval())) return true;
    return false;
  }

  @Override
  public int getSysparaValueAsInt(int paraid, int defaultValue) {
    TSyspara syspara = sysparaDao.findByParaid(paraid);
    if (null != syspara && NumberUtil.isNumber(syspara.getParaval())) return Integer.parseInt(syspara.getParaval());
    return defaultValue;
  }

  @Override
  public double getSysparaValueAsDouble(int paraid, double defaultValue) {
    TSyspara syspara = sysparaDao.findByParaid(paraid);
    if (null != syspara && NumberUtil.isDecimal(syspara.getParaval())) return Double.parseDouble(syspara.getParaval());
    return defaultValue;
  }

  @Override
  public TSyspara getSyspara(int paraid){
    return sysparaDao.findByParaid(paraid);
  }

  @Override
  public TSyspara getSysparaValueForUpdate(int paraid){
    return sysparaDao.findByParaidWithLock(paraid);
  }

  @Override
  public TSyspara getSysparaValueForUpdateNowait(int paraid){
    return sysparaDao.findByParaidWithLockNowait(paraid);
  }

  @Override
  public String getBusinessValue(String parakey) {
    if (!StringUtil.isEmpty(parakey)) {
      TBusinesspara businesspara = businessparaDao.findByParakey(parakey.trim());
      if (null != businesspara) return businesspara.getParaval();
    }
    return null;
  }

  @Override
  public String getBusinessValue(String parakey, String defaultValue) {
    String paraval = getBusinessValue(parakey);
    if (!StringUtil.isEmpty(paraval)) return paraval;
    return defaultValue;
  }

  @Override
  public TBusinesspara getBusiness(String parakey) {
    if (!StringUtil.isEmpty(parakey)) return businessparaDao.findByParakey(parakey.trim());
    return null;
  }

  @Override
  public TBusinesspara getBusinessValueForUpdate(String parakey){
    if (!StringUtil.isEmpty(parakey)) return businessparaDao.findByParakeyForUpdate(parakey.trim());
    return null;
  }

  @Override
  public TBusinesspara getBusinessValueForUpdateNowait(String parakey) {
    if (!StringUtil.isEmpty(parakey)) return businessparaDao.findByParakeyForUpdateNowait(parakey.trim());
    return null;
  }

  @Override
  public String getSubsystemSignKey(String syscode){
    // fixme: 验证数据无误性签名秘钥
    return "";
  }

}
