package com.supwisdom.dlpay.api

import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.util.APIRequestParam
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import mu.KotlinLogging
import org.aspectj.lang.ProceedingJoinPoint
import org.aspectj.lang.annotation.Around
import org.aspectj.lang.annotation.Aspect
import org.aspectj.lang.annotation.Pointcut
import org.springframework.http.ResponseEntity
import org.springframework.stereotype.Component
import org.springframework.web.bind.annotation.ExceptionHandler
import org.springframework.web.bind.annotation.RestControllerAdvice
import javax.servlet.http.HttpServletRequest


@RestControllerAdvice("com.supwisdom.dlpay.api")
class RestControllerAdvice {
    val logger = KotlinLogging.logger { }

    @ExceptionHandler
    fun handleException(ex: Exception, request: HttpServletRequest): ResponseEntity<Any> {
        logger.error { "Request unhandler exception, url<${request.requestURI}>, ex<${ex.cause}>" }
        return ResponseEntity.ok().body(ResponseBodyBuilder.create()
                .exception(TradeErrorCode.BUSINESS_DEAL_ERROR, ex.cause))
    }
}

@Component
@Aspect
class RestControllerAspect {

    private val logger = KotlinLogging.logger { }

    @Pointcut("@within(org.springframework.web.bind.annotation.RestController)")
    fun restcontroller() {
    }

    @Pointcut("@annotation(org.springframework.web.bind.annotation.PostMapping)")
    fun postmapping() {
    }

    @Around("restcontroller() && postmapping() && args(.., @RequestBody body)")
    @Throws(Throwable::class)
    fun logPostMethods(joinPoint: ProceedingJoinPoint, body: Any): Any {
        return try {
            if (body is APIRequestParam) {
                body.checkParam()
            }
            val result = joinPoint.proceed()
            logger.debug(result.toString())
            result
        } catch (t: Throwable) {
            logger.error { "request error , ex<${t.message}>" }
            throw t
        }
    }
}
