package com.supwisdom.dlpay.api.service.impl

import com.supwisdom.dlpay.api.TransactionBuilder
import com.supwisdom.dlpay.api.dao.*
import com.supwisdom.dlpay.api.domain.*
import com.supwisdom.dlpay.api.repositories.ShopaccRepositoryImpl
import com.supwisdom.dlpay.api.repositories.TAccountRepositoryImpl
import com.supwisdom.dlpay.api.service.TransactionService
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.exception.TransactionProcessException
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.Subject
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import java.sql.SQLException
import java.sql.Timestamp


@Service
class TransactionServiceImpl : TransactionService {

    companion object {
        val PERSON_BALANCE_FLAG = 2
        val SHOP_BALANCE_FLAG = 2
    }

    @Autowired
    lateinit var transactionMainDao: TransactionMainDao

    @Autowired
    lateinit var persondtlDao: PersondtlDao

    @Autowired
    lateinit var accountRepository: TAccountRepositoryImpl

    @Autowired
    lateinit var shopaccRepository: ShopaccRepositoryImpl

    @Autowired
    lateinit var systemUtilService: SystemUtilService

    private fun preCheck(builder: TransactionBuilder) {
        if (builder.outId.isNotEmpty() && builder.outtradeno.isEmpty()) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "第三方账号未指定第三方流水号")
        }

        if (builder.transDate.length != 8 || builder.transTime.length != 6) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "交易发生日期错误")
        }

        if (builder.transCode <= 0) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                    "交易码错误")
        }

        if (builder.hasPerson()) {
            builder.person().also {
                if (!it.hasOpposite()) {
                    throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "个人交易对方账户未设置")
                }
                if (it.payinfo.isEmpty() || it.paytype.isEmpty()) {
                    throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "个人交易信息未设置")
                }
                when (it.tradeFlag()) {
                    TradeDict.TRADE_FLAG_IN -> {
                        throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                "不支持的交易")
                    }
                    TradeDict.TRADE_FLAG_OUT -> {
                        if (builder.reverseFlag != TradeDict.REVERSE_FLAG_NONE && it.amount > 0) {
                            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                    "消费冲正交易，余额不能大于0")
                        } else if (builder.reverseFlag == TradeDict.REVERSE_FLAG_NONE && it.amount < 0) {
                            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                    "消费交易，余额不能小于0")
                        }
                    }
                    else -> {
                        throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                "个人交易方向未设置")
                    }
                }

            }
        }

        if (builder.hasShop()) {
            builder.shop().also {
                if (!it.hasOpposite()) {
                    throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "商户交易对方账户未设置")
                }
                when (it.tradeFlag()) {
                    TradeDict.TRADE_FLAG_IN -> {
                        if (it.amount < 0) {
                            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                    "商户收入交易金额不能小于0")
                        }
                    }
                    TradeDict.TRADE_FLAG_OUT -> {
                        if (it.amount > 0) {
                            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                                    "商户支出交易金额不能小于0")
                        }
                    }
                    else -> throw  TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                            "商户交易方向未设置")
                }
            }
        }
    }

    private fun builderRecords(builder: TransactionBuilder, status: String): TTransactionMain {
        try {
            preCheck(builder)
            // 记录三方的交易流水（个人，商户，科目)
            val transaction = TTransactionMain().apply {
                refno = systemUtilService.refno
                accdate = systemUtilService.accdate
                outTradeNo = builder.outtradeno
                outId = builder.outId
                createTime = Timestamp(systemUtilService.sysdatetime.sysdate.time)
            }

            if (builder.hasPerson()) {
                TPersondtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    userid = builder.person().person.userid
                    accountNo = builder.person().person.accno
                    userName = builder.person().person.accname
                    outtradeno = builder.outtradeno
                    transdate = builder.transDate
                    transtime = builder.transTime
                    befbal = builder.person().person.availbal
                    amount = if (builder.person().tradeFlag() == TradeDict.TRADE_FLAG_IN) {
                        builder.person().amount
                    } else {
                        -builder.person().amount
                    }
                    paytype = builder.person().paytype
                    payinfo = builder.person().payinfo
                    transcode = builder.transCode
                    tradeflag = builder.person().tradeFlag()
                    transdesc = builder.person().description
                    oppositeAccNo = builder.person().opposite.getAccountNo()
                    oppositeAccName = builder.person().opposite.getAccountName()
                    reverseFlag = builder.reverseFlag
                    remark = builder.person().remark
                    this.status = status
                }.also {
                    // save persondtl
                    persondtlDao.save(it)
                    transaction.personDtl = it
                    transaction.person = true
                }
            }
            if (builder.hasShop()) {
                TShopdtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    this.amount = builder.shop().amount
                    this.payInfo = builder.shop().payinfo
                    this.payType = builder.shop().paytype
                    this.transDate = builder.transDate
                    this.transTime = builder.transTime
                    this.tradeCode = builder.transCode
                    amount = if (builder.shop().tradeFlag() == TradeDict.TRADE_FLAG_IN) {
                        builder.shop().amount
                    } else {
                        -builder.shop().amount
                    }
                    this.tradeflag = builder.shop().tradeFlag()
                    this.shopaccno = builder.shop().shopacc.shopaccno
                    this.shopname = builder.shop().shopacc.shopname
                    this.transdesc = builder.shop().description
                    this.oppositeAccNo = builder.shop().opposite.getAccountNo()
                    this.oppositeAccName = builder.shop().opposite.getAccountName()
                    this.remark = builder.shop().remark
                    this.status = status
                }.also {
                    // save shopdtl
                    transaction.shopDtl = it
                    transaction.shop = true
                }
            }
            if (builder.hasSubject()) {
                // save subjectdtl
                TSubjectdtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    this.amount = builder.subject().amount
                    this.payInfo = builder.subject().payinfo
                    this.payType = builder.subject().paytype
                    this.tradeCode = builder.transCode
                    this.transDate = builder.transDate
                    this.transTime = builder.transTime
                    this.oppositeAccNo = builder.subject().opposite.getAccountNo()
                    this.oppositeAccName = builder.subject().opposite.getAccountName()
                    this.status = status
                }.also {
                    transaction.subjectDtl = it
                    transaction.subject = true
                }
            }

            builder.getAllDetails().map { line ->
                TDebitCreditDtl().apply {
                    this.refno = transaction.refno
                    this.accdate = transaction.accdate
                    this.seqno = line.seqno
                    this.draccno = line.debit.getAccountNo()
                    this.drsubjno = line.debit.getSubjectNo()
                    this.craccno = line.credit.getAccountNo()
                    this.crsubjno = line.credit.getSubjectNo()
                    this.amount = line.amount
                    this.summary = line.summary
                }
            }.apply {
                transaction.details = this
            }


            if (builder.hasPerson()) {
                val dc = getDebitOrCredit(builder.person().tradeFlag())
                transaction.sumAmountByAccno(builder.person().person.accno,
                        Subject.SUBJNO_PERSONAL_DEPOSIT,
                        PERSON_BALANCE_FLAG, dc).also {
                    if (transaction.personDtl.amount != it) {
                        throw TransactionProcessException(TradeErrorCode.INPUT_DATA_ERROR,
                                "输入金额错误，个人余额不符<${transaction.personDtl.amount}>")
                    }
                }
            }

            if (builder.hasShop()) {
                val dc = getDebitOrCredit(builder.shop().tradeFlag())
                transaction.sumAmountByAccno(builder.shop().shopacc.shopaccno,
                        Subject.SUBJNO_MACHANT_INCOME,
                        SHOP_BALANCE_FLAG, dc).also {
                    if (transaction.shopDtl.amount != it) {
                        throw TransactionProcessException(TradeErrorCode.INPUT_DATA_ERROR,
                                "输入金额错误，商户余额不符<${transaction.shopDtl.amount}>")
                    }
                }
            }

            if (builder.hasSubject()) {
                transaction.sumAmountByAccno(builder.subject().subject.subjno,
                        builder.subject().subject.subjno,
                        builder.subject().subject.balflag,
                        "both").also {
                    if (transaction.subjectDtl.amount != it) {
                        throw TransactionProcessException(TradeErrorCode.INPUT_DATA_ERROR,
                                "输入金额错误，科目余额不符<${transaction.subjectDtl.amount}>")
                    }
                }
            }

            transaction.status = status
            transactionMainDao.save(transaction)
            return transaction
        } catch (ex: SQLException) {
            throw TransactionProcessException(TradeErrorCode.BUSINESS_DEAL_ERROR, ex.message)
        }
    }

    private fun getDebitOrCredit(tradeFlag: String): String {
        return when (tradeFlag) {
            TradeDict.TRADE_FLAG_IN -> "credit"
            TradeDict.TRADE_FLAG_OUT -> "debit"
            else -> "both"
        }
    }

    override fun init(builder: TransactionBuilder): TTransactionMain {
        return builderRecords(builder, TradeDict.DTL_STATUS_INIT)
    }

    override fun wip(builder: TransactionBuilder): TTransactionMain {
        return builderRecords(builder, TradeDict.DTL_STATUS_WIP)
    }

    private fun updateRecordStatus(transaction: TTransactionMain, status: String) {
        if (transaction.person) {
            transaction.personDtl?.also {
                if (it.status != transaction.status) {
                    throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED,
                            "个人流水<${transaction.refno}>状态错误")
                }
                it.status = status
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                    "个人流水<${transaction.refno}>不存在")
        }

        if (transaction.shop) {
            transaction.shopDtl?.also {
                if (it.status != transaction.status) {
                    throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED,
                            "商户流水<${transaction.refno}>状态错误")
                }
                it.status = status
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                    "商户流水<${transaction.refno}>不存在")
        }

        if (transaction.subject) {
            transaction.subjectDtl?.also {
                if (it.status != transaction.status) {
                    throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED,
                            "科目流水<${transaction.refno}>状态错误")
                }
                it.status = status
            } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                    "科目流水<${transaction.refno}>不存在")
        }
        transaction.status = status
    }

    override fun wip(refno: String): TTransactionMain {
        val transaction = transactionMainDao.findByRefnoForUpdate(refno)
                ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>参考号错误")

        if (transaction.status != TradeDict.DTL_STATUS_INIT
                && transaction.status != TradeDict.DTL_STATUS_WIP) {
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>状态错误")
        }

        updateRecordStatus(transaction, TradeDict.DTL_STATUS_WIP)
        transactionMainDao.save(transaction)
        return transaction
    }

    override fun fail(refno: String): TTransactionMain {
        return fail(refno, "")
    }


    override fun success(refno: String): TTransactionMain {
        return success(refno, "")
    }

    override fun fail(refno: String, remark: String): TTransactionMain {
        val transaction = transactionMainDao.findByRefnoForUpdate(refno)
                ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>参考号错误")

        val errorStatus = setOf(TradeDict.DTL_STATUS_SUCCESS, TradeDict.DTL_STATUS_CANCEL)
        if (transaction.status in errorStatus) {
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>状态错误")
        }

        updateRecordStatus(transaction, TradeDict.DTL_STATUS_FAIL)
        if (transaction.person) {
            transaction.personDtl.remark = remark
        }
        if (transaction.shop) {
            transaction.shopDtl.remark = remark
        }
        transaction.endTime = Timestamp(systemUtilService.sysdatetime.sysdate.time)
        transactionMainDao.save(transaction)
        return transaction
    }

    override fun success(refno: String, remark: String): TTransactionMain {
        val transaction = transactionMainDao.findByRefnoForUpdate(refno)
                ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>参考号错误")

        val errorStatus = setOf(TradeDict.DTL_STATUS_SUCCESS, TradeDict.DTL_STATUS_CANCEL)
        if (transaction.status in errorStatus) {
            throw TransactionProcessException(TradeErrorCode.TRANSACTION_IS_FINISHED, "流水<$refno>状态错误")
        }
        transaction.status = TradeDict.DTL_STATUS_SUCCESS
        transaction.accdate = systemUtilService.accdate
        if (transaction.person) {
            // update account balance
            val amount = transaction.sumAmountByAccno(
                    transaction.personDtl.accountNo, Subject.SUBJNO_PERSONAL_DEPOSIT,
                    PERSON_BALANCE_FLAG, "both")
            if (amount.compareTo(0.0) != 0) {
                transaction.personDtl?.let {
                    if (accountRepository.recalcAccountBalance(it, amount, false) != 1) {
                        throw TransactionProcessException(TradeErrorCode.ACCOUNT_TRADE_BUSY,
                                "个人账户交易冲突")
                    }
                    transaction.personDtl.accdate = transaction.accdate
                } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                        "个人流水<${transaction.refno}>不存在")
            }
            transaction.personDtl.status = TradeDict.DTL_STATUS_SUCCESS
            transaction.personDtl.remark = remark
        }
        if (transaction.shop) {
            // update shop balance
            val amount = transaction.sumAmountByAccno(
                    transaction.shopDtl.shopaccno, Subject.SUBJNO_MACHANT_INCOME,
                    SHOP_BALANCE_FLAG, "both")
            if (amount.compareTo(0.0) != 0) {
                transaction.shopDtl?.let {
                    if (shopaccRepository.recalcShopBalance(it, it.amount, false) != 1) {
                        throw TransactionProcessException(TradeErrorCode.ACCOUNT_TRADE_BUSY,
                                "商户账户交易冲突")
                    }
                } ?: throw TransactionProcessException(TradeErrorCode.TRANSACTION_NOT_EXISTS,
                        "商户流水<${transaction.refno}>不存在")
            }
            transaction.shopDtl.status = TradeDict.DTL_STATUS_SUCCESS
            transaction.shopDtl.accdate = transaction.accdate
            transaction.shopDtl.remark = remark
        }

        if (transaction.subject) {
            // update subject balance
        }

        transaction.endTime = Timestamp(systemUtilService.sysdatetime.sysdate.time)
        transactionMainDao.save(transaction)
        return transaction
    }
}