package com.supwisdom.dlpay.controller

import com.supwisdom.dlpay.MvcBaseTest
import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.util.HmacUtil
import io.restassured.RestAssured
import io.restassured.RestAssured.*
import io.restassured.http.ContentType
import io.restassured.path.json.JsonPath.from
import org.hamcrest.MatcherAssert.assertThat
import org.hamcrest.Matchers.*
import org.junit.Before
import org.junit.Test
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.test.context.SpringBootTest
import org.springframework.boot.web.server.LocalServerPort
import org.springframework.test.context.ActiveProfiles
import java.text.SimpleDateFormat
import java.util.*

@ActiveProfiles("devel-pg-local")
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
class SecurityControllerTest : MvcBaseTest() {
    val appid = "100001"
    val appsecret = "oUw2NmA09ficiVWD4TUQLDOkPyzQa3VzbjjsW0B2qTk="

    @LocalServerPort
    private var port: Int = 0

    @Autowired
    lateinit var jwtConfig: JwtConfig

    @Before
    fun setUp() {
        RestAssured.port = port
    }

    fun getJwt(id: String, secret: String): String {
        val token = given().param("appid", id)
                .`when`()
                .get("/api/auth/gettoken")
                .then()
                .contentType(ContentType.JSON)
                .statusCode(200)
                .body("token", notNullValue())
                .extract().path<String>("token")

        val tokenCrypt = HmacUtil.HMACSHA256(token, secret)

        return given().param("appid", id)
                .param("secret", tokenCrypt)
                .`when`()
                .get("/api/auth/authentication")
                .then()
                .statusCode(200)
                .contentType(ContentType.JSON)
                .body("jwt", notNullValue())
                .extract().response().let {
                    val exp = it.path<String>("expiredAt").run {
                        SimpleDateFormat("yyyy-MM-dd HH:mm:ss z").parse(this)
                    }
                    val now = Calendar.getInstance()
                    assertThat(now.time, lessThanOrEqualTo(exp))
                    it.path<String>("jwt")
                }
    }

    @Test
    fun testGetJwt() {
        getJwt(appid, appsecret)
    }

    @Test
    fun testGetJwtClient() {
        val clientid = "000030450"
        val response = given().param("appid", appid)
                .`when`()
                .get("/api/auth/gettoken/$clientid")
                .then()
                .statusCode(200)
                .body("token", notNullValue())
                .extract()

        val token = from(response.body().asString()).getString("token")
        val secret = HmacUtil.HMACSHA256(token, appsecret)

        given().param("appid", appid)
                .param("secret", secret)
                .`when`()
                .get("/api/auth/authentication/$clientid")
                .then()
                .statusCode(200)
                .body("jwt", notNullValue())
    }

    @Test
    fun testJwtRefresh() {
        getJwt(appid, appsecret).also { jwt ->
            given().header(jwtConfig.header, "${jwtConfig.tokenHeader}$jwt")
                    .`when`()
                    .get("/api/auth/refresh")
                    .then()
                    .statusCode(200)
                    .body("jwt", notNullValue())
        }
    }

    @Test
    fun testAuthencationFail() {
        given().param("appid", appid)
                .`when`()
                .get("/api/auth/gettoken")
                .then()
                .body("token", notNullValue())
                .extract()
                .path<String>("token").also { token ->
                    given().param("appid", appid)
                            .param("secret", token)
                            .`when`()
                            .get("/api/auth/authentication")
                            .then()
                            .statusCode(401)
                }
    }

}