package com.supwisdom.dlpay.consume

import com.supwisdom.dlpay.consume.domain.TAccount
import com.supwisdom.dlpay.consume.domain.TPerson
import com.supwisdom.dlpay.consume.domain.TUserdtl
import com.supwisdom.dlpay.consume.service.AccountUtilServcie
import com.supwisdom.dlpay.consume.service.PersonBalancePayService
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.framework.domain.TShopacc
import com.supwisdom.dlpay.framework.domain.TSubject
import com.supwisdom.dlpay.framework.domain.TTranstype
import com.supwisdom.dlpay.framework.util.*

class AccountHolder<T> private constructor(val accountId: String, val idType: Int) {
    companion object {
        const val IDTYPE_PERSON = 1
        const val IDTYPE_SHOP = 2
        const val IDTYPE_SUBJECT = 3
        const val IDTYPE_TRANSTYPE = 4

        fun person(userid: String): AccountHolder<TAccount> {
            return AccountHolder<TAccount>(userid, IDTYPE_PERSON)
        }

        fun shop(shopid: String): AccountHolder<TShopacc> {
            return AccountHolder<TShopacc>(shopid, IDTYPE_SHOP)
        }

        fun subject(subjNo: String): AccountHolder<TSubject> {
            return AccountHolder(subjNo, IDTYPE_SUBJECT)
        }

        fun transType(transType: Int): AccountHolder<TTranstype> {
            return AccountHolder("$transType", IDTYPE_TRANSTYPE)
        }
    }

    internal lateinit var builder: PersonTransBuilder

    @Suppress("UNCHECKED_CAST", "IMPLICIT_CAST_TO_ANY")
    fun <T> get(): T {
        return when (idType) {
            IDTYPE_PERSON -> builder.accountUtil.readAccount(accountId)
            IDTYPE_SHOP -> builder.accountUtil.readShopAcc(accountId.toInt())
            IDTYPE_SUBJECT -> builder.accountUtil.readSubject(accountId)
            IDTYPE_TRANSTYPE -> builder.accountUtil.readTranstype(accountId.toInt())
            else -> throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "账户类型未知type<$idType>")
        } as T
    }

    fun withLock(nowait: Boolean): T {
        TODO("not implement")
    }
}

class PersonTransBuilder private constructor(accUitl: AccountUtilServcie) {
    companion object {
        fun newBuilder(accUitl: AccountUtilServcie) = PersonTransBuilder(accUitl)
    }

    inner class TransDetail(val debitAccNo: String, val debitSubjNo: String,
                            val creditAccNo: String, val creditSubjNo: String,
                            val amount: Double, val summary: String,
                            val rowno: Int)

    // 输入参数，调用接口时指定的参数值
    lateinit var person: TPerson
    lateinit var tradetype:Tradetype

    var transcode = 0

    var transDate = ""
    var transTime = ""
    /**
     * 是否允许透支
     */
    var overdraft = false
    var description = ""

    // 内部参数，不需要调用者处理
    val details = mutableListOf<TransDetail>()
    var amount: Double = 0.0
    var accountUtil = accUitl


    /**
     * 支付方式
     * */
    var paytype = ""  //枚举？
    var payinfo = ""

    /**
     * 外部流水号
     * */
    var outtradeno = "" //发起支付系统的流水号（对接系统根据此流水号查询本地流水对账）


    fun setOwner(per: TPerson): PersonTransBuilder {
        this.person = per
        return this
    }

    fun enableOverdraft(b: Boolean = false): PersonTransBuilder {
        this.overdraft = b
        return this
    }

    fun setTransDatetime(date: String, time: String): PersonTransBuilder {
        this.transDate = date
        this.transTime = time
        return this
    }

    fun setTransinfo(transcode: Int, description: String): PersonTransBuilder {
        this.transcode = transcode
        this.description = description
        return this
    }

    fun selectPaytype(paytype: String, payinfo: String): PersonTransBuilder {
        this.paytype = paytype
        this.payinfo = payinfo
        return this
    }

    fun setOuttradeno(outtradeno: String): PersonTransBuilder {
        this.outtradeno = outtradeno
        return this
    }

    fun chooseTradetype(tradetype: Tradetype): PersonTransBuilder {
        this.tradetype = tradetype
        return this
    }

    fun setTradeno(outtradeno: String): PersonTransBuilder {
        this.outtradeno = outtradeno
        return this
    }

    private fun prepareData() {
        if (null == this.tradetype) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未指定交易类型")
        }
        if(transcode==0){
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未指定交易码")
        }

        when(this.tradetype){
            //充值必须指明用户和支付方式
            Tradetype.RECHARGE -> {
                if (null == this.person)
                    throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未指定充值账户")

                if (StringUtil.isEmpty(this.paytype))
                    throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未指定充值方式")
            }

            //消费
            Tradetype.CONSUME ->{
                if (StringUtil.isEmpty(this.paytype))
                    this.paytype = TradeDict.PAYTYPE_BALANCE   //默认余额支付
            }
        }

        amount = 0.0
        if (this.details.size < 1) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未指定交易明细")
        }
        if (null != this.person) {
            var buyer = accountUtil.readAccount(person.userid) //判断是一个人的流水
            for (detail in details) {
                if (detail.debitSubjNo.equals(buyer.subjno) && !detail.debitAccNo.equals(buyer.accno))
                    throw throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "交易明细用户错误")

                if (detail.creditSubjNo.equals(buyer.subjno) && !detail.creditAccNo.equals(buyer.accno))
                    throw throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "交易明细用户错误")
            }
        }
        amount = this.details.sumByDouble { it.amount }

        if (!StringUtil.isEmpty(this.transDate) && !DateUtil.checkDatetimeValid(this.transDate, "yyyyMMdd")) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "交易日期格式错误[yyyyMMdd]")
        }
        if (!StringUtil.isEmpty(this.transTime) && !DateUtil.checkDatetimeValid(this.transTime, "HHmmss")) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "交易时间格式错误[HHmmss]")
        }

        if(StringUtil.isEmpty(this.outtradeno)){
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未传递外部流水号")
        }
    }

    private fun preCheckAccount() {
        if (null != this.person) {
            when (person.status) {
                TradeDict.STATUS_CLOSED -> throw TransactionCheckException(TradeErrorCode.PERSON_STATUS_ERROR, "用户已注销")
                TradeDict.STATUS_LOCKED -> throw TransactionCheckException(TradeErrorCode.PERSON_STATUS_ERROR, "用户已冻结锁定")
            }
        }
    }

    fun <T, U> addDetail(debit: AccountHolder<T>, credit: AccountHolder<U>,
                         amount: Double, summary: String): PersonTransBuilder {
        debit.builder = this
        credit.builder = this

        var debitAccNo = ""
        var debitSubjNo = ""
        when (debit.idType) {
            AccountHolder.IDTYPE_PERSON -> debit.get<TAccount>().also {
                debitAccNo = it.accno
                debitSubjNo = it.subjno
            }
            AccountHolder.IDTYPE_SHOP -> debit.get<TShopacc>().also {
                debitAccNo = it.shopaccno
                debitSubjNo = it.subjno
            }
            AccountHolder.IDTYPE_SUBJECT -> debit.get<TSubject>().also {
                debitAccNo = it.subjno
                debitSubjNo = it.subjno
            }
            AccountHolder.IDTYPE_TRANSTYPE -> TODO("not implements")
        }

        var creditAccNo = ""
        var creditSubjNo = ""
        when (credit.idType) {
            AccountHolder.IDTYPE_PERSON -> credit.get<TAccount>().also {
                creditAccNo = it.accno
                creditSubjNo = it.subjno
            }
            AccountHolder.IDTYPE_SHOP -> credit.get<TShopacc>().also {
                creditAccNo = it.shopaccno
                creditSubjNo = it.subjno
            }
            AccountHolder.IDTYPE_SUBJECT -> credit.get<TSubject>().also {
                creditAccNo = it.subjno
                creditSubjNo = it.subjno
            }
            AccountHolder.IDTYPE_TRANSTYPE -> TODO("not implements")
        }

        this.details.add(TransDetail(debitAccNo, debitSubjNo, creditAccNo, creditSubjNo,
                amount, summary, this.details.size + 1))
        return this
    }

    /**
     * 单步交易，一次完成交易记账过程
     * @param service
     * @param isFinished - true : 完成交易过程,扣除账户余额; false - 完成检查
     */
    fun done(service: PersonBalancePayService, isFinished: Boolean): TUserdtl {
        prepareData()
        preCheckAccount()
        return when (isFinished) {
            true -> service.process(this)
            false -> service.init(this)
        }
    }

    /**
     * 两步交易，在 paydtl 基础完成确认过程
     * @param paydtl
     * @param status - 完成交易状态，见 TradeDict.DTL_STATUS_FAIL
     */
    fun done(paydtl: TUserdtl, status: String, service: PersonBalancePayService): TUserdtl {
        return service.finish(paydtl, status, null)
    }

    fun done(refno: String, status: String, service: PersonBalancePayService): TUserdtl {
        return service.finish(refno, status, null)
    }
}