package com.supwisdom.dlpay

import io.lettuce.core.ReadFrom
import org.springframework.beans.factory.annotation.Value
import org.springframework.boot.autoconfigure.SpringBootApplication
import org.springframework.boot.runApplication
import org.springframework.boot.web.servlet.ServletComponentScan
import org.springframework.cache.annotation.EnableCaching
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.data.redis.connection.RedisConnectionFactory
import org.springframework.data.redis.connection.RedisPassword
import org.springframework.data.redis.connection.RedisStandaloneConfiguration
import org.springframework.data.redis.connection.lettuce.LettuceClientConfiguration
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory
import org.springframework.data.redis.core.RedisTemplate
import org.springframework.data.redis.repository.configuration.EnableRedisRepositories
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer
import org.springframework.data.redis.serializer.StringRedisSerializer
import org.springframework.scheduling.annotation.EnableScheduling


@Configuration
@EnableRedisRepositories
class AppConfig {

    @Value("\${spring.redis.host}")
    lateinit var server: String

    @Value("\${spring.redis.port}")
    var port: Int = 0

    @Value("\${spring.redis.database}")
    var database: Int = 0

    @Value("\${spring.redis.password}")
    lateinit var password: String

    @Bean
    fun redisConnectionFactory(): RedisConnectionFactory {
        val clientConfig = LettuceClientConfiguration.builder()
                .readFrom(ReadFrom.SLAVE_PREFERRED)
                .build()
        val serverConfig = RedisStandaloneConfiguration(server, port)
        if (password.isNotEmpty()) {
            serverConfig.password = RedisPassword.of(password)
        }
        serverConfig.database = database

        return LettuceConnectionFactory(serverConfig, clientConfig)
    }
}

@Configuration
class HttpSessionConfig {
    @Bean
    fun sessionRedisTemplate(
            connectionFactory: RedisConnectionFactory): RedisTemplate<Any, Any> {
        val template = RedisTemplate<Any, Any>()
        template.keySerializer = StringRedisSerializer()
        template.hashKeySerializer = StringRedisSerializer()

        template.setDefaultSerializer(GenericJackson2JsonRedisSerializer())
        template.setConnectionFactory(connectionFactory)
        return template
    }
}


@SpringBootApplication
@EnableScheduling
@EnableCaching
@ServletComponentScan
class DoorApplication

fun main(args: Array<String>) {
    runApplication<DoorApplication>(*args)
}
