package com.supwisdom.dlpay.framework

import com.google.gson.Gson
import com.supwisdom.dlpay.exception.RequestParamCheckException
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import mu.KotlinLogging


class ResponseBodyBuilder private constructor() {
    companion object {
        private const val INVALIDE_RETCODE = -0x7FFFFFFF
        private val RESERVED_KEY = setOf("retcode", "retmsg")
        private val LOGGER = KotlinLogging.logger {}
        fun create() = ResponseBodyBuilder()
    }

    private var retCode = INVALIDE_RETCODE
    private var retMsg = ""

    private val respData = mutableMapOf<String, Any>()

    fun result(code: Int, msg: String? = null): ResponseBodyBuilder {
        this.retCode = code
        retMsg = if (retCode == 0) {
            msg ?: "成功"
        } else {
            msg ?: "失败"
        }
        return this
    }

    fun success(msg: String? = null): Map<String, Any> {
        result(0, msg)
        return build()
    }

    fun fail(code: Int, msg: String): Map<String, Any> {
        if (code == 0) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "错误码未正确定义")
        }
        result(code, msg)
        return build()
    }

    fun exception(code: Int, exception: Exception, msg: String? = null): Map<String, Any> {
        data("exception", exception.message ?: "$exception")
        result(code, msg)
        return build()
    }

    fun exception(code: Int, cause: Throwable?, msg: String? = null): Map<String, Any> {
        data("exception", cause?.message ?: "unkonw excpetion")
        result(code, msg)
        return build()
    }

    fun transException(exception: TransactionException, msg: String): Map<String, Any> {
        data("exception", exception.message!!)
        result(exception.code(), "$msg - [${exception.message}]")
        return build()
    }

    fun requestException(exception: RequestParamCheckException, msg: String): Map<String, Any> {
        data("exception", exception.message!!)
        result(exception.errCode, "$msg - [${exception.message}]")
        return build()
    }

    fun data(name: String, value: Any): ResponseBodyBuilder {
        if (name in RESERVED_KEY) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "返回值 <$name> 为保留值，不能使用")
        }
        this.respData[name] = value
        return this
    }

    private fun build(): Map<String, Any> {
        if (retCode == INVALIDE_RETCODE) {
            throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR, "未设置返回码!")
        } else if (retCode != 0) {
            LOGGER.error("【 ==== ERROR ==== 】: " + Gson().toJson(this.respData))
        } else {
            LOGGER.info("retcode=[0],retmsg=[$retMsg] return success!!! \n" + Gson().toJson(this.respData))
        }
        return this.respData.plus(mapOf("retcode" to retCode, "retmsg" to retMsg))
    }
}