package com.visionsmarts;

import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;

public class VSReaderQR {
    public static final int MODE_ALPHANUMERIC = 2;
    public static final int MODE_BYTE = 4;
    public static final int MODE_KANJI = 8;
    public static final int MODE_NUMERIC = 1;
    public static final String QR_CODE_TYPE_ID = "QR";

    public static class DecoderValues {
        public Point corner1;
        public Point corner2;
        public Point corner3;
        public Point corner4;
        public int evaluationDays;
        public int mode;

        public DecoderValues() {
            this.corner1 = new Point();
            this.corner2 = new Point();
            this.corner3 = new Point();
            this.corner4 = new Point();
        }
    }

    private static class ECIMarker {
        public int eci;
        public int position;

        private ECIMarker() {
        }
    }

    public static class Point {
        public int x;
        public int y;
    }

    public static native byte[] decodeImage(byte[] bArr, int i, int i2, DecoderValues decoderValues);

    static {
        System.loadLibrary("VSReaderQR");
    }

    public static String format(byte[] data, int mode) {
        Charset charset = Charset.forName("UTF-8");
        if ((mode & MODE_KANJI) == MODE_KANJI) {
            charset = Charset.forName("Shift_JIS");
        }
        int position = 0;
        if (data.length > 3 && data[0] == 93 && data[MODE_NUMERIC] == 81) {
            if (data[MODE_ALPHANUMERIC] == 51 || data[MODE_ALPHANUMERIC] == 52) {
                position = 0 + 3;
            }
            if ((data[MODE_ALPHANUMERIC] == 53 || data[MODE_ALPHANUMERIC] == 54) && data.length > 5) {
                position += 5;
            }
        }
        StringBuilder result = new StringBuilder(data.length);
        do {
            ECIMarker nextMarker = findFirstECIMarker(data, position);
            String next = decodeData(ByteBuffer.wrap(data, position, Math.min(nextMarker.position, data.length) - position), charset);
            if (next != null) {
                result.append(next);
            }
            if (nextMarker.position != data.length) {
                position = nextMarker.position + 7;
                charset = getCharsetForECI(nextMarker.eci);
                if (charset == null) {
                    charset = getCharsetForECI(0);
                }
            }
            if (nextMarker.position == data.length) {
                break;
            }
        } while (position < data.length);
        return result.toString();
    }

    private static ECIMarker findFirstECIMarker(byte[] data, int fromPosition) {
        ECIMarker eciMarker = new ECIMarker();
        eciMarker.position = data.length;
        int position = fromPosition;
        while (position < data.length - 6) {
            if (data[position] == 92 && Character.isDigit(data[position + MODE_NUMERIC]) && Character.isDigit(data[position + MODE_ALPHANUMERIC]) && Character.isDigit(data[position + 3]) && Character.isDigit(data[position + MODE_BYTE]) && Character.isDigit(data[position + 5]) && Character.isDigit(data[position + 6])) {
                eciMarker.position = position;
                eciMarker.eci = ((((((data[position + MODE_NUMERIC] - 48) * 100000) + ((data[position + MODE_ALPHANUMERIC] - 48) * 10000)) + ((data[position + 3] - 48) * 1000)) + ((data[position + MODE_BYTE] - 48) * 100)) + ((data[position + 5] - 48) * 10)) + (data[position + 6] - 48);
                break;
            }
            position += MODE_NUMERIC;
        }
        return eciMarker;
    }

    private static Charset getCharsetForECI(int eci) {
        switch (eci) {

            case MODE_NUMERIC /*1*/:
                return Charset.forName("ISO-8859-1");
            case MODE_ALPHANUMERIC /*2*/:
                return Charset.forName("ISO-8859-1");
            case MODE_BYTE /*4*/:
                return Charset.forName("ISO-8859-2");
            case 5:
                return Charset.forName("ISO-8859-3");
            case 6:
                return Charset.forName("ISO-8859-4");
            case 7:
                return Charset.forName("ISO-8859-5");
            case MODE_KANJI /*8*/:
                return Charset.forName("ISO-8859-6");
            case 9:
                return Charset.forName("ISO-8859-7");
            case 10:
                return Charset.forName("ISO-8859-8");
            case 11:
                return Charset.forName("ISO-8859-9");
            case 12:
                return Charset.forName("ISO-8859-10");
            case 13:
                return Charset.forName("ISO-8859-11");
            case 15:
                return Charset.forName("ISO-8859-13");
            case VSBarcodeReader.BARCODE_TYPE_CODE39 /*16*/:
                return Charset.forName("ISO-8859-14");
            case 17:
                return Charset.forName("ISO-8859-15");
            case 18:
                return Charset.forName("ISO-8859-16");
            case 20:
                return Charset.forName("Shift_JIS");
            case 21:
                return Charset.forName("windows-1250");
            case 22:
                return Charset.forName("windows-1251");
            case 23:
                return Charset.forName("windows-1252");
            case 24:
                return Charset.forName("windows-1256");
            case 25:
                return Charset.forName("UTF-16BE");
            case 26:
                return Charset.forName("UTF-8");
            case 27:
                return Charset.forName("EUC-JP");
            case 28:
                return Charset.forName("Big5");
            case 29:
                return Charset.forName("x-EUC-CN");
            case 30:
                return Charset.forName("EUC-KR");
            case 899:
                return Charset.forName("UTF-8");
            default:
                return null;
        }
    }

    public static String decodeData(ByteBuffer data, Charset charset) {
        try {
            return charset.newDecoder().decode(data).toString();
        } catch (CharacterCodingException e) {
            data.rewind();
            try {
                return Charset.forName("Shift_JIS").newDecoder().decode(data).toString();
            } catch (CharacterCodingException e2) {
                data.rewind();
                try {
                    return Charset.forName("UTF-8").newDecoder().decode(data).toString();
                } catch (CharacterCodingException e3) {
                    data.rewind();
                    try {
                        return Charset.forName("ISO-8859-1").newDecoder().decode(data).toString();
                    } catch (CharacterCodingException e4) {
                        data.rewind();
                        try {
                            return Charset.forName("US-ASCII").newDecoder().decode(data).toString();
                        } catch (CharacterCodingException e5) {
                            return null;
                        }
                    }
                }
            }
        }
    }
}
