package com.visionsmarts.pic2shop.view;

import android.app.Activity;
import android.content.Context;
import android.graphics.ImageFormat;
import android.graphics.Point;
import android.hardware.Camera;
import android.hardware.Camera.AutoFocusCallback;
import android.hardware.Camera.Parameters;
import android.hardware.Camera.PreviewCallback;
import android.hardware.Camera.Size;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.AttributeSet;
import android.util.Log;
import android.view.Display;
import android.view.SurfaceHolder;
import android.view.SurfaceHolder.Callback;
import android.view.SurfaceView;
import android.view.WindowManager;

import com.google.zxing.client.android.camera.CameraConfigurationUtils;
import com.visionsmarts.VSBarcodeReader;
import com.visionsmarts.VSBarcodeReader.DecoderValues;
import com.visionsmarts.VSReaderQR;

import java.io.IOException;
import java.util.regex.Pattern;

public class ScannerLiveView extends SurfaceView implements Callback {
    private Point screen;
    private static final int BARCODE_TYPES_MASK = 7;
    private static final Pattern COMMA_PATTERN;
    private static final int DESIRED_ZOOM_X10 = 27;
    private static final String TAG;
    private AutoFocusCallback mAutoFocusCallback;
    private volatile byte[] mCallbackBuffer;
    private volatile Camera mCamera;
    private DecodingThread mDecodingThread;
    volatile boolean mFoundBarcodeOrQRCode;
    volatile byte[] mFrameData;
    volatile boolean mFrameDataIsAutoFocusInProgress;
    volatile boolean mHasCameraAutoFocus;
    volatile boolean mIsAutoFocusInProgress;
    volatile boolean mIsPreviewStarted;
    Listener mListener;
    private PreviewCallback mPreviewCallback;
    private int mPreviewFrameSize;
    volatile Size mPreviewSize;
    final ScannerHandler mScannerHandler;
    private volatile int mCameraDisplayOrientation;

    public interface Listener {
        void onCodeScanned(String str, String str2);
        void onCameraIssue();
    }

    /* renamed from: com.visionsmarts.pic2shop.view.ScannerLiveView.1 */
    class ScanAutoFocusCallback implements AutoFocusCallback {
        ScanAutoFocusCallback() {
        }

        public void onAutoFocus(boolean success, Camera camera) {
            ScannerLiveView.this.mIsAutoFocusInProgress = false;
        }
    }

    /* renamed from: com.visionsmarts.pic2shop.view.ScannerLiveView.2 */
    class ScanPreviewCallback implements PreviewCallback {
        ScanPreviewCallback() {
        }

        public synchronized void onPreviewFrame(byte[] frameData, Camera camera) {
            if (!(!ScannerLiveView.this.mIsPreviewStarted || ScannerLiveView.this.mDecodingThread == null || ScannerLiveView.this.mFoundBarcodeOrQRCode)) {
                boolean shouldSendDecodeRequest = ScannerLiveView.this.mFrameData == null;
                ScannerLiveView.this.mFrameData = frameData;
                ScannerLiveView.this.mFrameDataIsAutoFocusInProgress = ScannerLiveView.this.mIsAutoFocusInProgress;
                if (shouldSendDecodeRequest) {
                    ScannerLiveView.this.mDecodingThread.getHandler().obtainMessage(BarQrCodeUtil.DECODE).sendToTarget();
                }
            }
        }
    }

    private class DecodingThread extends Thread {
        DecoderValues mBarcodeDecoderValues;
        private Handler mHandler;
        private int mNumberConsecutiveFailedDecodingSinceFocusEnded;
        private byte[] mPreviewLine;
        VSReaderQR.DecoderValues mQRDecoderValues;

        /* renamed from: com.visionsmarts.pic2shop.view.ScannerLiveView.DecodingThread.1 */
        class DecodeMessageHandler extends Handler {
            DecodeMessageHandler() {
            }

            public void handleMessage(Message message) {
                switch (message.what) {
                    case BarQrCodeUtil.DECODE:
                        if (ScannerLiveView.this.mIsPreviewStarted && !ScannerLiveView.this.mFoundBarcodeOrQRCode) {
                            DecodingThread.this.decode();
                        }
                        break;
                    case BarQrCodeUtil.QUIT:
                        Looper.myLooper().quit();
                        break;
                    default:
                }
            }
        }

        public DecodingThread() {
            this.mNumberConsecutiveFailedDecodingSinceFocusEnded = 0;
            this.mBarcodeDecoderValues = new DecoderValues();
            this.mQRDecoderValues = new VSReaderQR.DecoderValues();
            setPriority(10);
            initialize();
        }

        synchronized Handler getHandler() {
            if (this.mHandler == null) {
                try {
                    ScannerLiveView.this.mDecodingThread.wait(1000);
                } catch (InterruptedException e) {
                }
            }
            return this.mHandler;
        }

        public void initialize() {
            if (ScannerLiveView.this.mPreviewSize != null && (this.mPreviewLine == null || this.mPreviewLine.length != ScannerLiveView.this.mPreviewSize.height)) {
                this.mPreviewLine = new byte[ScannerLiveView.this.mPreviewSize.height];
            }
            this.mNumberConsecutiveFailedDecodingSinceFocusEnded = 0;
            ScannerLiveView.this.mFoundBarcodeOrQRCode = false;
            VSBarcodeReader.reset();
        }

        public void run() {
            Looper.prepare();
            synchronized (this) {
                this.mHandler = new DecodeMessageHandler();
                notify();
            }
            Looper.loop();
        }

        private void decode() {
            byte[] previewLine = this.mPreviewLine;
            int width = ScannerLiveView.this.mPreviewSize.width;
            int height = ScannerLiveView.this.mPreviewSize.height;
            synchronized (ScannerLiveView.this.mPreviewCallback) {
                if (ScannerLiveView.this.mFrameData == null) {
                    return;
                }
                byte[] frameData = ScannerLiveView.this.mFrameData;
                boolean wasAutoFocusInProgress = ScannerLiveView.this.mFrameDataIsAutoFocusInProgress;
                int offset  =  Math.round((float) (screen.y  / 2));
                Log.e(TAG,"offset="+offset);
                try {
                    for (int i = previewLine.length - 1; i >= 0; i--) {
                        previewLine[i] = frameData[offset];
                        offset += width;
                    }
                } catch (ArrayIndexOutOfBoundsException e) {
                }
                if (ScannerLiveView.this.mFrameData != null) {
                    String barcode = VSBarcodeReader.decodeNextImage(previewLine, ScannerLiveView.this.mHasCameraAutoFocus ? 1 : 0, ScannerLiveView.BARCODE_TYPES_MASK, this.mBarcodeDecoderValues);
                    if (ScannerLiveView.this.mFrameData != null) {
                        if (barcode.length() > 0) {
                            ScannerLiveView.this.mFoundBarcodeOrQRCode = true;
                            ScannerLiveView.this.mScannerHandler.obtainMessage(BarQrCodeUtil.BARCODE, barcode).sendToTarget();
                        } else {
                            try {
                                byte[] qrData = VSReaderQR.decodeImage(frameData, width, height, this.mQRDecoderValues);
                                if (qrData.length > 0) {
                                    ScannerLiveView.this.mFoundBarcodeOrQRCode = true;
                                    ScannerLiveView.this.mScannerHandler.obtainMessage(BarQrCodeUtil.QRCODE, qrData).sendToTarget();
                                } else if (wasAutoFocusInProgress) {
                                    this.mNumberConsecutiveFailedDecodingSinceFocusEnded = 0;
                                } else {
                                    this.mNumberConsecutiveFailedDecodingSinceFocusEnded++;
                                    if (ScannerLiveView.this.mHasCameraAutoFocus && !ScannerLiveView.this.mIsAutoFocusInProgress && this.mNumberConsecutiveFailedDecodingSinceFocusEnded >= 4) {
                                        ScannerLiveView.this.mScannerHandler.obtainMessage(BarQrCodeUtil.AUTO_FOCUS).sendToTarget();
                                    }
                                }
                            }catch (Exception e){
                                e.printStackTrace();
                            }
                        }
                        synchronized (ScannerLiveView.this.mPreviewCallback) {
                            ScannerLiveView.this.mFrameData = null;
                            if (!(ScannerLiveView.this.mFoundBarcodeOrQRCode || ScannerLiveView.this.mCallbackBuffer == null)) {
                                ScannerLiveView.this.mCamera.addCallbackBuffer(ScannerLiveView.this.mCallbackBuffer);
                            }
                        }
                    }
                }
            }
        }
    }

    private class ScannerHandler extends Handler {
        private ScannerHandler() {
        }

        public void handleMessage(Message message) {
            switch (message.what) {
                case BarQrCodeUtil.AUTO_FOCUS:
                    ScannerLiveView.this.requestAutoFocus();
                    break;
                case BarQrCodeUtil.BARCODE:
                    if (ScannerLiveView.this.mIsPreviewStarted) {
                        ScannerLiveView.this.stopCamera();
                        ScannerLiveView.this.mListener.onCodeScanned("BAR", (String)message.obj);
                    }
                    break;
                case BarQrCodeUtil.QRCODE:
                    if (ScannerLiveView.this.mIsPreviewStarted) {
                        ScannerLiveView.this.stopCamera();
                        ScannerLiveView.this.mListener.onCodeScanned("QR", BarQrCodeUtil.qrToString((byte[])message.obj));
                    }
                    break;
                default:
            }
        }
    }

    public ScannerLiveView(Context context,Listener mListener) {
        super(context);
        this.mPreviewFrameSize = 0;
        this.mScannerHandler = new ScannerHandler();
        this.mIsPreviewStarted = false;
        this.mIsAutoFocusInProgress = false;
        this.mFrameData = null;
        this.mFrameDataIsAutoFocusInProgress = false;
        this.mFoundBarcodeOrQRCode = false;
        this.mListener = mListener;
        initialize(context);
    }

    public ScannerLiveView(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.mPreviewFrameSize = 0;
        this.mScannerHandler = new ScannerHandler();
        this.mIsPreviewStarted = false;
        this.mIsAutoFocusInProgress = false;
        this.mFrameData = null;
        this.mFrameDataIsAutoFocusInProgress = false;
        this.mFoundBarcodeOrQRCode = false;

        initialize(context);
    }

    public ScannerLiveView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        this.mPreviewFrameSize = 0;
        this.mScannerHandler = new ScannerHandler();
        this.mIsPreviewStarted = false;
        this.mIsAutoFocusInProgress = false;
        this.mFrameData = null;
        this.mFrameDataIsAutoFocusInProgress = false;
        this.mFoundBarcodeOrQRCode = false;
        initialize(context);
    }

    private void initialize(Context context) {
        //this.mListener = (Listener) context;
        SurfaceHolder holder = getHolder();
        holder.addCallback(this);
        holder.setType(3);
        holder.setKeepScreenOn(true);
        this.mAutoFocusCallback = new ScanAutoFocusCallback();
        this.mPreviewCallback = new ScanPreviewCallback();
        Display display = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
        screen = new Point();
        display.getSize(screen);
    }


    public void surfaceCreated(SurfaceHolder holder) {
        if (holder == null) {
            throw new IllegalStateException("No SurfaceHolder provided");
        } else if (this.mCamera == null) {
            try {
                this.mCamera = null;
                try {
                    Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
                    int cameraCount = Camera.getNumberOfCameras();
                    for (int camIdx = 0; camIdx < cameraCount; camIdx++) {
                        Camera.getCameraInfo(camIdx, cameraInfo);
                        if (cameraInfo.facing == 0) {
                            this.mCamera = Camera.open(camIdx);
                            //this.mCamera.setDisplayOrientation(90);
                            setCameraDisplayOrientation(camIdx);
                            break;
                        }
                    }
                } catch (LinkageError e) {
                }
                if (this.mCamera == null) {
                    try {
                        this.mCamera = Camera.open(0);
                        //this.mCamera.setDisplayOrientation(90);
                        setCameraDisplayOrientation(0);
                    } catch (LinkageError e2) {
                    }
                }
                if (this.mCamera == null) {
                    this.mCamera = Camera.open();
                }
                this.mCamera.setPreviewDisplay(holder);
                this.mIsPreviewStarted = false;
            } catch (IOException e3) {
                this.mListener.onCameraIssue();
            } catch (RuntimeException e4) {
                this.mListener.onCameraIssue();
            }
        }
    }

    public void setCameraDisplayOrientation(int cameraId) {
        Camera.CameraInfo info = new Camera.CameraInfo();
        Camera.getCameraInfo(cameraId, info);
        int degrees = 0;
        switch (((Activity) getContext()).getWindowManager().getDefaultDisplay().getRotation()) {
            case 0:
                degrees = 0;
                break;
            case 1:
                degrees = 90;
                break;
            case 2:
                degrees = 180;
                break;
            case 3:
                degrees = 270;
                break;
        }
        if (info.facing == 1) {
            this.mCameraDisplayOrientation = (info.orientation + degrees) % 360;
            this.mCameraDisplayOrientation = (360 - this.mCameraDisplayOrientation) % 360;
        } else {
            this.mCameraDisplayOrientation = ((info.orientation - degrees) + 360) % 360;
        }
        this.mCamera.setDisplayOrientation(this.mCameraDisplayOrientation);
    }

    public void surfaceDestroyed(SurfaceHolder holder) {
        if (this.mCamera != null) {
            stopCamera();
            this.mPreviewFrameSize = 0;
        }
    }

    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
        if (this.mCamera != null) {
            stopCamera();
            setCameraParameters(screen.y,screen.x);
            if (isShown()) {
                startCamera();
            }
        }
    }

    public void onPause() {
        stopCamera();
        if (this.mCamera != null) {
            this.mCamera.release();
            this.mCamera = null;
        }
        if(this.mDecodingThread!=null) {
          Message.obtain(this.mDecodingThread.getHandler(), BarQrCodeUtil.QUIT).sendToTarget();
          try {
            this.mDecodingThread.join();
          } catch (InterruptedException e) {
          }
        }
        if(this.mScannerHandler!=null) {
          this.mScannerHandler.removeMessages(BarQrCodeUtil.AUTO_FOCUS);
          this.mScannerHandler.removeMessages(BarQrCodeUtil.BARCODE);
          this.mScannerHandler.removeMessages(BarQrCodeUtil.QRCODE);
        }
        this.mDecodingThread = null;
    }

    public void setTorch(boolean on) {
        if (this.mCamera != null) {
            boolean isOn = isTorchOn();
            if (on != isOn) {

                Camera.Parameters parameters = this.mCamera.getParameters();
                CameraConfigurationUtils.setTorch(parameters, on);
                this.mCamera.setParameters(parameters);
            }
        }
    }

    public boolean isTorchOn() {
        Camera.Parameters parameters = this.mCamera.getParameters();
        if (parameters != null) {
            String flashMode = parameters.getFlashMode();
            return flashMode != null &&
                (Camera.Parameters.FLASH_MODE_ON.equals(flashMode) ||
                    Camera.Parameters.FLASH_MODE_TORCH.equals(flashMode));
        } else {
            return false;
        }
    }

    public void onResume() {
        /*this.mScannerOverlay = (ScannerOverlay) ((Activity) getContext()).findViewById(R.id.scanner_overlay);
        this.mScannerOverlay.setBarcodeLocation(-1, -1);*/
        this.mDecodingThread = new DecodingThread();
        this.mDecodingThread.start();
        startCamera();
    }

    public void startCamera() {
        if (this.mCamera != null && !this.mIsPreviewStarted) {
            this.mIsAutoFocusInProgress = false;
            this.mFrameData = null;
            if (this.mDecodingThread != null) {
                this.mDecodingThread.initialize();
            }
            if (this.mPreviewFrameSize > 0) {
                try {
                    this.mCallbackBuffer = new byte[this.mPreviewFrameSize];
                } catch (OutOfMemoryError e) {
                    this.mCallbackBuffer = null;
                }
            }
            if (this.mCallbackBuffer != null) {
                this.mCamera.setPreviewCallbackWithBuffer(this.mPreviewCallback);
                this.mCamera.addCallbackBuffer(this.mCallbackBuffer);
            } else {
                this.mCamera.setPreviewCallback(this.mPreviewCallback);
            }
            try {
                this.mCamera.startPreview();
                this.mIsPreviewStarted = true;
                requestAutoFocus();
            } catch (RuntimeException e2) {
                this.mCamera.setPreviewCallback(null);
                this.mCallbackBuffer = null;
                this.mListener.onCameraIssue();
            }
        }
    }

    public void stopCamera() {
        /*if (this.mCamera != null && this.mIsPreviewStarted) {
            this.mCamera.setPreviewCallback(null);
            this.mCamera.stopPreview();
            this.mIsPreviewStarted = false;
            if (this.mDecodingThread.getHandler() != null) {
                this.mDecodingThread.getHandler().removeMessages(R.id.decode);
            }
            this.mScannerHandler.removeMessages(R.id.auto_focus);
            this.mScannerHandler.removeMessages(R.id.msg_barcode_found);
        }*/

        if (this.mCamera != null && this.mIsPreviewStarted) {
            synchronized (this.mPreviewCallback) {
                this.mCamera.setPreviewCallback(null);
                this.mCamera.stopPreview();
                this.mIsPreviewStarted = false;
                this.mCamera.cancelAutoFocus();
                this.mFrameData = null;
                this.mCallbackBuffer = null;
            }
            if (!(this.mDecodingThread == null || this.mDecodingThread.getHandler() == null)) {
                this.mDecodingThread.getHandler().removeMessages(BarQrCodeUtil.DECODE);
            }
            this.mScannerHandler.removeMessages(BarQrCodeUtil.AUTO_FOCUS);
            this.mScannerHandler.removeMessages(BarQrCodeUtil.BARCODE);
            this.mScannerHandler.removeMessages(BarQrCodeUtil.QRCODE);
        }

    }

    void requestAutoFocus() {
        if (this.mCamera != null && this.mHasCameraAutoFocus && this.mIsPreviewStarted && !this.mIsAutoFocusInProgress) {
            try {
                this.mCamera.autoFocus(this.mAutoFocusCallback);
                this.mIsAutoFocusInProgress = true;
            } catch (RuntimeException e) {
                Log.e(TAG, "Camera auto-focus failed: " + e.getLocalizedMessage());
            }
        }
    }


    private void setCameraParameters(int screenWidth, int screenHeight) {
        Parameters parameters = this.mCamera.getParameters();
        Log.e(TAG,"screenWidth="+screenWidth+",screenHeight="+screenHeight);
        setPreviewSize(screenWidth, screenHeight, parameters);
        setZoom(parameters);
        String focusModeValues = parameters.get("focus-mode-values");
        if (focusModeValues != null && focusModeValues.contains("macro")) {
            parameters.set("focus-mode", "macro");
            this.mHasCameraAutoFocus = true;
        } else if (focusModeValues == null || !focusModeValues.contains("auto")) {
            this.mHasCameraAutoFocus = false;
        } else {
            parameters.set("focus-mode", "auto");
            this.mHasCameraAutoFocus = true;
        }
        try {
            this.mCamera.setParameters(parameters);
        } catch (RuntimeException e) {
            this.mListener.onCameraIssue();
        }
        parameters = this.mCamera.getParameters();
        this.mPreviewSize = parameters.getPreviewSize();
        this.mPreviewFrameSize = ((this.mPreviewSize.width * this.mPreviewSize.height) * ImageFormat.getBitsPerPixel(parameters.getPreviewFormat())) / 8;
    }

    private void setPreviewSize(int screenWidth, int screenHeight, Parameters parameters) {
        String previewSizeValuesString = parameters.get("preview-size-values");
        if (previewSizeValuesString == null) {
            previewSizeValuesString = parameters.get("preview-size-value");
        }
        Point bestPreviewSize = null;
        if (previewSizeValuesString != null) {
            bestPreviewSize = findBestPreviewSize(previewSizeValuesString, screenWidth, screenHeight);
        }
        if (bestPreviewSize != null) {
            parameters.setPreviewSize(bestPreviewSize.x, bestPreviewSize.y);
        } else {
            parameters.setPreviewSize((screenWidth >> 4) << 4, (screenHeight >> 1) << 1);
        }
    }

    private static Point findBestPreviewSize(CharSequence previewSizeValuesString, int screenWidth, int screenHeight) {
        int bestPreviewWidth = 0;
        int bestPreviewHeight = 0;
        int minDifference = Integer.MAX_VALUE;
        for (String previewSizeValueString : COMMA_PATTERN.split(previewSizeValuesString)) {
            String previewSizeValueString2 = previewSizeValueString.trim();
            int separatorPosition = previewSizeValueString2.indexOf(120);
            if (separatorPosition >= 0) {
                try {
                    int previewSizeValueWidth = Integer.parseInt(previewSizeValueString2.substring(0, separatorPosition));
                    int previewSizeValueHeight = Integer.parseInt(previewSizeValueString2.substring(separatorPosition + 1));
                    int difference = Math.abs(previewSizeValueWidth - screenWidth) + Math.abs(previewSizeValueHeight - screenHeight);
                    if (difference == 0) {
                        bestPreviewWidth = previewSizeValueWidth;
                        bestPreviewHeight = previewSizeValueHeight;
                        break;
                    } else if (difference < minDifference) {
                        bestPreviewWidth = previewSizeValueWidth;
                        bestPreviewHeight = previewSizeValueHeight;
                        minDifference = difference;
                    }
                } catch (NumberFormatException e) {
                }
            }
        }
        if (bestPreviewWidth <= 0 || bestPreviewHeight <= 0) {
            return null;
        }
        Log.e(TAG,"bestPreviewWidth="+bestPreviewWidth+",bestPreviewHeight="+bestPreviewHeight);
        return new Point(bestPreviewWidth, bestPreviewHeight);
    }

    private void setZoom(Parameters parameters) {
        String zoomSupportedString = parameters.get("zoom-supported");
        if (zoomSupportedString == null || Boolean.parseBoolean(zoomSupportedString)) {
            int maxZoomX10;
            int desiredZoomX10 = DESIRED_ZOOM_X10;
            String maxZoomString = parameters.get("max-zoom");
            if (maxZoomString != null) {
                try {
                    maxZoomX10 = (int) (10.0d * Double.parseDouble(maxZoomString));
                    if (DESIRED_ZOOM_X10 > maxZoomX10) {
                        desiredZoomX10 = maxZoomX10;
                    }
                } catch (NumberFormatException e) {
                }
            }
            String takingPictureZoomMaxString = parameters.get("taking-picture-zoom-max");
            if (takingPictureZoomMaxString != null) {
                try {
                    maxZoomX10 = Integer.parseInt(takingPictureZoomMaxString);
                    if (desiredZoomX10 > maxZoomX10) {
                        desiredZoomX10 = maxZoomX10;
                    }
                } catch (NumberFormatException e2) {
                }
            }
            String motZoomValuesString = parameters.get("mot-zoom-values");
            if (motZoomValuesString != null) {
                desiredZoomX10 = findBestMotZoomX10(motZoomValuesString, desiredZoomX10);
            }
            String motZoomStepString = parameters.get("mot-zoom-step");
            if (motZoomStepString != null) {
                try {
                    int motZoomStepX10 = (int) (10.0d * Double.parseDouble(motZoomStepString.trim()));
                    if (motZoomStepX10 > 1) {
                        desiredZoomX10 -= desiredZoomX10 % motZoomStepX10;
                    }
                } catch (NumberFormatException e3) {
                }
            }
            if (!(maxZoomString == null && motZoomValuesString == null)) {
                parameters.set("zoom", String.valueOf(((double) desiredZoomX10) / 10.0d));
            }
            if (takingPictureZoomMaxString != null) {
                parameters.set("taking-picture-zoom", desiredZoomX10);
            }
        }
    }

    private static int findBestMotZoomX10(CharSequence motZoomValuesString, int desiredZoomX10) {
        int bestMotZoomX10 = 0;
        String[] split = COMMA_PATTERN.split(motZoomValuesString);
        int length = split.length;
        int i = 0;
        while (i < length) {
            try {
                int motZoomValueX10 = (int) (10.0d * Double.parseDouble(split[i].trim()));
                if (Math.abs(desiredZoomX10 - motZoomValueX10) < Math.abs(desiredZoomX10 - bestMotZoomX10)) {
                    bestMotZoomX10 = motZoomValueX10;
                }
                i++;
            } catch (NumberFormatException e) {
                return desiredZoomX10;
            }
        }
        return bestMotZoomX10;
    }
    /*private void setCameraParameters(int screenWidth, int screenHeight) {
        Parameters parameters = this.mCamera.getParameters();
        setPreviewSize(screenWidth, screenHeight, parameters);
        parameters.set("flash-mode", "off");
        parameters.set("flash-value", 2);
        setZoom(parameters);
        int apiLevel = Integer.parseInt(VERSION.SDK);
        if (apiLevel <= 3) {
            this.mHasCameraAutoFocus = true;
        } else if (apiLevel >= 5) {
            String focusModeValues = parameters.get("focus-mode-values");
            if (focusModeValues != null && focusModeValues.contains("macro")) {
                parameters.set("focus-mode", "macro");
                this.mHasCameraAutoFocus = true;
            } else if (focusModeValues == null || !focusModeValues.contains("auto")) {
                this.mHasCameraAutoFocus = false;
            } else {
                parameters.set("focus-mode", "auto");
                this.mHasCameraAutoFocus = true;
            }
        } else {
            String model = Build.MODEL.toLowerCase();
            if (model.contains("devour") || model.contains("tattoo")) {
                this.mHasCameraAutoFocus = false;
            } else {
                this.mHasCameraAutoFocus = true;
            }
        }
        this.mCamera.setParameters(parameters);
        this.mPreviewSize = parameters.getPreviewSize();
    }

    private void setPreviewSize(int screenWidth, int screenHeight, Parameters parameters) {
        String previewSizeValuesString = parameters.get("preview-size-values");
        if (previewSizeValuesString == null) {
            previewSizeValuesString = parameters.get("preview-size-value");
        }
        Point bestPreviewSize = null;
        if (previewSizeValuesString != null) {
            bestPreviewSize = findBestPreviewSize(previewSizeValuesString, screenWidth, screenHeight);
        }
        if (bestPreviewSize != null) {
            parameters.setPreviewSize(bestPreviewSize.x, bestPreviewSize.y);
        } else {
            parameters.setPreviewSize((screenWidth >> 3) << 3, (screenHeight >> 3) << 3);
        }
    }

    private static Point findBestPreviewSize(CharSequence previewSizeValuesString, int screenWidth, int screenHeight) {
        int bestPreviewWidth = 0;
        int bestPreviewHeight = 0;
        int minDifference = Integer.MAX_VALUE;
        for (String previewSizeValueString : COMMA_PATTERN.split(previewSizeValuesString)) {
            String previewSizeValueString2 = previewSizeValueString.trim();
            int separatorPosition = previewSizeValueString2.indexOf(120);
            if (separatorPosition >= 0) {
                try {
                    int previewSizeValueWidth = Integer.parseInt(previewSizeValueString2.substring(0, separatorPosition));
                    int previewSizeValueHeight = Integer.parseInt(previewSizeValueString2.substring(separatorPosition + 1));
                    int difference = Math.abs(previewSizeValueWidth - screenWidth) + Math.abs(previewSizeValueHeight - screenHeight);
                    if (difference == 0) {
                        bestPreviewWidth = previewSizeValueWidth;
                        bestPreviewHeight = previewSizeValueHeight;
                        break;
                    } else if (difference < minDifference) {
                        bestPreviewWidth = previewSizeValueWidth;
                        bestPreviewHeight = previewSizeValueHeight;
                        minDifference = difference;
                    }
                } catch (NumberFormatException e) {
                }
            }
        }
        if (bestPreviewWidth <= 0 || bestPreviewHeight <= 0) {
            return null;
        }
        return new Point(bestPreviewWidth, bestPreviewHeight);
    }

    private void setZoom(Parameters parameters) {
        String zoomSupportedString = parameters.get("zoom-supported");
        if (zoomSupportedString == null || Boolean.parseBoolean(zoomSupportedString)) {
            int maxZoomX10;
            int desiredZoomX10 = DESIRED_ZOOM_X10;
            String maxZoomString = parameters.get("max-zoom");
            if (maxZoomString != null) {
                try {
                    maxZoomX10 = (int) (10.0d * Double.parseDouble(maxZoomString));
                    if (DESIRED_ZOOM_X10 > maxZoomX10) {
                        desiredZoomX10 = maxZoomX10;
                    }
                } catch (NumberFormatException e) {
                }
            }
            String takingPictureZoomMaxString = parameters.get("taking-picture-zoom-max");
            if (takingPictureZoomMaxString != null) {
                try {
                    maxZoomX10 = Integer.parseInt(takingPictureZoomMaxString);
                    if (desiredZoomX10 > maxZoomX10) {
                        desiredZoomX10 = maxZoomX10;
                    }
                } catch (NumberFormatException e2) {
                }
            }
            String motZoomValuesString = parameters.get("mot-zoom-values");
            if (motZoomValuesString != null) {
                desiredZoomX10 = findBestMotZoomX10(motZoomValuesString, desiredZoomX10);
            }
            String motZoomStepString = parameters.get("mot-zoom-step");
            if (motZoomStepString != null) {
                try {
                    int motZoomStepX10 = (int) (10.0d * Double.parseDouble(motZoomStepString.trim()));
                    if (motZoomStepX10 > 1) {
                        desiredZoomX10 -= desiredZoomX10 % motZoomStepX10;
                    }
                } catch (NumberFormatException e3) {
                }
            }
            if (!(maxZoomString == null && motZoomValuesString == null)) {
                parameters.set("zoom", String.valueOf(((double) desiredZoomX10) / 10.0d));
            }
            if (takingPictureZoomMaxString != null) {
                parameters.set("taking-picture-zoom", desiredZoomX10);
            }
        }
    }

    private static int findBestMotZoomX10(CharSequence motZoomValuesString, int desiredZoomX10) {
        int bestMotZoomX10 = 0;
        String[] split = COMMA_PATTERN.split(motZoomValuesString);
        int length = split.length;
        int i = 0;
        while (i < length) {
            try {
                int motZoomValueX10 = (int) (10.0d * Double.parseDouble(split[i].trim()));
                if (Math.abs(desiredZoomX10 - motZoomValueX10) < Math.abs(desiredZoomX10 - bestMotZoomX10)) {
                    bestMotZoomX10 = motZoomValueX10;
                }
                i++;
            } catch (NumberFormatException e) {
                return desiredZoomX10;
            }
        }
        return bestMotZoomX10;
    }*/

    static {
        TAG = ScannerLiveView.class.getSimpleName();
        COMMA_PATTERN = Pattern.compile(",");
    }
}
