package com.visionsmarts.pic2shop.view;

import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.graphics.ImageFormat;
import android.graphics.Point;
import android.hardware.Camera;
import android.hardware.Camera.AutoFocusCallback;
import android.hardware.Camera.CameraInfo;
import android.hardware.Camera.Parameters;
import android.hardware.Camera.PreviewCallback;
import android.hardware.Camera.Size;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.AttributeSet;
import android.util.Log;
import android.view.SurfaceHolder;
import android.view.SurfaceHolder.Callback;
import android.view.SurfaceView;

import com.visionsmarts.VSBarcodeReader;
import com.visionsmarts.VSBarcodeReader.DecoderValues;
import com.visionsmarts.VSReaderQR;

import java.io.IOException;
import java.util.regex.Pattern;

public class ScannerLiveViewLiner extends SurfaceView implements Callback {
    private static final int BARCODE_TYPES_MASK = 7;
    private static final Pattern COMMA_PATTERN;
    private static final int DESIRED_ZOOM_X10 = 27;
    private static final String TAG;
    private AutoFocusCallback mAutoFocusCallback;
    private volatile byte[] mCallbackBuffer;
    private volatile Camera mCamera;
    private volatile int mCameraDisplayOrientation;
    private DecodingThread mDecodingThread;
    volatile boolean mFoundBarcodeOrQRCode;
    volatile byte[] mFrameData;
    volatile boolean mFrameDataIsAutoFocusInProgress;
    volatile boolean mHasCameraAutoFocus;
    volatile boolean mIsAutoFocusInProgress;
    volatile boolean mIsPreviewStarted;
    Listener mListener;
    private PreviewCallback mPreviewCallback;
    private int mPreviewFrameSize;
    volatile Size mPreviewSize;
    final ScannerHandler mScannerHandler;
    ScannerOverlay mScannerOverlay;

    /* renamed from: com.visionsmarts.pic2shop.view.ScannerLiveViewLiner.1 */
    class C01731 implements AutoFocusCallback {
        C01731() {
        }

        public void onAutoFocus(boolean success, Camera camera) {
            ScannerLiveViewLiner.this.mIsAutoFocusInProgress = false;
        }
    }

    /* renamed from: com.visionsmarts.pic2shop.view.ScannerLiveViewLiner.2 */
    class C01742 implements PreviewCallback {
        C01742() {
        }

        public synchronized void onPreviewFrame(byte[] frameData, Camera camera) {
            if (!(!ScannerLiveViewLiner.this.mIsPreviewStarted || ScannerLiveViewLiner.this.mDecodingThread == null || ScannerLiveViewLiner.this.mFoundBarcodeOrQRCode)) {
                boolean shouldSendDecodeRequest = ScannerLiveViewLiner.this.mFrameData == null;
                ScannerLiveViewLiner.this.mFrameData = frameData;
                ScannerLiveViewLiner.this.mFrameDataIsAutoFocusInProgress = ScannerLiveViewLiner.this.mIsAutoFocusInProgress;
                if (shouldSendDecodeRequest) {
                    ScannerLiveViewLiner.this.mDecodingThread.getHandler().obtainMessage(BarQrCodeUtil.DECODE).sendToTarget();
                }
            }
        }
    }

    private class DecodingThread extends Thread {
        DecoderValues mBarcodeDecoderValues;
        private Handler mHandler;
        private int mNumberConsecutiveFailedDecodingSinceFocusEnded;
        private byte[] mPreviewLine;
        VSReaderQR.DecoderValues mQRDecoderValues;

        /* renamed from: com.visionsmarts.pic2shop.view.ScannerLiveViewLiner.DecodingThread.1 */
        class C01751 extends Handler {
            C01751() {
            }

            public void handleMessage(Message message) {
                switch (message.what) {
                    case BarQrCodeUtil.DECODE:
                        if (ScannerLiveViewLiner.this.mIsPreviewStarted && !ScannerLiveViewLiner.this.mFoundBarcodeOrQRCode) {
                            DecodingThread.this.decode();
                        }
                        break;
                    case BarQrCodeUtil.QUIT:
                        Looper.myLooper().quit();
                        break;
                    default:
                }
            }
        }

        public DecodingThread() {
            this.mNumberConsecutiveFailedDecodingSinceFocusEnded = 0;
            this.mBarcodeDecoderValues = new DecoderValues();
            this.mQRDecoderValues = new VSReaderQR.DecoderValues();
            setPriority(10);
            initialize();
        }

        synchronized Handler getHandler() {
            if (this.mHandler == null) {
                try {
                    ScannerLiveViewLiner.this.mDecodingThread.wait(1000);
                } catch (InterruptedException e) {
                }
            }
            return this.mHandler;
        }

        public void initialize() {
            if (ScannerLiveViewLiner.this.mPreviewSize != null && (this.mPreviewLine == null || this.mPreviewLine.length != ScannerLiveViewLiner.this.mPreviewSize.height)) {
                this.mPreviewLine = new byte[ScannerLiveViewLiner.this.mPreviewSize.height];
            }
            this.mNumberConsecutiveFailedDecodingSinceFocusEnded = 0;
            ScannerLiveViewLiner.this.mFoundBarcodeOrQRCode = false;
            VSBarcodeReader.reset();
        }

        public void run() {
            Looper.prepare();
            synchronized (this) {
                this.mHandler = new C01751();
                notify();
            }
            Looper.loop();
        }

        @TargetApi(8)
        private void decode() {
            byte[] previewLine = this.mPreviewLine;
            int width = ScannerLiveViewLiner.this.mPreviewSize.width;
            int height = ScannerLiveViewLiner.this.mPreviewSize.height;
            synchronized (ScannerLiveViewLiner.this.mPreviewCallback) {
                if (ScannerLiveViewLiner.this.mFrameData == null) {
                    return;
                }
                byte[] frameData = ScannerLiveViewLiner.this.mFrameData;
                boolean wasAutoFocusInProgress = ScannerLiveViewLiner.this.mFrameDataIsAutoFocusInProgress;
                int offset = Math.round(ScannerLiveViewLiner.this.mScannerOverlay.getNormalizedLaserLinePosition() * ((float) width));
                Log.e(TAG,"offset="+offset);
                int i = previewLine.length - 1;
                while (i >= 0) {
                    try {
                        previewLine[i] = frameData[offset];
                        offset += width;
                        i--;
                    } catch (ArrayIndexOutOfBoundsException e) {
                        throw new ArrayIndexOutOfBoundsException(e.getMessage() + "; width=" + width + "; height=" + height + "; previewLine.length=" + previewLine.length + "; initOffset=" + Math.round(ScannerLiveViewLiner.this.mScannerOverlay.getNormalizedLaserLinePosition() * ((float) width)));
                    }
                }
                if (ScannerLiveViewLiner.this.mFrameData != null) {
                    String barcode = VSBarcodeReader.decodeNextImage(previewLine, ScannerLiveViewLiner.this.mHasCameraAutoFocus ? 1 : 0, ScannerLiveViewLiner.BARCODE_TYPES_MASK, this.mBarcodeDecoderValues);
                    if (ScannerLiveViewLiner.this.mFrameData != null) {
                        if (this.mBarcodeDecoderValues.left >= 0) {
                            int left = this.mBarcodeDecoderValues.left;
                            int right = this.mBarcodeDecoderValues.right;
                            float lineLength = (float) previewLine.length;
                            if (ScannerLiveViewLiner.this.mCameraDisplayOrientation == 180) {
                                ScannerLiveViewLiner.this.mScannerOverlay.setNormalizedBarcodeLocation(((float) (height - right)) / lineLength, ((float) (height - left)) / lineLength);
                            } else {
                                ScannerLiveViewLiner.this.mScannerOverlay.setNormalizedBarcodeLocation(((float) left) / lineLength, ((float) right) / lineLength);
                            }
                        } else {
                            ScannerLiveViewLiner.this.mScannerOverlay.setNoBarcodeLocation();
                        }
                        if (barcode.length() > 0) {
                            ScannerLiveViewLiner.this.mFoundBarcodeOrQRCode = true;
                            ScannerLiveViewLiner.this.mScannerHandler.obtainMessage(BarQrCodeUtil.BARCODE, barcode).sendToTarget();
                        } else {
                            byte[] qrData = VSReaderQR.decodeImage(frameData, width, height, this.mQRDecoderValues);
                            if (qrData.length > 0) {
                                ScannerLiveViewLiner.this.mFoundBarcodeOrQRCode = true;
                                ScannerLiveViewLiner.this.mScannerHandler.obtainMessage(BarQrCodeUtil.QRCODE, qrData).sendToTarget();
                            } else if (wasAutoFocusInProgress) {
                                this.mNumberConsecutiveFailedDecodingSinceFocusEnded = 0;
                            } else {
                                this.mNumberConsecutiveFailedDecodingSinceFocusEnded++;
                                if (ScannerLiveViewLiner.this.mHasCameraAutoFocus && !ScannerLiveViewLiner.this.mIsAutoFocusInProgress && this.mNumberConsecutiveFailedDecodingSinceFocusEnded >= 4) {
                                    ScannerLiveViewLiner.this.mScannerHandler.obtainMessage(BarQrCodeUtil.AUTO_FOCUS).sendToTarget();
                                }
                            }
                        }
                        synchronized (ScannerLiveViewLiner.this.mPreviewCallback) {
                            ScannerLiveViewLiner.this.mFrameData = null;
                            if (!(ScannerLiveViewLiner.this.mFoundBarcodeOrQRCode || ScannerLiveViewLiner.this.mCallbackBuffer == null)) {
                                ScannerLiveViewLiner.this.mCamera.addCallbackBuffer(ScannerLiveViewLiner.this.mCallbackBuffer);
                            }
                        }
                    }
                }
            }
        }
    }

    public interface Listener {
        void onCameraIssue();

        void onCodeScanned(String str, String str2);
    }

    private class ScannerHandler extends Handler {
        private ScannerHandler() {
        }

        public void handleMessage(Message message) {
            switch (message.what) {
                case BarQrCodeUtil.AUTO_FOCUS:
                    ScannerLiveViewLiner.this.requestAutoFocus();
                    break;
                case BarQrCodeUtil.BARCODE:
                    if (ScannerLiveViewLiner.this.mIsPreviewStarted) {
                        ScannerLiveViewLiner.this.stopCamera();
                        ScannerLiveViewLiner.this.mListener.onCodeScanned("BAR", (String)message.obj);
                    }
                    break;
                case BarQrCodeUtil.QRCODE:
                    if (ScannerLiveViewLiner.this.mIsPreviewStarted) {
                        ScannerLiveViewLiner.this.stopCamera();
                        ScannerLiveViewLiner.this.mListener.onCodeScanned("QR", BarQrCodeUtil.qrToString((byte[])message.obj));
                    }
                    break;
                default:
            }
        }
    }

    public ScannerLiveViewLiner(Context context) {
        super(context);
        this.mPreviewFrameSize = 0;
        this.mScannerHandler = new ScannerHandler();
        this.mIsPreviewStarted = false;
        this.mIsAutoFocusInProgress = false;
        this.mFrameData = null;
        this.mFrameDataIsAutoFocusInProgress = false;
        this.mFoundBarcodeOrQRCode = false;
        initialize(context);
    }

    public ScannerLiveViewLiner(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.mPreviewFrameSize = 0;
        this.mScannerHandler = new ScannerHandler();
        this.mIsPreviewStarted = false;
        this.mIsAutoFocusInProgress = false;
        this.mFrameData = null;
        this.mFrameDataIsAutoFocusInProgress = false;
        this.mFoundBarcodeOrQRCode = false;
        initialize(context);
    }

    public ScannerLiveViewLiner(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        this.mPreviewFrameSize = 0;
        this.mScannerHandler = new ScannerHandler();
        this.mIsPreviewStarted = false;
        this.mIsAutoFocusInProgress = false;
        this.mFrameData = null;
        this.mFrameDataIsAutoFocusInProgress = false;
        this.mFoundBarcodeOrQRCode = false;
        initialize(context);
    }

    private void initialize(Context context) {
        this.mListener = (Listener) context;
        SurfaceHolder holder = getHolder();
        holder.addCallback(this);
        holder.setType(3);
        holder.setKeepScreenOn(true);
        this.mAutoFocusCallback = new C01731();
        this.mPreviewCallback = new C01742();
    }

    @TargetApi(9)
    public void surfaceCreated(SurfaceHolder holder) {
        if (holder == null) {
            throw new IllegalStateException("No SurfaceHolder provided");
        } else if (this.mCamera == null) {
            try {
                this.mCamera = null;
                try {
                    CameraInfo cameraInfo = new CameraInfo();
                    int cameraCount = Camera.getNumberOfCameras();
                    for (int camIdx = 0; camIdx < cameraCount; camIdx++) {
                        Camera.getCameraInfo(camIdx, cameraInfo);
                        if (cameraInfo.facing == 0) {
                            this.mCamera = Camera.open(camIdx);
                            setCameraDisplayOrientation(camIdx);
                            break;
                        }
                    }
                } catch (LinkageError e) {
                }
                if (this.mCamera == null) {
                    try {
                        this.mCamera = Camera.open(0);
                        setCameraDisplayOrientation(0);
                    } catch (LinkageError e2) {
                    }
                }
                if (this.mCamera == null) {
                    this.mCamera = Camera.open();
                }
                this.mCamera.setPreviewDisplay(holder);
                this.mIsPreviewStarted = false;
            } catch (IOException e3) {
                this.mListener.onCameraIssue();
            } catch (RuntimeException e4) {
                this.mListener.onCameraIssue();
            }
        }
    }

    public void surfaceDestroyed(SurfaceHolder holder) {
        if (this.mCamera != null) {
            stopCamera();
            this.mPreviewFrameSize = 0;
        }
    }

    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
        if (this.mCamera != null) {
            stopCamera();
            setCameraParameters(width, height);
            if (isShown()) {
                startCamera();
            }
        }
    }

    public void onPause() {
        stopCamera();
        if (this.mCamera != null) {
            this.mCamera.release();
            this.mCamera = null;
        }
        Message.obtain(this.mDecodingThread.getHandler(), BarQrCodeUtil.QUIT).sendToTarget();
        try {
            this.mDecodingThread.join();
        } catch (InterruptedException e) {
        }
        this.mScannerHandler.removeMessages(BarQrCodeUtil.AUTO_FOCUS);
        this.mScannerHandler.removeMessages(BarQrCodeUtil.BARCODE);
        this.mScannerHandler.removeMessages(BarQrCodeUtil.QRCODE);
        this.mDecodingThread = null;
    }

    @TargetApi(8)
    public void startCamera() {
        if (this.mCamera != null && !this.mIsPreviewStarted) {
            this.mIsAutoFocusInProgress = false;
            this.mFrameData = null;
            if (this.mDecodingThread != null) {
                this.mDecodingThread.initialize();
            }
            if (this.mScannerOverlay != null) {
                this.mScannerOverlay.setNoBarcodeLocation();
            }
            if (this.mPreviewFrameSize > 0) {
                try {
                    this.mCallbackBuffer = new byte[this.mPreviewFrameSize];
                } catch (OutOfMemoryError e) {
                    this.mCallbackBuffer = null;
                }
            }
            if (this.mCallbackBuffer != null) {
                this.mCamera.setPreviewCallbackWithBuffer(this.mPreviewCallback);
                this.mCamera.addCallbackBuffer(this.mCallbackBuffer);
            } else {
                this.mCamera.setPreviewCallback(this.mPreviewCallback);
            }
            try {
                this.mCamera.startPreview();
                this.mIsPreviewStarted = true;
                requestAutoFocus();
            } catch (RuntimeException e2) {
                this.mCamera.setPreviewCallback(null);
                this.mCallbackBuffer = null;
                this.mListener.onCameraIssue();
            }
        }
    }

    public void stopCamera() {
        if (this.mCamera != null && this.mIsPreviewStarted) {
            synchronized (this.mPreviewCallback) {
                this.mCamera.setPreviewCallback(null);
                this.mCamera.stopPreview();
                this.mIsPreviewStarted = false;
                this.mCamera.cancelAutoFocus();
                this.mFrameData = null;
                this.mCallbackBuffer = null;
            }
            if (!(this.mDecodingThread == null || this.mDecodingThread.getHandler() == null)) {
                this.mDecodingThread.getHandler().removeMessages(BarQrCodeUtil.DECODE);
            }
            this.mScannerHandler.removeMessages(BarQrCodeUtil.AUTO_FOCUS);
            this.mScannerHandler.removeMessages(BarQrCodeUtil.BARCODE);
            this.mScannerHandler.removeMessages(BarQrCodeUtil.QRCODE);
        }
    }

    void requestAutoFocus() {
        if (this.mCamera != null && this.mHasCameraAutoFocus && this.mIsPreviewStarted && !this.mIsAutoFocusInProgress) {
            try {
                this.mCamera.autoFocus(this.mAutoFocusCallback);
                this.mIsAutoFocusInProgress = true;
            } catch (RuntimeException e) {
                Log.e(TAG, "Camera auto-focus failed: " + e.getLocalizedMessage());
            }
        }
    }

    @TargetApi(9)
    public void setCameraDisplayOrientation(int cameraId) {
        CameraInfo info = new CameraInfo();
        Camera.getCameraInfo(cameraId, info);
        int degrees = 0;
        switch (((Activity) getContext()).getWindowManager().getDefaultDisplay().getRotation()) {
            case 0 /*0*/:
                degrees = 0;
                break;
            case 1 /*1*/:
                degrees = 90;
                break;
            case 2 /*2*/:
                degrees = 180;
                break;
            case 3 /*3*/:
                degrees = 270;
                break;
        }
        if (info.facing == 1) {
            this.mCameraDisplayOrientation = (info.orientation + degrees) % 360;
            this.mCameraDisplayOrientation = (360 - this.mCameraDisplayOrientation) % 360;
        } else {
            this.mCameraDisplayOrientation = ((info.orientation - degrees) + 360) % 360;
        }
        this.mCamera.setDisplayOrientation(this.mCameraDisplayOrientation);
    }

    @TargetApi(8)
    private void setCameraParameters(int screenWidth, int screenHeight) {
        Log.e(TAG,"screenWidth="+screenWidth+",screenHeight="+screenHeight);
        Parameters parameters = this.mCamera.getParameters();
        setPreviewSize(screenWidth, screenHeight, parameters);
        setZoom(parameters);
        String focusModeValues = parameters.get("focus-mode-values");
        if (focusModeValues != null && focusModeValues.contains("macro")) {
            parameters.set("focus-mode", "macro");
            this.mHasCameraAutoFocus = true;
        } else if (focusModeValues == null || !focusModeValues.contains("auto")) {
            this.mHasCameraAutoFocus = false;
        } else {
            parameters.set("focus-mode", "auto");
            this.mHasCameraAutoFocus = true;
        }
        try {
            this.mCamera.setParameters(parameters);
        } catch (RuntimeException e) {
            this.mListener.onCameraIssue();
        }
        parameters = this.mCamera.getParameters();
        this.mPreviewSize = parameters.getPreviewSize();
        this.mPreviewFrameSize = ((this.mPreviewSize.width * this.mPreviewSize.height) * ImageFormat.getBitsPerPixel(parameters.getPreviewFormat())) / 8;
    }

    private void setPreviewSize(int screenWidth, int screenHeight, Parameters parameters) {
        String previewSizeValuesString = parameters.get("preview-size-values");
        if (previewSizeValuesString == null) {
            previewSizeValuesString = parameters.get("preview-size-value");
        }
        Point bestPreviewSize = null;
        if (previewSizeValuesString != null) {
            bestPreviewSize = findBestPreviewSize(previewSizeValuesString, screenWidth, screenHeight);
        }
        if (bestPreviewSize != null) {
            parameters.setPreviewSize(bestPreviewSize.x, bestPreviewSize.y);
        } else {
            parameters.setPreviewSize((screenWidth >> 4) << 4, (screenHeight >> 1) << 1);
        }
    }

    private static Point findBestPreviewSize(CharSequence previewSizeValuesString, int screenWidth, int screenHeight) {
        int bestPreviewWidth = 0;
        int bestPreviewHeight = 0;
        int minDifference = Integer.MAX_VALUE;
        for (String previewSizeValueString : COMMA_PATTERN.split(previewSizeValuesString)) {
            String previewSizeValueString2 = previewSizeValueString.trim();
            int separatorPosition = previewSizeValueString2.indexOf(120);
            if (separatorPosition >= 0) {
                try {
                    int previewSizeValueWidth = Integer.parseInt(previewSizeValueString2.substring(0, separatorPosition));
                    int previewSizeValueHeight = Integer.parseInt(previewSizeValueString2.substring(separatorPosition + 1));
                    int difference = Math.abs(previewSizeValueWidth - screenWidth) + Math.abs(previewSizeValueHeight - screenHeight);
                    if (difference == 0) {
                        bestPreviewWidth = previewSizeValueWidth;
                        bestPreviewHeight = previewSizeValueHeight;
                        break;
                    } else if (difference < minDifference) {
                        bestPreviewWidth = previewSizeValueWidth;
                        bestPreviewHeight = previewSizeValueHeight;
                        minDifference = difference;
                    }
                } catch (NumberFormatException e) {
                }
            }
        }
        if (bestPreviewWidth <= 0 || bestPreviewHeight <= 0) {
            return null;
        }
        Log.e(TAG,"bestPreviewWidth="+bestPreviewWidth+",bestPreviewHeight="+bestPreviewHeight);
        return new Point(bestPreviewWidth, bestPreviewHeight);
    }

    private void setZoom(Parameters parameters) {
        String zoomSupportedString = parameters.get("zoom-supported");
        if (zoomSupportedString == null || Boolean.parseBoolean(zoomSupportedString)) {
            int maxZoomX10;
            int desiredZoomX10 = DESIRED_ZOOM_X10;
            String maxZoomString = parameters.get("max-zoom");
            if (maxZoomString != null) {
                try {
                    maxZoomX10 = (int) (10.0d * Double.parseDouble(maxZoomString));
                    if (DESIRED_ZOOM_X10 > maxZoomX10) {
                        desiredZoomX10 = maxZoomX10;
                    }
                } catch (NumberFormatException e) {
                }
            }
            String takingPictureZoomMaxString = parameters.get("taking-picture-zoom-max");
            if (takingPictureZoomMaxString != null) {
                try {
                    maxZoomX10 = Integer.parseInt(takingPictureZoomMaxString);
                    if (desiredZoomX10 > maxZoomX10) {
                        desiredZoomX10 = maxZoomX10;
                    }
                } catch (NumberFormatException e2) {
                }
            }
            String motZoomValuesString = parameters.get("mot-zoom-values");
            if (motZoomValuesString != null) {
                desiredZoomX10 = findBestMotZoomX10(motZoomValuesString, desiredZoomX10);
            }
            String motZoomStepString = parameters.get("mot-zoom-step");
            if (motZoomStepString != null) {
                try {
                    int motZoomStepX10 = (int) (10.0d * Double.parseDouble(motZoomStepString.trim()));
                    if (motZoomStepX10 > 1) {
                        desiredZoomX10 -= desiredZoomX10 % motZoomStepX10;
                    }
                } catch (NumberFormatException e3) {
                }
            }
            if (!(maxZoomString == null && motZoomValuesString == null)) {
                parameters.set("zoom", String.valueOf(((double) desiredZoomX10) / 10.0d));
            }
            if (takingPictureZoomMaxString != null) {
                parameters.set("taking-picture-zoom", desiredZoomX10);
            }
        }
    }

    private static int findBestMotZoomX10(CharSequence motZoomValuesString, int desiredZoomX10) {
        int bestMotZoomX10 = 0;
        String[] split = COMMA_PATTERN.split(motZoomValuesString);
        int length = split.length;
        int i = 0;
        while (i < length) {
            try {
                int motZoomValueX10 = (int) (10.0d * Double.parseDouble(split[i].trim()));
                if (Math.abs(desiredZoomX10 - motZoomValueX10) < Math.abs(desiredZoomX10 - bestMotZoomX10)) {
                    bestMotZoomX10 = motZoomValueX10;
                }
                i++;
            } catch (NumberFormatException e) {
                return desiredZoomX10;
            }
        }
        return bestMotZoomX10;
    }

    static {
        TAG = ScannerLiveViewLiner.class.getSimpleName();
        COMMA_PATTERN = Pattern.compile(",");
    }
}
