package com.visionsmarts.pic2shop.view;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Paint.Align;
import android.graphics.Paint.Style;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.text.Layout.Alignment;
import android.text.StaticLayout;
import android.text.TextPaint;
import android.util.AttributeSet;
import android.view.Display;
import android.view.View;
import android.view.WindowManager;

public class ScannerOverlay extends View {
    private static final float LASER_LINE_WIDTH_160_DPI = 7.0f;
    private static final float TEXT_FONT_SIZE = 20.0f;
    private Paint mBarcodeLinePaint;
    private Rect mBarcodeLineRect;
    private Rect mCanvasClipRect;
    private Paint mLaserLinePaint;
    private int mLaserLinePosition;
    private Rect mLaserLineRect;
    private float mScreenDensity;
    private Point mScreenResolution;
    private Paint mTextBackgroundPaint;
    private StaticLayout mTextLayout;

    public ScannerOverlay(Context context) {
        super(context);
        this.mCanvasClipRect = new Rect();
        this.mLaserLineRect = new Rect();
        this.mBarcodeLineRect = new Rect(-1, -1, -1, -1);
        initialize(context);
    }

    public ScannerOverlay(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.mCanvasClipRect = new Rect();
        this.mLaserLineRect = new Rect();
        this.mBarcodeLineRect = new Rect(-1, -1, -1, -1);
        initialize(context);
    }

    public ScannerOverlay(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        this.mCanvasClipRect = new Rect();
        this.mLaserLineRect = new Rect();
        this.mBarcodeLineRect = new Rect(-1, -1, -1, -1);
        initialize(context);
    }

    private void initialize(Context context) {
        this.mTextBackgroundPaint = new Paint();
        this.mTextBackgroundPaint.setStyle(Style.FILL);
        this.mTextBackgroundPaint.setColor(Color.GREEN);
        this.mTextBackgroundPaint.setAlpha(204);
        TextPaint textPaint = new TextPaint();
        textPaint.setStyle(Style.FILL);
        textPaint.setStrokeWidth(1.0f);
        textPaint.setColor(-1);
        textPaint.setAlpha(204);
        textPaint.setAntiAlias(true);
        textPaint.setTypeface(Typeface.DEFAULT_BOLD);
        textPaint.setTextSize(TEXT_FONT_SIZE * context.getResources().getDisplayMetrics().scaledDensity);
        textPaint.setTextAlign(Align.CENTER);
        this.mTextLayout = new StaticLayout("将二维码/条形码放入框内, 即可自动扫描", textPaint, context.getResources().getDisplayMetrics().heightPixels, Alignment.ALIGN_NORMAL, 1.0f, 0.0f, true);
        this.mLaserLinePaint = new Paint();
        this.mLaserLinePaint.setStyle(Style.FILL);
        this.mLaserLinePaint.setColor(-65536);
        this.mLaserLinePaint.setAlpha(255);
        this.mBarcodeLinePaint = new Paint();
        this.mBarcodeLinePaint.setStyle(Style.FILL);
        this.mBarcodeLinePaint.setColor(-16711936);
        this.mBarcodeLinePaint.setAlpha(255);
        Display display = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
        this.mScreenResolution = new Point(display.getWidth(), display.getHeight());
        this.mScreenDensity = getContext().getResources().getDisplayMetrics().density;
        this.mLaserLinePosition = Math.round((float) (this.mScreenResolution.x / 2));
    }

    protected void onDraw(Canvas canvas) {
        float center = ((float) canvas.getHeight()) / 2.0f;
        canvas.rotate(-90.0f, center, center);
        if (canvas.getClipBounds(this.mCanvasClipRect)) {
            if (this.mCanvasClipRect.height() == canvas.getWidth()) {
                canvas.drawRect(0.0f, (float) (this.mCanvasClipRect.bottom - this.mTextLayout.getHeight()), (float) this.mCanvasClipRect.right, (float) this.mCanvasClipRect.bottom, this.mTextBackgroundPaint);
                canvas.save();
                canvas.translate(this.mCanvasClipRect.exactCenterX(), (float) (this.mCanvasClipRect.bottom - this.mTextLayout.getHeight()));
                this.mTextLayout.draw(canvas);
                canvas.restore();
                int lineWidth = Math.round(LASER_LINE_WIDTH_160_DPI * this.mScreenDensity);
                this.mLaserLineRect.set(this.mCanvasClipRect.width() / 8, this.mLaserLinePosition - (lineWidth / 2), (this.mCanvasClipRect.width() * 7) / 8, this.mLaserLinePosition + ((lineWidth - 1) / 2));
                this.mBarcodeLineRect.top = this.mLaserLineRect.top;
                this.mBarcodeLineRect.bottom = this.mLaserLineRect.bottom;
            }
            canvas.drawRect(this.mLaserLineRect, this.mLaserLinePaint);
            if (this.mBarcodeLineRect.left >= 0 && this.mBarcodeLineRect.right >= 0) {
                canvas.drawRect(this.mBarcodeLineRect, this.mBarcodeLinePaint);
            }
        }
    }

    public float getNormalizedLaserLinePosition() {
        return ((float) this.mLaserLinePosition) / ((float) this.mScreenResolution.x);
    }

    public void setNormalizedBarcodeLocation(float left, float right) {
        this.mBarcodeLineRect.left = Math.round(((float) this.mScreenResolution.y) * left);
        this.mBarcodeLineRect.right = Math.round(((float) this.mScreenResolution.y) * right);
        postInvalidate(this.mLaserLineRect.top, 0, this.mLaserLineRect.bottom, this.mScreenResolution.y - 1);
    }

    public void setNoBarcodeLocation() {
        this.mBarcodeLineRect.left = -1;
        this.mBarcodeLineRect.right = -1;
        postInvalidate(this.mLaserLineRect.top, 0, this.mLaserLineRect.bottom, this.mScreenResolution.y - 1);
    }
}
