package com.supwisdom.dlpay

import com.supwisdom.dlpay.framework.tenant.TenantCacheKeyGen
import io.lettuce.core.ReadFrom
import net.javacrumbs.shedlock.core.LockProvider
import net.javacrumbs.shedlock.provider.redis.spring.RedisLockProvider
import net.javacrumbs.shedlock.spring.annotation.EnableSchedulerLock
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.boot.SpringApplication
import org.springframework.boot.autoconfigure.SpringBootApplication
import org.springframework.boot.autoconfigure.data.redis.RedisProperties
import org.springframework.boot.builder.SpringApplicationBuilder
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.boot.web.servlet.FilterRegistrationBean
import org.springframework.boot.web.servlet.ServletComponentScan
import org.springframework.boot.web.servlet.support.SpringBootServletInitializer
import org.springframework.cache.annotation.EnableCaching
import org.springframework.cache.interceptor.KeyGenerator
import org.springframework.cloud.client.discovery.EnableDiscoveryClient
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.data.redis.connection.RedisConnectionFactory
import org.springframework.data.redis.connection.RedisPassword
import org.springframework.data.redis.connection.RedisStandaloneConfiguration
import org.springframework.data.redis.connection.lettuce.LettuceClientConfiguration
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory
import org.springframework.data.redis.core.RedisTemplate
import org.springframework.data.redis.repository.configuration.EnableRedisRepositories
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer
import org.springframework.data.redis.serializer.StringRedisSerializer
import org.springframework.http.client.SimpleClientHttpRequestFactory
import org.springframework.scheduling.annotation.EnableScheduling
import org.springframework.stereotype.Component
import org.springframework.web.client.RestTemplate
import java.net.InetSocketAddress
import java.net.Proxy


@Configuration
@EnableRedisRepositories
class AppConfig {

    @Autowired
    private lateinit var redis: RedisProperties

    @Bean
    fun redisConnectionFactory(): RedisConnectionFactory {
        val clientConfig = LettuceClientConfiguration.builder()
                .readFrom(ReadFrom.SLAVE_PREFERRED)
                .build()
        val serverConfig = RedisStandaloneConfiguration(redis.host, redis.port)
        if (redis.password.isNotEmpty()) {
            serverConfig.password = RedisPassword.of(redis.password)
        }
        serverConfig.database = redis.database
        return LettuceConnectionFactory(serverConfig, clientConfig)
    }

    @Bean("tenantCacheKey")
    fun createTenantCacheableKey(): KeyGenerator {
        return TenantCacheKeyGen()
    }

    @Bean
    fun lockProvider(connectionFactory: RedisConnectionFactory): LockProvider {
        return RedisLockProvider(connectionFactory, "prod")
    }
}

@Configuration
class JwtAuthFilterRegistration {
    @Autowired
    private lateinit var apiJwtAuthenticationFilter: ApiJwtAuthenticationFilter

    @Autowired
    private lateinit var mobileSecurityFilter: MobileSecurityFilter

    @Bean
    fun registerJwtFilter(): FilterRegistrationBean<ApiJwtAuthenticationFilter> {
        val registration = FilterRegistrationBean<ApiJwtAuthenticationFilter>()
        registration.filter = apiJwtAuthenticationFilter
        registration.addUrlPatterns("/api/*")
        registration.order = 1
        return registration
    }

    @Bean
    fun registerMobileFilter(): FilterRegistrationBean<MobileSecurityFilter> {
        val registrationBean = FilterRegistrationBean<MobileSecurityFilter>()
        registrationBean.filter = mobileSecurityFilter
        registrationBean.addUrlPatterns("/mobile/*")
        registrationBean.order = 2
        return registrationBean
    }
}

@Configuration
class HttpSessionConfig {
    @Bean
    fun sessionRedisTemplate(
            connectionFactory: RedisConnectionFactory): RedisTemplate<Any, Any> {
        val template = RedisTemplate<Any, Any>()
        template.keySerializer = StringRedisSerializer()
        template.hashKeySerializer = StringRedisSerializer()

        template.setDefaultSerializer(GenericJackson2JsonRedisSerializer())
        template.setConnectionFactory(connectionFactory)
        return template
    }
}

@Configuration
class RestTemplateConfig {
    @Component
    @ConfigurationProperties("resttemplate.proxy")
    class RestTemplateProxyProperties {
        @Value("\${type:}")
        lateinit var type: String
        @Value("\${host:}")
        lateinit var host: String
        @Value("\${port:0}")
        var port: Int = 0
    }

    @Bean
    fun simpleClientHttpRequestFactory(proxyProperties: RestTemplateProxyProperties):
            SimpleClientHttpRequestFactory {
        val factory = SimpleClientHttpRequestFactory()
        factory.setConnectTimeout(15000)
        factory.setReadTimeout(5000)
        if (proxyProperties.type.isNotEmpty()) {
            val proxyType = when (proxyProperties.type) {
                "http" -> Proxy.Type.HTTP
                "socks5" -> Proxy.Type.SOCKS
                else -> Proxy.Type.DIRECT
            }
            if (proxyType != Proxy.Type.DIRECT) {
                factory.setProxy(Proxy(proxyType,
                        InetSocketAddress(proxyProperties.host, proxyProperties.port)))
            }
        }
        return factory
    }

    @Bean
    fun restTemplate(factory: SimpleClientHttpRequestFactory): RestTemplate {
        return RestTemplate(factory)
    }
}

@EnableSchedulerLock(defaultLockAtMostFor = "PT30M")
@SpringBootApplication
@EnableDiscoveryClient
@EnableScheduling
@EnableCaching
@ServletComponentScan
class PayApiApplication : SpringBootServletInitializer() {

    override fun configure(builder: SpringApplicationBuilder): SpringApplicationBuilder {
        return builder.sources(PayApiApplication::class.java)
    }
}

fun main(args: Array<String>) {
    SpringApplication.run(PayApiApplication::class.java, * args)
}