package com.supwisdom.dlpay.agent.service.impl

import com.google.gson.Gson
import com.sun.jersey.api.client.Client
import com.sun.jersey.api.client.ClientResponse
import com.sun.jersey.core.util.MultivaluedMapImpl
import com.supwisdom.dlpay.agent.citizencard.DlpayResp
import com.supwisdom.dlpay.api.service.CardService
import com.supwisdom.dlpay.api.service.SourceTypeService
import com.supwisdom.dlpay.api.service.UserService
import com.supwisdom.dlpay.api.types.IDTypes
import com.supwisdom.dlpay.agent.service.CitizencardPayService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.agent.citizencard.YnrccUtil
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import javax.ws.rs.core.MediaType

@Service
class CitizencardPayServiceImpl : CitizencardPayService {
    @Autowired
    lateinit var sourceTypeService: SourceTypeService
    @Autowired
    lateinit var systemUtilService: SystemUtilService
    @Autowired
    lateinit var cardService: CardService
    @Autowired
    lateinit var userService: UserService

    private val logger = KotlinLogging.logger { }

    private fun checkCitizencardConfig(config: Map<String, String?>, resp: DlpayResp): Boolean {
        val agentUrl = config[YnrccUtil.YNRCC_ANGENT_URL]
        val signKey = config[YnrccUtil.YNRCC_SIGNKEY]
        if (agentUrl.isNullOrEmpty()) {
            resp.code = YnrccUtil.PARAM_CONFIG_ERROR
            resp.message = "系统参数未配置[农商行前置地址前缀]"
            logger.error(resp.message)
            return false
        }
        if (signKey.isNullOrEmpty()) {
            resp.code = YnrccUtil.PARAM_CONFIG_ERROR
            resp.message = "系统参数未配置[农商行前置签名秘钥]"
            logger.error(resp.message)
            return false
        }
        return true
    }

    override fun bindCard(bankcardno: String, username: String, idtype: String, idno: String, phone: String): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }
        val idType = IDTypes.findByValue(idtype)
        if (idType < 0) {
            resp.code = YnrccUtil.PARAM_VALUE_ERROR
            resp.message = "证件类型未识别[$idtype]"
            logger.error(resp.message)
            return resp
        }
        val systime = systemUtilService.sysdatetime
        val refno = systemUtilService.refno
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_BIND_TRANSCODE
        params["transdate"] = systime.hostdate
        params["transtime"] = systime.hosttime
        params["categorie"] = YnrccUtil.DLPAY_CATEGORIE
        params["refno"] = refno
        params["bankcardno"] = bankcardno
        params["username"] = username
        params["idtype"] = idType.toString()
        params["idno"] = idno
        params["phone"] = phone
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/bindcard"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        val client = Client.create()
        client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
        client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
        val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
        return if (200 == respClient.status) {
            val jsonStr = respClient.getEntity(String::class.java)
            logger.error("refno=[$refno], url=[$url], return=[$jsonStr]")
            resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
            resp
        } else {
            resp.code = "99"
            resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
            logger.error(resp.message)
            resp
        }
    }

    override fun signCard(bankcardno: String, username: String, idtype: String, idno: String, phone: String, transtype: String,captcha:String): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }
        val idType = IDTypes.findByValue(idtype)
        if (idType < 0) {
            resp.code = YnrccUtil.PARAM_VALUE_ERROR
            resp.message = "证件类型未识别[$idtype]"
            logger.error(resp.message)
            return resp
        }
        if (YnrccUtil.TRANSTYPE_SIGNCARD != transtype && YnrccUtil.TRANSTYPE_UNSIGNCARD != transtype) {
            resp.code = YnrccUtil.PARAM_VALUE_ERROR
            resp.message = "签约类型错误[$transtype]"
            logger.error(resp.message)
            return resp
        }

        val systime = systemUtilService.sysdatetime
        val refno = systemUtilService.refno
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_SIGN_TRANSCODE
        params["transdate"] = systime.hostdate
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["categorie"] = YnrccUtil.DLPAY_CATEGORIE
        params["transtype"] = transtype
        params["bankcardno"] = bankcardno
        params["username"] = username
        params["idtype"] = idType.toString()
        params["idno"] = idno
        params["phone"] = phone
        params["captcha"] = captcha
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/signcard"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        val client = Client.create()
        client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
        client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
        val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
        return if (200 == respClient.status) {
            val jsonStr = respClient.getEntity(String::class.java)
            logger.error("refno=[$refno], url=[$url], return=[$jsonStr]")
            resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
            resp
        } else {
            resp.code = "99"
            resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
            logger.error(resp.message)
            resp
        }
    }

    override fun cardPay(shopaccno: String, userid: String, accdate: String, amount: Int, scenario: String, refno: String): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getConsumePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, shopaccno, false, false)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }
        val merchantBankcardno = config[YnrccUtil.YNRCC_MERCHANT_BANKCARDNO]
        val merchantBankaccname = config[YnrccUtil.YNRCC_MERCHANT_BANKACCNAME]
        if (StringUtil.isEmpty(merchantBankcardno) || StringUtil.isEmpty(merchantBankaccname)) {
            resp.code = YnrccUtil.PARAM_CONFIG_ERROR
            resp.message = "系统参数未配置[商户收款银行账号]"
            logger.error(resp.message)
            return resp
        }
        val person = userService.findOnePersonByUserid(userid)
        val idType = IDTypes.findByValue(person.idtype)
        if (idType < 0) {
            resp.code = YnrccUtil.PARAM_VALUE_ERROR
            resp.message = "证件类型未识别[${person.idtype}]"
            logger.error(resp.message)
            return resp
        }
        val userBankcard = cardService.getBankcardByUserid(person.userid)
        if (null == userBankcard) {
            resp.code = "99"
            resp.message = "用户[${person.userid}]未绑定银行卡"
            logger.error(resp.message)
            return resp
        } else if (TradeDict.STATUS_NORMAL != userBankcard.status) {
            resp.code = "99"
            resp.message = "用户[${person.userid}]绑定银行卡状态异常"
            logger.error(resp.message)
            return resp
        } else if (!userBankcard.signed) {
            resp.code = "99"
            resp.message = "用户[${person.userid}]绑定银行卡未签约"
            logger.error(resp.message)
            return resp
        }
        val bankcardno = userBankcard.cardno

        val systime = systemUtilService.sysdatetime
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_PAY_TRANSCODE
        params["transdate"] = accdate
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["categorie"] = YnrccUtil.DLPAY_CATEGORIE
        params["bankcardno"] = bankcardno
        params["username"] = person.name
        params["idtype"] = idType.toString()
        params["idno"] = person.idno
        params["merchant_bankcardno"] = merchantBankcardno!!
        params["merchant_bankaccname"] = merchantBankaccname!!
        params["amount"] = amount.toString()
        params["scenario"] = scenario
        params["description"] = "市民卡代扣消费"
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/cardpay"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        try {
            val client = Client.create()
            client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
            client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
            val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
            return if (200 == respClient.status) {
                val jsonStr = respClient.getEntity(String::class.java)
                logger.error("refno=[$refno], url=[$url], return=[$jsonStr]")
                resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
                resp
            } else {
                resp.code = "99"
                resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
                logger.error(resp.message)
                resp
            }
        } catch (e: Exception) {
            e.printStackTrace()
            resp.code = YnrccUtil.CODE_EXCEPTION
            resp.message = "请求前置抛出异常"
            logger.error(resp.message)
            return resp
        }
    }

    override fun cardPayRefund(refno: String, accdate: String, orignRefno: String, amount: Int): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }

        val systime = systemUtilService.sysdatetime
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_PAYREFUND_TRANSCODE
        params["transdate"] = accdate //我们的记账日期
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["refundRefno"] = orignRefno
        params["amount"] = amount.toString()
        params["description"] = "市民卡代扣消费退款"
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/cardpayrefund"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        try {
            val client = Client.create()
            client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
            client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
            val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
            return if (200 == respClient.status) {
                val jsonStr = respClient.getEntity(String::class.java)
                logger.error("refno=[$refno],refundRefno=[$orignRefno], url=[$url], return=[$jsonStr]")
                resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
                resp
            } else {
                resp.code = "99"
                resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
                logger.error(resp.message)
                resp
            }
        } catch (e: Exception) {
            e.printStackTrace()
            resp.code = YnrccUtil.CODE_EXCEPTION
            resp.message = "请求前置抛出异常"
            logger.error(resp.message)
            return resp
        }
    }

    override fun queryResult(orignRefno: String): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }

        val systime = systemUtilService.sysdatetime
        val refno = systemUtilService.refno
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_QUERYRESULT_TRANSCODE
        params["transdate"] = systime.hostdate
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["orignRefno"] = orignRefno
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/queryresult"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        try {
            val client = Client.create()
            client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
            client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
            val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
            return if (200 == respClient.status) {
                val jsonStr = respClient.getEntity(String::class.java)
                logger.error("refno=[$orignRefno], url=[$url], return=[$jsonStr]")
                resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
                resp
            } else {
                resp.code = "99"
                resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
                logger.error(resp.message)
                resp
            }
        } catch (e: Exception) {
            e.printStackTrace()
            resp.code = YnrccUtil.CODE_EXCEPTION
            resp.message = "请求前置抛出异常"
            logger.error(resp.message)
            return resp
        }
    }

    override fun getChkfilename(checkdate: String, merchantBankcardno: String?): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }
        val systime = systemUtilService.sysdatetime
        val refno = systemUtilService.refno
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_CHKFILE_TRANSCODE
        params["transdate"] = systime.hostdate
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["checkdate"] = checkdate
        params["merchant_bankcardno"] = merchantBankcardno ?: ""
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/checkdtl"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        val client = Client.create()
        client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
        client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
        val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
        return if (200 == respClient.status) {
            val jsonStr = respClient.getEntity(String::class.java)
            logger.error("refno=[$refno],chkdate=[$checkdate], url=[$url], return=[$jsonStr]")
            resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
            resp
        } else {
            resp.code = "99"
            resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
            logger.error(resp.message)
            resp
        }
    }

    override fun makeSureCheckResult(trxdate: String, transcnt: Int, transamt: Long): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }

        val systime = systemUtilService.sysdatetime
        val refno = systemUtilService.refno
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_CHKMAKESURE_TRANSCODE
        params["transdate"] = systime.hostdate
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["stltrxdate"] = trxdate
        params["stlamt"] = transamt.toString()
        params["jnlcount"] = transcnt.toString()
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/checkmakesure"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        try {
            val client = Client.create()
            client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
            client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
            val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
            return if (200 == respClient.status) {
                val jsonStr = respClient.getEntity(String::class.java)
                logger.error("refno=[$refno],trxdate=[$trxdate], url=[$url], return=[$jsonStr]")
                resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
                resp
            } else {
                resp.code = "99"
                resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
                logger.error(resp.message)
                resp
            }
        } catch (e: Exception) {
            e.printStackTrace()
            resp.code = YnrccUtil.CODE_EXCEPTION
            resp.message = "请求前置抛出异常"
            logger.error(resp.message)
            return resp
        }
    }

    override fun bankCardLoss(bankcardno: String, username: String, idtype: String, idno: String): DlpayResp {
        var resp = DlpayResp()
        val config = sourceTypeService.getChargePaytypeConfig(TradeDict.PAYTYPE_CITIZEN_CARD, true)
        if (!checkCitizencardConfig(config, resp)) {
            return resp
        }
        val idType = IDTypes.findByValue(idtype)
        if (idType < 0) {
            resp.code = YnrccUtil.PARAM_VALUE_ERROR
            resp.message = "证件类型未识别[$idtype]"
            logger.error(resp.message)
            return resp
        }

        val systime = systemUtilService.sysdatetime
        val refno = systemUtilService.refno
        val params = hashMapOf<String, String>()
        params["transcode"] = YnrccUtil.BANKCARD_BANKCARDLOSS_TRANSCODE
        params["transdate"] = systime.hostdate
        params["transtime"] = systime.hosttime
        params["refno"] = refno
        params["bankcardno"] = bankcardno
        params["username"] = username
        params["idtype"] = idType.toString()
        params["idno"] = idno
        params["sign_type"] = "MD5"
        val sign = MD5.encodeByMD5(StringUtil.createLinkString(StringUtil.paraFilter(params)) + config[YnrccUtil.YNRCC_SIGNKEY]!!.trim())
        params["sign"] = sign

        val postData = MultivaluedMapImpl()
        params.forEach { (t, u) -> postData.add(t, u) }

        val url = config[YnrccUtil.YNRCC_ANGENT_URL]!!.trim() + "/bankcardloss"
        logger.error("url=[$url], params=[" + Gson().toJson(params) + "]")
        try {
            val client = Client.create()
            client.setConnectTimeout(YnrccUtil.AGENT_CONNECT_TIMEOUT * 1000)
            client.setReadTimeout(YnrccUtil.AGENT_READ_TIMEOUT * 1000)
            val respClient = client.resource(url).type(MediaType.APPLICATION_FORM_URLENCODED_TYPE).accept(MediaType.APPLICATION_JSON).post(ClientResponse::class.java, postData)
            return if (200 == respClient.status) {
                val jsonStr = respClient.getEntity(String::class.java)
                logger.error("refno=[$refno],bankcardno=[$bankcardno], url=[$url], return=[$jsonStr]")
                resp = Gson().fromJson(jsonStr, DlpayResp::class.java)
                resp
            } else {
                resp.code = "99"
                resp.message = "请求前置返回失败[httpStatus=$respClient.status]"
                logger.error(resp.message)
                resp
            }
        } catch (e: Exception) {
            e.printStackTrace()
            resp.code = YnrccUtil.CODE_EXCEPTION
            resp.message = "请求前置抛出异常"
            logger.error(resp.message)
            return resp
        }
    }
}