package com.supwisdom.dlpay.mobile

import com.fasterxml.jackson.databind.ObjectMapper
import com.supwisdom.dlpay.api.bean.JsonResult
import com.supwisdom.dlpay.api.service.UserService
import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.core.JwtTokenUtil
import com.supwisdom.dlpay.framework.domain.JwtRedis
import com.supwisdom.dlpay.framework.redisrepo.ApiJwtRepository
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.mobile.dao.MobileUserDao
import com.supwisdom.dlpay.mobile.domain.TBMobileUser
import com.supwisdom.dlpay.mobile.exception.UserLoginFailException
import com.supwisdom.dlpay.mobile.service.MobileApiService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.HttpStatus
import org.springframework.security.authentication.BadCredentialsException
import org.springframework.security.authentication.LockedException
import org.springframework.security.core.Authentication
import org.springframework.security.core.AuthenticationException
import org.springframework.security.web.authentication.SimpleUrlAuthenticationFailureHandler
import org.springframework.security.web.authentication.SimpleUrlAuthenticationSuccessHandler
import org.springframework.stereotype.Component
import java.io.IOException
import javax.servlet.ServletException
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@Component("authLoginSuccessHandler")
class AuthLoginSuccessHandler : SimpleUrlAuthenticationSuccessHandler() {
    @Autowired
    lateinit var mobileApiService: MobileApiService
    @Autowired
    lateinit var objectMapper: ObjectMapper
    @Autowired
    lateinit var jwtConfig: JwtConfig
    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository
    @Autowired
    lateinit var systemUtilService: SystemUtilService
    @Autowired
    lateinit var userService: UserService

    override fun onAuthenticationSuccess(request: HttpServletRequest, response: HttpServletResponse, authentication: Authentication) {
        val platform = request.getParameter("platform")
        logger.error(platform)
        val temp = authentication.principal as TBMobileUser
        val user = mobileApiService.findUserById(temp.uid)
        val exp = systemUtilService.getSysparaValueAsInt(SysparaUtil.MOBILE_LOGIN_EXPIRE_IN_SECONDS,60*60*24*3)
        jwtConfig.expiration = exp.toLong()
        if (user != null) {
            //TODO 从数据取jwtConfig.expiration
            val token = JwtTokenUtil(jwtConfig).generateToken(
                    mapOf("uid" to user.uid, "issuer" to "payapi",
                            "audience" to user.loginid,
                            Constants.JWT_CLAIM_TENANTID to "mobile",
                            Constants.JWT_CLAIM_AUTHORITIES to temp.authorities))
            val jwt = JwtRedis().apply {
                jti = token.jti
                uid = user.loginid
                status = TradeDict.JWT_STATUS_NORMAL
                expiration = token.expiration.valueInMillis
            }.apply {
                //删除之前的token
                if (!user.jti.isNullOrEmpty()) {
                    apiJwtRepository.deleteById(user.jti!!)
                }
                apiJwtRepository.save(this)
            }
            if (user.loginpwderror > 0) {
                user.loginpwderror = 0
                user.loginpwderrortime = null
            }
            user.lastlogin = DateUtil.getNow()
            user.lastloginplatform = platform
            user.jti = jwt.jti
            mobileApiService.saveUser(user)
            var payseted = false
            if(!user.paypwd.isNullOrEmpty()){
                payseted = true
            }
            var name = ""
            var signed=""
            if (!user.userid.isNullOrEmpty()) {
                val person = userService.findOnePersonByUserid(user.userid!!)
                var card = mobileApiService.findCardByUserid(user.userid!!)
                name = person.name
                if(card!=null&&card.signed){
                    signed = TradeDict.STATUS_YES
                }
            }
            response.status = HttpStatus.OK.value()
            response.contentType = "application/json;charset=UTF-8"
            response.writer.write(objectMapper.writeValueAsString(JsonResult.ok()
                    .put("token", token.jwtToken)
                    ?.put("expire",token.expiration.valueInMillis)
                    ?.put("now",System.currentTimeMillis())
                    ?.put("tenantid", "mobile")
                    ?.put("name", name)
                    ?.put("uid", user.uid)
                    ?.put("phone", StringUtil.phoneReplace(user.phone))
                    ?.put("paypwdset",payseted)
                    ?.put("signed", signed)
                    ?.put("userid",if(user.userid.isNullOrEmpty()) "" else user.userid)))
        } else {
            throw UserLoginFailException("登录错误")
        }
    }
}


@Component("authLoginFailHandler")
class AuthLoginFailHandler : SimpleUrlAuthenticationFailureHandler() {
    @Autowired
    lateinit var mobileUserDao: MobileUserDao

    @Autowired
    lateinit var objectMapper: ObjectMapper

    @Throws(IOException::class, ServletException::class)
    override fun onAuthenticationFailure(request: HttpServletRequest,
                                         response: HttpServletResponse, exception: AuthenticationException) {
        logger.error("登录失败:" + exception.message + "|" + exception.javaClass)
        val errmsg = when (exception) {
            is BadCredentialsException -> "手机号或密码错误"
            is LockedException -> "账户被锁定"
            else -> exception.message!!
        }
        val temp = request.getParameter("username")
        if(!temp.isNullOrEmpty()) {
            mobileUserDao.findByLoginid(temp)?.let {
                if (it.loginpwderror == 0) {
                    it.loginpwderror = 0
                    it.loginpwderrortime = System.currentTimeMillis()
                }
                it.loginpwderror += 1
                mobileUserDao.save(it)
            }
        }
        response.status = HttpStatus.OK.value()
        response.contentType = "application/json;charset=UTF-8"
        response.writer.write(objectMapper.writeValueAsString(JsonResult.error(errmsg)))
    }
}
