package com.supwisdom.dlpay

import com.supwisdom.dlpay.framework.core.PasswordBCryptConfig
import com.supwisdom.dlpay.system.service.ParamService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.data.redis.connection.RedisConnectionFactory
import org.springframework.security.authentication.AuthenticationManager
import org.springframework.security.core.GrantedAuthority
import org.springframework.security.core.authority.SimpleGrantedAuthority
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder
import org.springframework.security.oauth2.config.annotation.configurers.ClientDetailsServiceConfigurer
import org.springframework.security.oauth2.config.annotation.web.configuration.AuthorizationServerConfigurerAdapter
import org.springframework.security.oauth2.config.annotation.web.configuration.EnableAuthorizationServer
import org.springframework.security.oauth2.config.annotation.web.configurers.AuthorizationServerEndpointsConfigurer
import org.springframework.security.oauth2.config.annotation.web.configurers.AuthorizationServerSecurityConfigurer
import org.springframework.security.oauth2.provider.ClientDetails
import org.springframework.security.oauth2.provider.ClientDetailsService
import org.springframework.security.oauth2.provider.client.BaseClientDetails
import org.springframework.security.oauth2.provider.token.store.redis.RedisTokenStore
import java.security.SecureRandom
import java.util.*


class OAuthDetailService : ClientDetailsService {
    @Autowired
    private lateinit var paramService: ParamService;
    override fun loadClientByClientId(clientId: String?): ClientDetails {
        val details = BaseClientDetails()
        if (clientId.isNullOrEmpty()) {
            return details
        }
        details.clientId = clientId
        paramService.getApiClientByAppid(clientId)?.let {
            details.setAuthorizedGrantTypes(Arrays.asList("password","authorization_code","refresh_token"))
            details.setScope(Arrays.asList("read"))
            details.setResourceIds(Arrays.asList("oauth2-resource"))
            val authorities = HashSet<GrantedAuthority>()
            authorities.add(SimpleGrantedAuthority("ROLE_THIRD_ADMIN"))
            details.authorities = authorities
            details.setAutoApproveScopes(Arrays.asList("true"))
            details.clientSecret = it.bcryptSecret
            details.accessTokenValiditySeconds = 3600
            details.refreshTokenValiditySeconds=43200
            if(!it.thirdurl.isNullOrEmpty()){
                val redir = HashSet<String>()
                when {
                    it.thirdurl.contains(",") -> redir.addAll(it.thirdurl.split(","))
                    it.thirdurl.contains(";") -> redir.addAll(it.thirdurl.split(";"))
                    else -> redir.add(it.thirdurl)
                }
                details.registeredRedirectUri = redir
            }

        }
        return details
    }
}


@Configuration
class OAuth2Config {

    @Configuration
    @EnableAuthorizationServer

    class AuthorizationServerConfigure : AuthorizationServerConfigurerAdapter() {

        @Autowired
        private lateinit var redisConnectionFactory: RedisConnectionFactory

        @Autowired
        private lateinit var authenticationManager: AuthenticationManager

        @Autowired
        lateinit var passwordBCryptConfig: PasswordBCryptConfig

        override fun configure(security: AuthorizationServerSecurityConfigurer?) {
            security?.allowFormAuthenticationForClients()
            security?.passwordEncoder(pwdEncoder())
        }
        @Bean
        fun pwdEncoder(): BCryptPasswordEncoder {
            return if (passwordBCryptConfig.seed.isBlank()) {
                BCryptPasswordEncoder()
            } else {
                BCryptPasswordEncoder(passwordBCryptConfig.length,
                        SecureRandom(passwordBCryptConfig.seed.toByteArray()))
            }
        }

        override fun configure(clients: ClientDetailsServiceConfigurer?) {
            clients?.withClientDetails(OAuthDetailService())
        }

        override fun configure(endpoints: AuthorizationServerEndpointsConfigurer?) {
            endpoints?.tokenStore(RedisTokenStore(redisConnectionFactory))
                    ?.authenticationManager(authenticationManager)
        }
    }
}