package com.supwisdom.dlpay.portal

import com.fasterxml.jackson.databind.ObjectMapper
import com.supwisdom.dlpay.api.bean.JsonResult
import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.core.JwtTokenUtil
import com.supwisdom.dlpay.framework.domain.JwtRedis
import com.supwisdom.dlpay.framework.domain.TOperator
import com.supwisdom.dlpay.framework.redisrepo.ApiJwtRepository
import com.supwisdom.dlpay.framework.service.OperatorDetailService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.Constants
import com.supwisdom.dlpay.framework.util.SysparaUtil
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.mobile.exception.UserLoginFailException
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.HttpStatus
import org.springframework.security.authentication.BadCredentialsException
import org.springframework.security.authentication.LockedException
import org.springframework.security.core.Authentication
import org.springframework.security.core.AuthenticationException
import org.springframework.security.web.authentication.SimpleUrlAuthenticationFailureHandler
import org.springframework.security.web.authentication.SimpleUrlAuthenticationSuccessHandler
import org.springframework.stereotype.Component
import java.io.IOException
import javax.servlet.ServletException
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@Component("operLoginSuccessHandler")
class OperLoginSuccessHandler : SimpleUrlAuthenticationSuccessHandler() {
    @Autowired
    lateinit var operatorDetailService: OperatorDetailService
    @Autowired
    lateinit var objectMapper: ObjectMapper
    @Autowired
    lateinit var jwtConfig: JwtConfig
    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository
    @Autowired
    lateinit var systemUtilService: SystemUtilService

    override fun onAuthenticationSuccess(request: HttpServletRequest, response: HttpServletResponse, authentication: Authentication) {
        val temp = authentication.principal as TOperator
        val operator = operatorDetailService.findByOperid(temp.operid)
        val exp = systemUtilService.getSysparaValueAsInt(SysparaUtil.MOBILE_LOGIN_EXPIRE_IN_SECONDS,60*60*24*3)
        jwtConfig.expiration = exp.toLong()
        if (operator != null) {
            //TODO 从数据取jwtConfig.expiration
            val token = JwtTokenUtil(jwtConfig).generateToken(
                    mapOf("uid" to operator.operid, "issuer" to "portal",
                            "audience" to operator.opername,
                            Constants.JWT_CLAIM_TENANTID to "portal",
                            Constants.JWT_CLAIM_AUTHORITIES to temp.authorities))
            val jwt = JwtRedis().apply {
                jti = token.jti
                uid = operator.opername
                status = TradeDict.JWT_STATUS_NORMAL
                expiration = token.expiration.valueInMillis
            }.apply {
                //删除之前的token
                if (!operator.jti.isNullOrEmpty()) {
                    apiJwtRepository.deleteById(operator.jti!!)
                }
                apiJwtRepository.save(this)
            }
            operator.jti = jwt.jti
            operatorDetailService.saveOper(operator)
            response.status = HttpStatus.OK.value()
            response.contentType = "application/json;charset=UTF-8"
            response.writer.write(objectMapper.writeValueAsString(JsonResult.ok()
                    .put("token", token.jwtToken)
                    ?.put("expire",token.expiration.valueInMillis)
                    ?.put("now",System.currentTimeMillis())
                    ?.put("tenantid", "mobile")
                    ?.put("uid", operator.operid)))
        } else {
            throw UserLoginFailException("登录错误")
        }
    }
}


@Component("operLoginFailHandler")
class OperLoginFailHandler : SimpleUrlAuthenticationFailureHandler() {
    @Autowired
    lateinit var objectMapper: ObjectMapper

    @Throws(IOException::class, ServletException::class)
    override fun onAuthenticationFailure(request: HttpServletRequest,
                                         response: HttpServletResponse, exception: AuthenticationException) {
        logger.error("登录失败:" + exception.message + "|" + exception.javaClass)
        val errmsg = when (exception) {
            is BadCredentialsException -> "账号或密码错误"
            is LockedException -> "账户被锁定"
            else -> exception.message!!
        }
        response.status = HttpStatus.OK.value()
        response.contentType = "application/json;charset=UTF-8"
        response.writer.write(objectMapper.writeValueAsString(JsonResult.error(errmsg)))
    }
}
