package com.supwisdom.dlpay.portal

import com.supwisdom.dlpay.api.bean.JsonResult
import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.core.JwtTokenUtil
import com.supwisdom.dlpay.framework.redisrepo.ApiJwtRepository
import com.supwisdom.dlpay.framework.service.OperatorDetailService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.StringUtil
import com.supwisdom.dlpay.portal.bean.*
import com.supwisdom.dlpay.portal.domain.TBArticle
import com.supwisdom.dlpay.portal.domain.TBColumn
import com.supwisdom.dlpay.portal.domain.TBMsgTemplate
import com.supwisdom.dlpay.portal.domain.TBReply
import com.supwisdom.dlpay.portal.service.ArticleService
import com.supwisdom.dlpay.portal.service.ColumnService
import com.supwisdom.dlpay.portal.service.FeedbackService
import com.supwisdom.dlpay.portal.service.MsgService
import com.supwisdom.dlpay.portal.util.PortalConstant
import mu.KotlinLogging
import org.jose4j.jwt.ReservedClaimNames
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.web.bind.annotation.*
import org.springframework.web.multipart.MultipartHttpServletRequest
import java.util.regex.Pattern

@RestController
@RequestMapping("/portalapi")
class PortalApi {
    @Autowired
    lateinit var operatorDetailService: OperatorDetailService
    @Autowired
    lateinit var feedbackService: FeedbackService
    @Autowired
    lateinit var jwtConfig: JwtConfig
    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository
    @Autowired
    lateinit var systemUtilService: SystemUtilService
    @Autowired
    lateinit var msgService: MsgService
    @Autowired
    lateinit var articleService: ArticleService
    @Autowired
    lateinit var columnService: ColumnService
    val logger = KotlinLogging.logger { }

    @RequestMapping("/test")
    fun test(): JsonResult {
        return JsonResult.ok("测试")
    }

    @RequestMapping("user/logout")
    fun logout(@RequestHeader("Authorization") auth: String?): ResponseEntity<Any> {
        if (auth == null) {
            return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build()
        }
        val jwt = auth.substring(jwtConfig.tokenHeader.length)
        val claims = JwtTokenUtil(jwtConfig).verifyToken(jwt)
        SecurityContextHolder.clearContext()
        apiJwtRepository.deleteById(claims[ReservedClaimNames.JWT_ID].toString())
        return ResponseEntity.ok().body(JsonResult.ok())
    }

    @RequestMapping("/user/info")
    fun getUserInfo(): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            val data = HashMap<String, String>()
            val url = systemUtilService.getBusinessValue(PortalConstant.SYSPARA_IMAGESERVER_URL)
            data["name"] = oper.opername
            data["roles"] = "admin"
            data["url"] = url
            JsonResult.ok().put("data", data)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询用户信息异常")
        }

    }

    @RequestMapping("/user/resource")
    fun getUserResource(): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            val resource = operatorDetailService.getResByRoleId(oper.roleid)
            JsonResult.ok().put("resource", resource)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询功能列表异常")
        }

    }

    @RequestMapping("/feedback/list")
    fun getFeedbackList(bean: FeedbackSearchBean): JsonResult? {
        return try {
            val page = feedbackService.getFeedbackList(bean)
            if (page.list == null || page.list.size == 0) {
                return JsonResult.ok().put("msg", "无数据")
            }
            return JsonResult.ok().put("page", page)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询用户留言列表异常")
        }
    }

    @RequestMapping("/feedback/reply/{fbid}")
    fun getFeedbackReply(@PathVariable("fbid") fbid: String): JsonResult? {
        return try {
            val list = feedbackService.getReplyListByFbId(fbid)
            if (list.isEmpty()) {
                return JsonResult.ok().put("msg", "无数据")
            }
            return JsonResult.ok().put("list", list)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询用户留言回复异常")
        }
    }

    @RequestMapping(value = ["/feedback/reply/save"], method = [RequestMethod.POST])
    fun saveFeedbackReply(@RequestBody reply: TBReply): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            reply.operid = oper.operid
            val msg = feedbackService.saveReplyListByFbId(reply)
            return if (StringUtil.isEmpty(msg)) {
                JsonResult.ok()
            } else {
                JsonResult.error(msg)
            }
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("保存留言回复异常")
        }
    }

    @RequestMapping("/template/list")
    fun getMsgTemplateList(bean: MsgTemplateSearchBean): JsonResult? {
        return try {
            val page = msgService.getMsgTemplateList(bean)
            if (page.list == null || page.list.size == 0) {
                return JsonResult.ok().put("msg", "无数据")
            }
            return JsonResult.ok().put("page", page)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询消息模板列表异常")
        }
    }

    @RequestMapping(value = ["/template/save"], method = [RequestMethod.POST])
    fun saveMsgTemplate(@RequestBody template: TBMsgTemplate): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            template.operid = oper.operid
            msgService.saveMsgTemplate(template)
            return JsonResult.ok()
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("保存消息推送模板异常")
        }
    }

    @RequestMapping(value = ["/template/sendmsg"], method = [RequestMethod.POST])
    fun sendMsg(@RequestBody bean: SendMsgBean): JsonResult? {
        return try {
            msgService.sendMsg(bean)
            return JsonResult.ok()
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error(e.message)
        }
    }

    @RequestMapping(value = ["/article/uploadpic"], method = [RequestMethod.POST])
    fun uploadPic(request: MultipartHttpServletRequest): JsonResult? {
        val map = articleService.uploadPic(request)
        return JsonResult.ok()
    }

    @RequestMapping("/column/all")
    fun getAllColumn(): JsonResult? {
        return try {
            val list = columnService.getAllColumnList()
            if (list == null || list.isEmpty()) {
                return JsonResult.ok().put("msg", "无数据")
            }
            return JsonResult.ok().put("list", list)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询所有栏目列表异常")
        }
    }

    @RequestMapping("/column/list")
    fun getColumnList(bean: ColumnSearchBean): JsonResult? {
        return try {
            val page = columnService.getColumnList(bean)
            if (page.list == null || page.list.size == 0) {
                return JsonResult.ok().put("msg", "无数据")
            }
            return JsonResult.ok().put("page", page)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询栏目列表异常")
        }
    }

    @RequestMapping(value = ["/column/save"], method = [RequestMethod.POST])
    fun saveColumn(@RequestBody column: TBColumn): JsonResult? {
        return try {
            val matches = Pattern.matches("^[a-zA-Z0-9]{1,30}$", column.code)
            if (!matches) {
                return JsonResult.error("请输入合法的栏目CODE！")
            }
            val msg = columnService.saveColumn(column)
            return if (StringUtil.isEmpty(msg)) {
                JsonResult.ok()
            } else {
                JsonResult.error(msg)
            }
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("保存栏目异常")
        }
    }

    @RequestMapping(value = ["/article/save"], method = [RequestMethod.POST])
    fun saveArticle(@RequestBody article: TBArticle): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            article.operid = oper.operid
            articleService.saveArticle(article)
            JsonResult.ok()
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("保存文章异常")
        }
    }

    @RequestMapping(value = ["/article/release"], method = [RequestMethod.POST])
    fun releaseArticle(@RequestBody article: TBArticle): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            article.operid = oper.operid
            articleService.releaseArticle(article)
            JsonResult.ok()
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("发布文章异常")
        }
    }

    @RequestMapping(value = ["/article/review"], method = [RequestMethod.POST])
    fun reviewArticle(@RequestBody article: TBArticle): JsonResult? {
        return try {
            val p = SecurityContextHolder.getContext().authentication
            val oper = operatorDetailService.findByOperid(p.name)
            article.operid = oper.operid
            articleService.reviewArticle(article,oper.opername)
            JsonResult.ok()
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("提交文章审核异常")
        }
    }

    @RequestMapping("/article/list")
    fun getArticleList(bean: ArticleSearchBean): JsonResult? {
        return try {
            val page = articleService.getArticleList(bean)
            if (page.list == null || page.list.size == 0) {
                return JsonResult.ok().put("msg", "无数据")
            }
            return JsonResult.ok().put("page", page)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("查询文章列表异常")
        }
    }

    @RequestMapping(value = ["/article/delete/{articleno}"], method = [RequestMethod.POST])
    fun deleteArticle(@PathVariable(value = "articleno") articleno: String): JsonResult? {
        return try {
            articleService.deleteArticle(articleno)
            JsonResult.ok()
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("删除文章异常")
        }
    }

    @RequestMapping(value = ["/article/get/{articleno}"], method = [RequestMethod.GET])
    fun getArticle(@PathVariable(value = "articleno") articleno: String): JsonResult? {
        return try {
            val article = articleService.getArticle(articleno)
            JsonResult.ok().put("article", article)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("获取文章详情文章异常")
        }
    }

    @RequestMapping(value = ["/article/switchdisplay/{articleno}"], method = [RequestMethod.POST])
    fun switchDisplay(@PathVariable(value = "articleno") articleno: String,
                      @RequestParam(value = "value") value: String): JsonResult? {
        return try {
            val result = articleService.switchDisplay(articleno, value)
            JsonResult.ok().put("result", result)
        } catch (e: Exception) {
            logger.error { e.message }
            JsonResult.error("切换文章是否显示异常")
        }
    }
}