package com.supwisdom.dlpay

import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.core.JwtTokenUtil
import com.supwisdom.dlpay.framework.core.PasswordBCryptConfig
import com.supwisdom.dlpay.framework.redisrepo.ApiJwtRepository
import com.supwisdom.dlpay.framework.service.OperatorDetailService
import com.supwisdom.dlpay.framework.tenant.TenantContext
import com.supwisdom.dlpay.framework.util.Constants
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.mobile.AuthLoginFailHandler
import com.supwisdom.dlpay.mobile.AuthLoginSuccessHandler
import com.supwisdom.dlpay.mobile.service.MobileUserService
import com.supwisdom.dlpay.portal.OperLoginFailHandler
import com.supwisdom.dlpay.portal.OperLoginSuccessHandler
import org.jose4j.jwt.ReservedClaimNames
import org.jose4j.jwt.consumer.InvalidJwtException
import org.jose4j.lang.JoseException
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.core.annotation.Order
import org.springframework.http.HttpStatus
import org.springframework.security.authentication.AuthenticationManager
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken
import org.springframework.security.authentication.dao.DaoAuthenticationProvider
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity
import org.springframework.security.config.annotation.web.builders.HttpSecurity
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
import org.springframework.security.config.http.SessionCreationPolicy
import org.springframework.security.core.authority.SimpleGrantedAuthority
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter
import org.springframework.stereotype.Component
import org.springframework.web.cors.CorsConfiguration
import org.springframework.web.cors.CorsConfigurationSource
import org.springframework.web.cors.UrlBasedCorsConfigurationSource
import org.springframework.web.filter.OncePerRequestFilter
import java.security.SecureRandom
import java.util.*
import javax.servlet.FilterChain
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@Component
class ApiJwtAuthenticationFilter : OncePerRequestFilter() {
    @Autowired
    lateinit var jwtConfig: JwtConfig

    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository

    private var jwtUtil: JwtTokenUtil? = null

    private fun getUtil(): JwtTokenUtil {
        if (jwtUtil == null) {
            jwtUtil = JwtTokenUtil((jwtConfig))
        }
        return jwtUtil as JwtTokenUtil
    }

    override fun doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain) {
        if (!jwtConfig.multiTenant) {
            TenantContext.setTenantSchema(Constants.DEFAULT_TENANTID)
        }
        request.getHeader(jwtConfig.header)?.let { authHeader ->
            try {
                val jwt = if (authHeader.startsWith(jwtConfig.tokenHeader)) {
                    authHeader.substring(jwtConfig.tokenHeader.length)
                } else {
                    throw JoseException("JWT Header error")
                }
                val claims = getUtil().verifyToken(jwt)
                apiJwtRepository.findById(claims[ReservedClaimNames.JWT_ID].toString()).let {
                    if (!it.isPresent) {
                        throw JoseException("JWT has not been register")
                    }
                    // token 已被设为黑名单
                    if (it.get().status != TradeDict.JWT_STATUS_NORMAL) {
                        throw JoseException("JWT status error : ${it.get().status}")
                    }
                }
                if (jwtConfig.multiTenant) {
                    val tenantId = request.getHeader(Constants.HEADER_TETANTID)
                    if (tenantId == null) {
                        response.status = HttpStatus.UNAUTHORIZED.value()
                        return
                    }
                    if (claims[Constants.JWT_CLAIM_TENANTID] != tenantId) {
                        response.status = HttpStatus.UNAUTHORIZED.value()
                        return
                    }
                    TenantContext.setTenantSchema(tenantId)
                }
                val auth = UsernamePasswordAuthenticationToken(claims[Constants.JWT_CLAIM_UID], null,
                        (claims[Constants.JWT_CLAIM_AUTHORITIES] as ArrayList<*>)
                                .map { SimpleGrantedAuthority(it as String) })
                SecurityContextHolder.getContext().authentication = auth
            } catch (e: InvalidJwtException) {
                SecurityContextHolder.clearContext()
                if (e.hasExpired()) {
                    // jwt 过期后返回 401
                    apiJwtRepository.deleteById(e.jwtContext.jwtClaims.jwtId)
                }
                response.status = HttpStatus.UNAUTHORIZED.value()
                return
            } catch (e: JoseException) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.status = HttpStatus.UNAUTHORIZED.value()
                response.contentType = "application/json;charset=UTF-8"
                return
            } catch (e: Exception) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.status = HttpStatus.UNAUTHORIZED.value()
                response.contentType = "application/json;charset=UTF-8"
                return
            }
        }
        filterChain.doFilter(request, response)
    }
}

@Component
class MobileSecurityFilter : OncePerRequestFilter() {
    @Autowired
    lateinit var jwtConfig: JwtConfig

    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository

    private var jwtUtil: JwtTokenUtil? = null

    private fun getUtil(): JwtTokenUtil {
        if (jwtUtil == null) {
            jwtUtil = JwtTokenUtil((jwtConfig))
        }
        return jwtUtil as JwtTokenUtil
    }

    override fun doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain) {
        var context: String? = request.contextPath
        if (context == null || "" == context.trim { it <= ' ' }) {
            context = "/"
        }
        if (request.requestURI.isEmpty()) {
            filterChain.doFilter(request, response)
            return
        }
        var url = request.requestURI
        if ("/" != context) {
            url = url.replace(context, "")
        }
        logger.info(url)
        request.getHeader(jwtConfig.header)?.let { authHeader ->
            try {
                val jwt = if (authHeader.startsWith(jwtConfig.tokenHeader)) {
                    authHeader.substring(jwtConfig.tokenHeader.length)
                } else {
                    throw JoseException("JWT Header error")
                }
                val claims = getUtil().verifyToken(jwt)
                apiJwtRepository.findById(claims[ReservedClaimNames.JWT_ID].toString()).let {
                    if (!it.isPresent) {
                        throw JoseException("JWT has not been register")
                    }
                    // token 已被设为黑名单
                    if (it.get().status != TradeDict.JWT_STATUS_NORMAL) {
                        throw JoseException("JWT status error : ${it.get().status}")
                    }
                }
                if (jwtConfig.multiTenant) {
                    val tenantId = request.getHeader(Constants.HEADER_TETANTID)
                    if (tenantId == null) {
                        response.status = HttpStatus.UNAUTHORIZED.value()
                        return
                    }
                    if (claims[Constants.JWT_CLAIM_TENANTID] != tenantId) {
                        response.status = HttpStatus.UNAUTHORIZED.value()
                        return
                    }
                    TenantContext.setTenantSchema(tenantId)
                }
                val auth = UsernamePasswordAuthenticationToken(claims[Constants.JWT_CLAIM_UID], null,
                        (claims[Constants.JWT_CLAIM_AUTHORITIES] as ArrayList<*>)
                                .map { SimpleGrantedAuthority(it as String) })
                SecurityContextHolder.getContext().authentication = auth
            } catch (e: InvalidJwtException) {
                SecurityContextHolder.clearContext()
                if (e.hasExpired()) {
                    // jwt 过期后返回 401
                    apiJwtRepository.deleteById(e.jwtContext.jwtClaims.jwtId)
                }
                response.status = HttpStatus.UNAUTHORIZED.value()
                return
            } catch (e: JoseException) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.status = HttpStatus.UNAUTHORIZED.value()
                response.contentType = "application/json;charset=UTF-8"
                return
            } catch (e: Exception) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.status = HttpStatus.UNAUTHORIZED.value()
                response.contentType = "application/json;charset=UTF-8"
                return
            }
        }
        response.setHeader("Access-Control-Allow-Origin", "*");
        response.setHeader("Access-Control-Allow-Methods", "GET,POST,DELETE,PUT,OPTIONS");
        response.setHeader("Access-Control-Allow-Headers", "*");
        response.setHeader("Access-Control-Allow-Credentials", "true")
        filterChain.doFilter(request, response)
    }
}

@Component
class PortalApiSecurityFilter : OncePerRequestFilter() {
    @Autowired
    lateinit var jwtConfig: JwtConfig

    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository

    private var jwtUtil: JwtTokenUtil? = null

    private fun getUtil(): JwtTokenUtil {
        if (jwtUtil == null) {
            jwtUtil = JwtTokenUtil((jwtConfig))
        }
        return jwtUtil as JwtTokenUtil
    }

    override fun doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain) {
        var context: String? = request.contextPath
        if (context == null || "" == context.trim { it <= ' ' }) {
            context = "/"
        }
        if (request.requestURI.isEmpty()) {
            filterChain.doFilter(request, response)
            return
        }
        var url = request.requestURI
        if ("/" != context) {
            url = url.replace(context, "")
        }
        logger.info(url)
        request.getHeader(jwtConfig.header)?.let { authHeader ->
            try {
                val jwt = if (authHeader.startsWith(jwtConfig.tokenHeader)) {
                    authHeader.substring(jwtConfig.tokenHeader.length)
                } else {
                    throw JoseException("JWT Header error")
                }
                val claims = getUtil().verifyToken(jwt)
                apiJwtRepository.findById(claims[ReservedClaimNames.JWT_ID].toString()).let {
                    if (!it.isPresent) {
                        throw JoseException("JWT has not been register")
                    }
                    // token 已被设为黑名单
                    if (it.get().status != TradeDict.JWT_STATUS_NORMAL) {
                        throw JoseException("JWT status error : ${it.get().status}")
                    }
                }
                if (jwtConfig.multiTenant) {
                    val tenantId = request.getHeader(Constants.HEADER_TETANTID)
                    if (tenantId == null) {
                        response.status = HttpStatus.UNAUTHORIZED.value()
                        return
                    }
                    if (claims[Constants.JWT_CLAIM_TENANTID] != tenantId) {
                        response.status = HttpStatus.UNAUTHORIZED.value()
                        return
                    }
                    TenantContext.setTenantSchema(tenantId)
                }
                val auth = UsernamePasswordAuthenticationToken(claims[Constants.JWT_CLAIM_UID], null,
                        (claims[Constants.JWT_CLAIM_AUTHORITIES] as ArrayList<*>)
                                .map { SimpleGrantedAuthority(it as String) })
                SecurityContextHolder.getContext().authentication = auth
            } catch (e: InvalidJwtException) {
                SecurityContextHolder.clearContext()
                if (e.hasExpired()) {
                    // jwt 过期后返回 401
                    apiJwtRepository.deleteById(e.jwtContext.jwtClaims.jwtId)
                }
                response.status = HttpStatus.UNAUTHORIZED.value()
                return
            } catch (e: JoseException) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.status = HttpStatus.UNAUTHORIZED.value()
                response.contentType = "application/json;charset=UTF-8"
                return
            } catch (e: Exception) {
                SecurityContextHolder.clearContext()
                // jwt 失效后返回 401
                response.status = HttpStatus.UNAUTHORIZED.value()
                response.contentType = "application/json;charset=UTF-8"
                return
            }
        }
        response.setHeader("Access-Control-Allow-Origin", "*");
        response.setHeader("Access-Control-Allow-Methods", "GET,POST,DELETE,PUT,OPTIONS");
        response.setHeader("Access-Control-Allow-Headers", "*");
        response.setHeader("Access-Control-Allow-Credentials", "true")
        filterChain.doFilter(request, response)
    }
}

@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true)
class WebSecurityConfig {

    companion object {
        @Configuration
        @Order(1)
        class ApiWebSecurityConfigurationAdapter : WebSecurityConfigurerAdapter() {
            @Autowired
            lateinit var apiJwtAuthenticationFilter: ApiJwtAuthenticationFilter
            override fun configure(http: HttpSecurity) {
                // 设置 API 访问权限管理
                http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                        .and()
                        .antMatcher("/api/**")
                        .addFilterAfter(apiJwtAuthenticationFilter,
                                UsernamePasswordAuthenticationFilter::class.java)
                        .authorizeRequests()
                        .antMatchers("/api/auth/**").permitAll()
                        .antMatchers("/api/notify/**").permitAll()
                        .antMatchers("/api/common/**").permitAll()
                        .antMatchers("/api/consume/**").hasRole("THIRD_CONSUME")
                        .antMatchers("/api/recharge/**").hasRole("THIRD_DEPOSIT")
                        .antMatchers("/api/user/**").hasRole("THIRD_ADMIN")
                        .antMatchers("/api/shop/**").hasRole("THIRD_SHOP")
                        .anyRequest().hasRole("THIRD_COMMON")
                        .and()
                        .csrf().ignoringAntMatchers("/api/**", "oauth/**")

            }

            @Bean
            override fun authenticationManager(): AuthenticationManager {
                return super.authenticationManagerBean()
            }
        }

        @Configuration
        @Order(2)
        class MobileApiSecurityConfigurationAdapter : WebSecurityConfigurerAdapter() {
            @Autowired
            lateinit var failureHandler: AuthLoginFailHandler
            @Autowired
            lateinit var successHandler: AuthLoginSuccessHandler
            @Autowired
            lateinit var passwordBCryptConfig: PasswordBCryptConfig

            @Autowired
            lateinit var userDetailsService: MobileUserService
            @Autowired
            lateinit var mobileSecurityFilter: MobileSecurityFilter


            override fun configure(auth: AuthenticationManagerBuilder) {
                auth.authenticationProvider(userProvider())
            }

            @Bean
            fun userProvider(): DaoAuthenticationProvider {
                return DaoAuthenticationProvider().apply {
                    setUserDetailsService(userDetailsService)
                    setPasswordEncoder(userPasswordEncoder())
                }
            }

            @Bean
            fun userPasswordEncoder(): BCryptPasswordEncoder {
                return if (passwordBCryptConfig.seed.isBlank()) {
                    BCryptPasswordEncoder()
                } else {
                    BCryptPasswordEncoder(passwordBCryptConfig.length,
                            SecureRandom(passwordBCryptConfig.seed.toByteArray()))
                }
            }

            override fun configure(http: HttpSecurity) {
                http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                        .and()
                        .cors()
                        .and()
                        .antMatcher("/mobileapi/**")
                        .addFilterAfter(mobileSecurityFilter,
                                UsernamePasswordAuthenticationFilter::class.java)
                        .authorizeRequests().antMatchers("/mobileapi/i/**", "/mobileapi/login")
                        .permitAll().anyRequest().authenticated()
                        .and()
                        .formLogin()
                        .loginProcessingUrl("/mobileapi/login")
                        .failureHandler(failureHandler)
                        .successHandler(successHandler)
                        .and().csrf().disable()
            }

            @Bean
            fun corsConfigurationSource(): CorsConfigurationSource {
                //手机端支持跨域请求
                val configuration = CorsConfiguration()
                configuration.allowedOrigins = listOf("*")
                configuration.allowedMethods = listOf("GET", "POST")
                configuration.allowedHeaders = listOf("*")
                val source = UrlBasedCorsConfigurationSource()
                source.registerCorsConfiguration("/**", configuration)
                return source
            }
        }

        @Configuration
        @Order(3)
        class PortalApiSecurityConfigurationAdapter : WebSecurityConfigurerAdapter() {
            @Autowired
            lateinit var failureHandler: OperLoginFailHandler
            @Autowired
            lateinit var successHandler: OperLoginSuccessHandler
            @Autowired
            lateinit var passwordBCryptConfig: PasswordBCryptConfig

            @Autowired
            lateinit var operatorDetailService: OperatorDetailService
            @Autowired
            lateinit var portalApiSecurityFilter: PortalApiSecurityFilter


            override fun configure(auth: AuthenticationManagerBuilder) {
                auth.authenticationProvider(operatorUserProvider())
            }

            @Bean
            fun operatorUserProvider(): DaoAuthenticationProvider {
                return DaoAuthenticationProvider().apply {
                    setUserDetailsService(operatorDetailService)
                    setPasswordEncoder(operatorUserPasswordEncoder())
                }
            }

            @Bean
            fun operatorUserPasswordEncoder(): BCryptPasswordEncoder {
                return if (passwordBCryptConfig.seed.isBlank()) {
                    BCryptPasswordEncoder()
                } else {
                    BCryptPasswordEncoder(passwordBCryptConfig.length,
                            SecureRandom(passwordBCryptConfig.seed.toByteArray()))
                }
            }

            override fun configure(http: HttpSecurity) {
                http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                        .and()
                        .cors()
                        .and()
                        .antMatcher("/portalapi/**")
                        .addFilterAfter(portalApiSecurityFilter,
                                UsernamePasswordAuthenticationFilter::class.java)
                        .authorizeRequests().antMatchers("/portalapi/login").permitAll()
                        .anyRequest().authenticated()
                        .and()
                        .formLogin()
                        .loginProcessingUrl("/portalapi/login")
                        .failureHandler(failureHandler)
                        .successHandler(successHandler)
                        .and().csrf().disable()
            }
        }
    }
}
