package com.supwisdom.dlpay.system.service.impl;

import com.supwisdom.dlpay.api.bean.JsonResult;
import com.supwisdom.dlpay.framework.dao.FunctionDao;
import com.supwisdom.dlpay.framework.dao.ResourceDao;
import com.supwisdom.dlpay.framework.dao.RoleFunctionDao;
import com.supwisdom.dlpay.framework.domain.TFunction;
import com.supwisdom.dlpay.framework.domain.TResource;
import com.supwisdom.dlpay.framework.util.PageResult;
import com.supwisdom.dlpay.framework.util.StringUtil;
import com.supwisdom.dlpay.system.bean.FunctionSearchBean;
import com.supwisdom.dlpay.system.service.FunctionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class FunctionServiceImpl implements FunctionService {
  @Autowired
  private FunctionDao functionDao;
  @Autowired
  private ResourceDao resourceDao;
  @Autowired
  private RoleFunctionDao roleFunctionDao;

  @Override
  @Cacheable(cacheNames = "oper_function_list", key = "#p0")
  public List<TFunction> getFunctionsByOperid(String operid) {
    List<TFunction> list = functionDao.getTFunctionsByOperid(StringUtil.isEmpty(operid) ? "" : operid.trim());
    if (!StringUtil.isEmpty(list)) return list;
    return new ArrayList<>(0);
  }

  @Override
  public List<Map<String, Object>> getMenuTree(List<TFunction> funcList, Integer parentId) {
    List<Map<String, Object>> list = new ArrayList<>(0);
    for (TFunction func : funcList) {
      if (parentId.equals(func.getParentId())) {
        Map<String, Object> map = new HashMap<>(0);
        map.put("menuName", func.getName());
        map.put("menuIcon", func.getMenuIcon());
        if ("#".equals(func.getMenuUrl())) {
          map.put("menuUrl", "javascript:;");
        } else {
          map.put("menuUrl", func.getMenuUrl());
        }
        map.put("subMenus", getMenuTree(funcList, func.getId()));
        list.add(map);
      }
    }
    return list;
  }

  @Override
  public PageResult<TFunction> getFunctionsByKey(FunctionSearchBean param) {
    Pageable pageable = PageRequest.of(param.getPageNo() - 1, param.getPageSize()
        , Sort.by("id"));
    if (!StringUtil.isEmpty(param.getFunctioname())) {
      return new PageResult<>(functionDao.findAllByNameContaining(param.getFunctioname(), pageable));
    }
    return new PageResult<>(functionDao.findAll(pageable));
  }

  @Override
  @CacheEvict(cacheNames = "oper_function_list")
  public TFunction getFunctionByNameAndId(String name, Integer id) {
    if (id != null && id != 0) {
      return functionDao.findByNameAndIdNot(name, id);
    }
    return functionDao.findByName(name);
  }

  @Override
  public JsonResult saveFunction(TFunction function) {
    TFunction temp = getFunctionByNameAndId(function.getName(), function.getId());
    if (temp != null) {
      return JsonResult.error("功能名称重复");
    }
    if (function.getParentId() == null) {
      function.setParentId(-1);
    }
    functionDao.save(function);
    return JsonResult.ok("成功");
  }

  @Override
  public TFunction getFunctionById(Integer id) {
    return functionDao.getOne(id);
  }

  @Override
  public List<TFunction> getParentFunction() {
    return functionDao.findByIsLeaf(0);
  }

  @Override
  public List<TResource> getResources(Integer function) {
    return resourceDao.findByFunctionId(function);
  }

  @Override
  public List<TFunction> getLeafFunction() {
    return functionDao.findByIsLeaf(1);
  }

  @Override
  public JsonResult saveRes(TResource resource) {
    if (resource.getId() != null && resource.getId() != 0) {
      TResource temp = resourceDao.findByUriAndIdNot(resource.getUri(), resource.getId());
      if (temp != null) {
        return JsonResult.error("资源路径已存在");
      }
    } else {
      TResource temp = resourceDao.findByUri(resource.getUri());
      if (temp != null) {
        return JsonResult.error("资源路径已存在");
      }
    }
    resourceDao.save(resource);
    return JsonResult.ok("成功");
  }

  @Override
  public JsonResult deleteResource(Integer resid) {
    resourceDao.deleteById(resid);
    return JsonResult.ok("成功");
  }

  @Override
  public JsonResult deleteFunction(Integer funcid) {
    resourceDao.deleteByFunctionId(funcid);
    roleFunctionDao.deleteByFunctionId(funcid);
    functionDao.deleteById(funcid);
    return JsonResult.ok("成功");
  }
}
