package com.supwisdom.dlpay.system.service.impl;

import com.supwisdom.dlpay.api.dao.PaytypeConfigDao;
import com.supwisdom.dlpay.api.dao.SourceTypeDao;
import com.supwisdom.dlpay.api.domain.TPaytypeConfig;
import com.supwisdom.dlpay.api.domain.TSourceType;
import com.supwisdom.dlpay.framework.dao.ApiClientDao;
import com.supwisdom.dlpay.framework.dao.BusinessparaDao;
import com.supwisdom.dlpay.framework.dao.SysparaDao;
import com.supwisdom.dlpay.framework.domain.TApiClient;
import com.supwisdom.dlpay.framework.domain.TBusinesspara;
import com.supwisdom.dlpay.framework.domain.TSyspara;
import com.supwisdom.dlpay.framework.tenant.TenantContext;
import com.supwisdom.dlpay.framework.util.PageResult;
import com.supwisdom.dlpay.framework.util.StringUtil;
import com.supwisdom.dlpay.system.service.ParamService;
import com.supwisdom.dlpay.util.ConstantUtil;
import com.supwisdom.dlpay.util.WebCheckException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


@Service
public class ParamServiceImpl implements ParamService {
  @Autowired
  private SysparaDao sysparaDao;
  @Autowired
  private BusinessparaDao businessparaDao;
  @Autowired
  private ApiClientDao apiClientDao;
  @Autowired
  private SourceTypeDao sourceTypeDao;
  @Autowired
  private PaytypeConfigDao paytypeConfigDao;

  @Override
  public PageResult<TSyspara> getSysparaPage(Integer paraid, String paraname, int pageNo, int pageSize) {
    Pageable pageable = PageRequest.of(pageNo - 1, pageSize, Sort.by("paraid"));
    Page<TSyspara> page = sysparaDao.findAll(new Specification<TSyspara>() {
      @Override
      public Predicate toPredicate(Root<TSyspara> root, CriteriaQuery<?> query, CriteriaBuilder criteriaBuilder) {
        List<Predicate> predicates = new ArrayList<>();
        if (null != paraid) {
          predicates.add(criteriaBuilder.equal(root.get("paraid").as(Integer.class), paraid));
        }
        if (!StringUtil.isEmpty(paraname)) {
          predicates.add(criteriaBuilder.like(root.get("paraname").as(String.class), "%" + paraname.trim() + "%"));
        }
        predicates.add(criteriaBuilder.equal(root.get("displayflag").as(String.class), ConstantUtil.ENABLE_YES));
        return criteriaBuilder.and(predicates.toArray(new Predicate[0]));
      }
    }, pageable);
    return new PageResult<>(page);
  }

  @Override
  public TSyspara getSysparaByParaid(int paraid) {
    return sysparaDao.findByParaid(paraid);
  }

  @Override
  public boolean updateSyspara(TSyspara syspara) {
    if (null != syspara) {
      sysparaDao.save(syspara);
      return true;
    }
    return false;
  }

  @Override
  public PageResult<TBusinesspara> getBusinessparaPage(String paraname, int pageNo, int pageSize) {
    Pageable pageable = PageRequest.of(pageNo - 1, pageSize, Sort.by("parakey"));
    if (!StringUtil.isEmpty(paraname)) {
      return new PageResult<>(businessparaDao.findAllByParakeyContaining(paraname.trim(), pageable));
    }
    return new PageResult<>(businessparaDao.findAll(pageable));
  }

  @Override
  public TBusinesspara getBusinessparaByParakey(String parakey) {
    if (!StringUtil.isEmpty(parakey)) {
      return businessparaDao.findByParakey(parakey.trim());
    }
    return null;
  }

  @Override
  public boolean deleteBusinesspara(TBusinesspara businesspara) {
    if (null != businesspara) {
      businessparaDao.delete(businesspara);
      return true;
    }
    return false;
  }


  @Override
  public PageResult<TApiClient> getApiClientparaPage(String appid, int pageNo, int pageSize) {
    Pageable pageable = PageRequest.of(pageNo - 1, pageSize, Sort.by("appid"));
    if (!StringUtil.isEmpty(appid)) {
      return new PageResult<>(apiClientDao.findByAppidContaining(appid.trim(), pageable));
    }
    return new PageResult<>(apiClientDao.findAll(pageable));
  }

  @Override
  public TApiClient getApiClientByAppid(String appid) {
    if (!StringUtil.isEmpty(appid)) {
      return apiClientDao.findByAppid(appid.trim());
    }
    return null;
  }

  @Override
  public boolean saveOrUpdateApiClient(TApiClient apiClient) {
    if (null != apiClient) {
      apiClientDao.save(apiClient);
      return true;
    }
    return false;
  }

  @Override
  public boolean deleteApiClient(TApiClient apiClient) {
    if (null != apiClient) {
      apiClientDao.delete(apiClient);
      return true;
    }
    return false;
  }

  @Override
  public PageResult<TSourceType> getSourceTypePage(String paytype, int pageNo, int pageSize) {
    Pageable pageable = PageRequest.of(pageNo - 1, pageSize, Sort.by("sourceType"));
    if (!StringUtil.isEmpty(paytype)) {
      return new PageResult<>(sourceTypeDao.findBySourceTypeContaining(paytype.trim(), pageable));
    }
    return new PageResult<>(sourceTypeDao.findAll(pageable));
  }

  @Override
  @Cacheable(cacheNames = "source_type_cache", key = "#p0")
  public TSourceType getSourceType(String paytype) {
    if (!StringUtil.isEmpty(paytype)) {
      return sourceTypeDao.getBySourceType(paytype.trim());
    }
    return null;
  }

  @Override
  @Cacheable(cacheNames = "source_type_cache")
  public List<TSourceType> getAllSourceType() {
    return sourceTypeDao.findAll();
  }


  @Override
  @CacheEvict(cacheNames = "source_type_cache", key = "#paytype.sourceType")
  public boolean deleteSourceType(TSourceType paytype) {
    if (null != paytype) {
      sourceTypeDao.delete(paytype);
      return true;
    }
    return false;
  }

  @Override
  @Cacheable(cacheNames = "source_type_config_cache", key = "#p0")
  public List<TPaytypeConfig> getSourceTypeConfigList(String paytype) {
    if (!StringUtil.isEmpty(paytype)) {
      List<TPaytypeConfig> list = paytypeConfigDao.getByPaytypeOrderByConfigid(paytype.trim());
      if (!StringUtil.isEmpty(list))
        return list;
    }
    return new ArrayList<>(0);
  }

  @Override
  @CacheEvict(cacheNames = "source_type_config_cache", key = "#p0")
  public boolean saveSourceTypeConfig(String paytype, Map<String, String> param) throws WebCheckException {
    TSourceType tPaytype = getSourceType(paytype);
    if (null == tPaytype) throw new WebCheckException("支付能力[" + paytype + "]不存在");
    for (String key : param.keySet()) {
      String value = param.get(key);
      TPaytypeConfig config = paytypeConfigDao.getByPaytypeAndAndConfigid(tPaytype.getSourceType(), key);
      if (null == config)
        throw new WebCheckException("支付能力[" + tPaytype.getSourceType() + "]不存在配置项[" + key + "],请重新查询");
      config.setConfigValue(StringUtil.isEmpty(value) ? null : value.trim());
      paytypeConfigDao.save(config);
    }
    return true;
  }


}
