package com.supwisdom.dlpay.water.service.impl;

import com.supwisdom.dlpay.exception.TransactionProcessException;
import com.supwisdom.dlpay.framework.dao.DictionaryDao;
import com.supwisdom.dlpay.framework.domain.TDictionary;
import com.supwisdom.dlpay.framework.util.*;
import com.supwisdom.dlpay.framework.util.Dictionary;
import com.supwisdom.dlpay.water.DeviceLineCheckParam;
import com.supwisdom.dlpay.water.DeviceLoginParam;
import com.supwisdom.dlpay.water.dao.AreaDao;
import com.supwisdom.dlpay.water.dao.DeviceDao;
import com.supwisdom.dlpay.water.dao.DeviceFeeConfigDao;
import com.supwisdom.dlpay.water.dao.FeeConfigDao;
import com.supwisdom.dlpay.water.domain.*;
import com.supwisdom.dlpay.water.service.DeviceService;
import com.supwisdom.dlpay.water.bean.DeviceSearchBean;
import org.hibernate.query.internal.NativeQueryImpl;
import org.hibernate.transform.Transformers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class DeviceServiceImpl implements DeviceService {

    @Autowired
    private DeviceDao deviceDao;

    @Autowired
    private AreaDao areaDao;

    @Autowired
    private FeeConfigDao feeConfigDao;

    @Autowired
    private DeviceFeeConfigDao deviceFeeConfigDao;

    @Autowired
    private DictionaryDao dictionaryDao;

    @PersistenceContext
    private EntityManager entityManager;

    private final static String paraSql = "select p.* from tb_device d left join " +
            "tb_areapara_bind b on d.areano = b.areano left join " +
            "tb_areapara p on b.groupid = p.groupid where d.deviceno = ?";

    @Override
    public boolean existSubDevice(Integer areano) {
        List<TDevice> devices = deviceDao.findByAreano(areano);
        return devices != null && devices.size() != 0;
    }

    @Override
    public PageResult<TDevice> queryDeviceList(DeviceSearchBean param) {
        Pageable pageable = PageRequest.of(param.getPageNo() - 1, param.getPageSize());
        Page<TDevice> devicePage = deviceDao.findAllByOrderByDeviceno(pageable);
        return queryAreaname(devicePage);
    }

    @Override
    public TDevice queryDeviceById(Integer deviceid) {
        Optional<TDevice> optional = deviceDao.findById(deviceid);
        return optional.orElse(null);
    }

    @Override
    public TDevice queryDeviceByDeviceno(String deviceno) {
        return deviceDao.findByDeviceno(deviceno);
    }

    @Override
    public boolean updateDevice(TDevice device) {
        Optional<TDevice> deviOption = deviceDao.findById(device.getDeviceid());
        Optional<TArea> regiOption = areaDao.findByAvailableAndAreano(1, device.getAreano());
        if (deviOption.isPresent() && regiOption.isPresent()) {
            TDevice oldDevice = deviOption.get();
            if (oldDevice.getDeviceStatus().equals("closed")) {
                return false;
            }
            TArea region = regiOption.get();
            oldDevice.setDevicename(device.getDevicename());
            oldDevice.setDeviceno(device.getDeviceno());
            oldDevice.setAreano(region.getAreano());
            deviceDao.save(oldDevice);
            return true;
        }
        return false;
    }

    @Override
    public boolean saveDevice(TDevice device) {
        Optional<TArea> regiOption = areaDao.findByAvailableAndAreano(1, device.getAreano());
        if (regiOption.isPresent()) {
            TArea region = regiOption.get();
            device.setAreano(region.getAreano());
            device.setDeviceStatus("normal");
            deviceDao.save(device);
            return true;
        }
        return false;
    }

    @Override
    public Map<String, Object> checkParams(TDevice device) {
        Map<String, Object> result = new HashMap<>();
        if (null == device.getDevicename()) {
            result.put("flag", false);
            result.put("msg", "设备名称不能为空");
            return result;
        }
        if (null == device.getDeviceno()) {
            result.put("flag", false);
            result.put("msg", "设备编号不能为空");
            return result;
        }
        if (device.getDeviceno().length() != 8) {
            result.put("flag", false);
            result.put("msg", "请输入合法的设备编号");
            return result;
        }
        TDevice queryDevicename = null;
        TDevice queryDevicenano = null;
        if (device.getDeviceid() != null) {
            queryDevicename = deviceDao.findByDevicenameAndDeviceidNot(device.getDevicename(), device.getDeviceid());
            queryDevicenano = deviceDao.findByDevicenoAndDeviceidNot(device.getDeviceno(), device.getDeviceid());
            result.put("type", "update");
        } else {
            queryDevicename = deviceDao.findByDevicename(device.getDevicename());
            queryDevicenano = deviceDao.findByDeviceno(device.getDeviceno());
            result.put("type", "insert");
        }
        if (queryDevicename != null) {
            result.put("flag", false);
            result.put("msg", "已存在的设备名");
            return result;
        }
        if (queryDevicenano != null) {
            result.put("flag", false);
            result.put("msg", "已存在的设备编号");
            return result;
        }
        result.put("flag", true);
        return result;
    }

    @Override
    public boolean deleteDevice(Integer deviceid) {
        TDevice device = queryDeviceById(deviceid);
        if (device != null) {
            device.setDeviceStatus("closed");
            deviceDao.save(device);
            return true;
        }
        return false;
    }

    @Override
    public List<TDictionary> groupStatus() {
        return dictionaryDao.findAllByDicttype(Dictionary.WATER_DEVICE_STATUS);
    }

    @Override
    public PageResult<TDevice> queryDeviceByParams(DeviceSearchBean param) {
        Pageable pageable = PageRequest.of(param.getPageNo() - 1, param.getPageSize(), Sort.by("deviceno"));
        Specification<TDevice> specification = (Specification<TDevice>) (root, query, cb) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (!StringUtil.isEmpty(param.getDeviceno())) {
                predicates.add(cb.like(root.get("deviceno"),"%" + param.getDeviceno() + "%"));
            }
            if (param.getAreano() != null) {
                predicates.add(cb.equal(root.get("areano").as(Integer.class), param.getAreano()));
            }
            if (!StringUtil.isEmpty(param.getDeviceStatus())) {
                predicates.add(cb.equal(root.get("deviceStatus").as(String.class), param.getDeviceStatus()));
            }
            if (!StringUtil.isEmpty(param.getDevicename())) {
                predicates.add(cb.like(root.get("devicename"), "%" + param.getDevicename() + "%"));
            }
            Predicate[] pre = new Predicate[predicates.size()];
            return query.where(predicates.toArray(pre)).getRestriction();
        };
        Page<TDevice> devicePage = deviceDao.findAll(specification, pageable);
        return queryAreaname(devicePage);
    }

    @Override
    public PageResult<TDevice> queryAreaname(Page<TDevice> devicePage) {
        devicePage.get().forEach(device -> {
            Optional<TArea> optional = areaDao.findByAvailableAndAreano(1, device.getAreano());
            optional.ifPresent(tArea -> device.setAreaname(tArea.getAreaName()));
        });
        return new PageResult<>(devicePage);
    }

    @Override
    public Map<String, Object> deviceLogin(DeviceLoginParam param) {
        Map<String, Object> result = new HashMap<>();
        TDevice device = deviceDao.findByDeviceno(param.getDeviceno());
        if (device != null) {
            if (device.getDeviceStatus().equals("closed")) {
                result.put("flag", false);
                result.put("errorMsg", "本设备已注销，签到失败");
                return result;
            }
            device.setSoftVer(param.getHwVer());
            deviceDao.save(device);
            Map<String, String> deviceParam = getParaMapByDeviceno(param.getDeviceno());
            result.put("flag", true);
            result.put(WaterDeviceParam.devOfflineMaxHour, deviceParam.get(WaterDeviceParam.devOfflineMaxHour));
            result.put(WaterDeviceParam.pulseInHML, deviceParam.get(WaterDeviceParam.pulseInHML));
            return result;
        }
        result.put("flag", false);
        result.put("errorMsg", "没有编号为" + param.getDeviceno() + "的设备");
        return result;
    }

    @Override
    public TDevice lineCheck(DeviceLineCheckParam param) {
        TDevice device = deviceDao.findByDeviceno(param.getDeviceno());
        if (device != null) {
            String deviceLineCheck = param.getTermdate() + param.getTermtime();
            device.setDeviceLineCheck(deviceLineCheck);
            deviceDao.save(device);
            return device;
        }
        return null;
    }

    @Override
    public TFeeConfig queryTFeeConfigByDeviceno(String deviceno) {
        TDevice device = deviceDao.findByDeviceno(deviceno);
        if (device != null) {
            TDeviceFeeConfig deviceFeeConfig = deviceFeeConfigDao.findByDeviceid(device.getDeviceid());
            if (deviceFeeConfig != null) {
                return feeConfigDao.findById(deviceFeeConfig.getFeeConfig()).orElse(null);
            }
        }
        return null;
    }

    @Override
    public Map<String, String> getParaMapByDeviceno(String deviceno) {
        Query query = entityManager.createNativeQuery(paraSql);
        query.setParameter(1, deviceno);
        query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.aliasToBean(TAreapara.class));
        List<TAreapara> list = query.getResultList();
        if (list == null || list.size() == 0 || list.get(0).getParaname() == null) {
            throw new RuntimeException("该设备未绑定参数,请求失败!");
        }
        return list.stream().collect(Collectors.toMap(TAreapara::getParaname, TAreapara::getParaval));
    }
}
