package com.supwisdom.dlpay.water.service.impl;

import com.supwisdom.dlpay.framework.service.SystemUtilService;
import com.supwisdom.dlpay.framework.util.PageResult;
import com.supwisdom.dlpay.framework.util.StringUtil;
import com.supwisdom.dlpay.util.WebCheckException;
import com.supwisdom.dlpay.water.bean.AreaparaSearchBean;
import com.supwisdom.dlpay.water.dao.AreaDao;
import com.supwisdom.dlpay.water.dao.AreaparaBindDao;
import com.supwisdom.dlpay.water.dao.TAreaparaDao;
import com.supwisdom.dlpay.water.dao.TAreaparaGroupDao;
import com.supwisdom.dlpay.water.domain.TArea;
import com.supwisdom.dlpay.water.domain.TAreapara;
import com.supwisdom.dlpay.water.domain.TAreaparaBind;
import com.supwisdom.dlpay.water.domain.TAreaparaGroup;
import com.supwisdom.dlpay.water.pojo.TAreaparaBindAreaDTO;
import com.supwisdom.dlpay.water.pojo.TAreaparaBindDTO;
import com.supwisdom.dlpay.water.service.TAreaparaService;
import org.hibernate.query.internal.NativeQueryImpl;
import org.hibernate.transform.Transformers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Service
public class TAreaparaServiceImpl implements TAreaparaService {

    @Autowired
    private TAreaparaGroupDao tAreaparaGroupDao;

    @Autowired
    private TAreaparaDao tAreaparaDao;

    @Autowired
    private SystemUtilService systemUtilService;

    @Autowired
    private AreaDao areaDao;

    @Autowired
    private AreaparaBindDao areaparaBindDao;

    @PersistenceContext
    private EntityManager entityManager;

    @Override
    public PageResult<TAreaparaGroup> queryParaGroupByParam(AreaparaSearchBean param) {
        Pageable pageable = PageRequest.of(param.getPageNo() - 1, param.getPageSize(), Sort.by("groupid"));
        if (StringUtil.isEmpty(param.getGroupname())) {
            return new PageResult<>(tAreaparaGroupDao.findAll(pageable));
        }
        return new PageResult<>(tAreaparaGroupDao.findByGroupnameIsContaining(param.getGroupname().trim(), pageable));
    }

    @Override
    public TAreaparaGroup getAreaparaGroupByGroupid(Integer groupid) {
        return tAreaparaGroupDao.findByGroupid(groupid);
    }

    @Override
    public List<TAreapara> getAreaparaInfo(Integer groupid) {
        List<TAreapara> result;
        if (null != groupid && groupid > 0) {
            result = tAreaparaDao.findByGroupid(groupid);
        } else {
            result = tAreaparaDao.findSystemDefaultAreapara(); //默认
        }
        if (!StringUtil.isEmpty(result)) {
            return result;
        } else {
            return new ArrayList<>(0);
        }
    }

    @Override
    public boolean saveOrUpdateAreapara(int groupid, String groupname, Map<String, String> param) throws WebCheckException {
        TAreaparaGroup group;
        if (groupid == 0) {
            if (tAreaparaGroupDao.checkGroupnameExists(groupname) > 0) {
                throw new WebCheckException("参数名称已经存在");
            }
            group = new TAreaparaGroup();
            group.setGlobalflag(false);
            group.setGroupname(groupname);
            group.setLastsaved(systemUtilService.getSysdatetime().getHostdatetime());
            group.setVerno(1L);
            group = tAreaparaGroupDao.save(group);

            List<TAreapara> configList = tAreaparaDao.findSystemDefaultAreapara(); //默认
            for (TAreapara areapara : configList) {
                TAreapara bean = new TAreapara();
                bean.setGroupid(group.getGroupid());
                bean.setParaname(areapara.getParaname());
                bean.setParaval(param.get(areapara.getParaname()));
                bean.setParadesc(areapara.getParadesc());
                bean.setValtype(areapara.getValtype());
                bean.setMinval(areapara.getMinval());
                bean.setMaxval(areapara.getMaxval());
                if (!bean.checkValue()) {
                    throw new WebCheckException(bean.getErrmsg());
                }
                tAreaparaDao.save(bean);
            }
        } else {
            group = tAreaparaGroupDao.findByGroupid(groupid);
            if (null == group) {
                throw new WebCheckException("原设备参数组不存在");
            }
            if (tAreaparaGroupDao.checkGroupnameExists(groupname, groupid) > 0) {
                throw new WebCheckException("参数名称已经存在");
            }
            group.setGroupname(groupname);
            group.setLastsaved(systemUtilService.getSysdatetime().getHostdatetime());
            tAreaparaGroupDao.save(group);

            for (String key : param.keySet()) {
                TAreapara bean = tAreaparaDao.findById(groupid, key);
                if (null == bean) {
                    throw new WebCheckException("设备参数[" + key + "]在参数组【" + group.getGroupname() + "】中不存在");
                }
                bean.setParaval(param.get(key));
                if (!bean.checkValue()) {
                    throw new WebCheckException(bean.getErrmsg());
                }
                tAreaparaDao.save(bean);
            }
        }
        return true;
    }

    @Override
    public PageResult<TAreaparaBindDTO> getAreaparaBindInfos(String searchkey, int pageNo, int pageSize) {
        StringBuffer querySql = new StringBuffer("select b.groupid,g.groupname,b.areano,a.areaname,b.lastsaved from tb_areapara_bind b" +
                " left join tb_areapara_group g on b.groupid = g.groupid" +
                " left join tb_area a on b.areano = a.areano where 1=1");
        StringBuffer countSql = new StringBuffer("select count(b.areano) from tb_areapara_bind b" +
                " left join tb_areapara_group g on b.groupid = g.groupid" +
                " left join tb_area a on b.areano = a.areano where 1=1");
        if (!StringUtil.isEmpty(searchkey)) {
            querySql.append(" and (g.groupname like :str or a.areaname like :str)");
            countSql.append(" and (g.groupname like :str or a.areaname like :str)");
        }
        querySql.append(" order by b.areano ");
        Query query = entityManager.createNativeQuery(querySql.toString());
        Query countQuery = entityManager.createNativeQuery(countSql.toString());
        if (!StringUtil.isEmpty(searchkey)) {
            query.setParameter("str", "%" + searchkey.trim() + "%");
            countQuery.setParameter("str", "%" + searchkey.trim() + "%");
        }
        query.setFirstResult((pageNo - 1) * pageSize);
        query.setMaxResults(pageSize);
        query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.aliasToBean(TAreaparaBindDTO.class));
        List<TAreaparaBindDTO> list = query.getResultList();
        BigInteger count = (BigInteger) countQuery.getSingleResult();
        return new PageResult<>(count.longValue(), list);
    }

    @Override
    public boolean deleteAreaparaGroup(TAreaparaGroup group) throws WebCheckException {
        if (null != group) {
            if (group.getGlobalflag()) {
                throw new WebCheckException("默认参数组不能删除");
            }
            if (areaparaBindDao.getAreaparaBindRecordcnt(group.getGroupid()) > 0) {
                throw new WebCheckException("该参数组下绑定了区域，请先解绑!");
            }
            tAreaparaDao.deleteAllByGroupid(group.getGroupid());
            tAreaparaGroupDao.delete(group);
            return true;
        }
        return false;
    }

    @Override
    public List<TAreaparaGroup> queryAllAreaparaGroups() {
        List<TAreaparaGroup> list = tAreaparaGroupDao.findAllByOrderByGroupid();
        if (!StringUtil.isEmpty(list)) {
            return list;
        }
        return new ArrayList<>(0);
    }

    @Override
    public boolean saveBindAreapara(int groupid, List<Integer> areanos) throws WebCheckException {
        TAreaparaGroup group = tAreaparaGroupDao.findByGroupid(groupid);
        if (null == group) {
            throw new WebCheckException("所选设备参数组不存在");
        }
        for (Integer areano : areanos) {
            Optional<TArea> option = areaDao.findByAvailableAndAreano(1, areano);
            if (!option.isPresent()) {
                throw new WebCheckException("区域编号为[" + areano + "]的区域不存在");
            }
            TAreaparaBind bind = new TAreaparaBind();
            bind.setAreano(option.get().getAreano());
            bind.setGroupid(groupid);
            bind.setLastsaved(systemUtilService.getSysdatetime().getHostdatetime());
            areaparaBindDao.save(bind);
        }
        return true;
    }

    @Override
    public TAreaparaBind getAreaparaBindByAreano(Integer areano) {
        if (null != areano) {
            return areaparaBindDao.findByAreano(areano);
        }
        return null;
    }

    @Override
    public boolean deleteAreaparaBind(TAreaparaBind bind) {
        if (null != bind) {
            areaparaBindDao.delete(bind);
            return true;
        }
        return false;
    }

    @Override
    public PageResult<TArea> getAreaBySearch(String searchKey) {
        StringBuffer sql = new StringBuffer("select a.areano,a.address,a.areaname,a.arealevel from tb_area a left join " +
                "tb_areapara_bind b on a.areano = b.areano where a.available=1 and b.areano is null");
        if (!StringUtil.isEmpty(searchKey)) {
            sql.append(" and a.areaname like :str");
        }
        sql.append(" order by a.areano");
        Query query = entityManager.createNativeQuery(sql.toString());
        if (!StringUtil.isEmpty(searchKey)) {
            query.setParameter("str", "%" + searchKey.trim() + "%");
        }
        query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.aliasToBean(TAreaparaBindAreaDTO.class));
        return new PageResult<>(query.getResultList());
    }
}
