package com.supwisdom.dlpay.framework.controller

import com.google.code.kaptcha.Producer
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.core.JwtConfig
import com.supwisdom.dlpay.framework.core.JwtTokenUtil
import com.supwisdom.dlpay.framework.dao.ApiClientDao
import com.supwisdom.dlpay.framework.domain.ApiClientRedis
import com.supwisdom.dlpay.framework.domain.JwtRedis
import com.supwisdom.dlpay.framework.domain.TOperator
import com.supwisdom.dlpay.framework.redisrepo.ApiClientRepository
import com.supwisdom.dlpay.framework.redisrepo.ApiJwtRepository
import com.supwisdom.dlpay.framework.security.validate.ImageCodeUtil
import com.supwisdom.dlpay.framework.security.validate.VerifyCode
import com.supwisdom.dlpay.framework.service.CommonService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.system.service.FunctionService
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.security.core.annotation.AuthenticationPrincipal
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.security.core.userdetails.UserDetails
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler
import org.springframework.social.connect.web.HttpSessionSessionStrategy
import org.springframework.stereotype.Controller
import org.springframework.ui.Model
import org.springframework.web.bind.annotation.*
import org.springframework.web.context.request.ServletWebRequest
import java.io.IOException
import java.lang.Exception
import java.security.Principal
import java.util.*
import javax.imageio.ImageIO
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@RestController
@RequestMapping("/api/auth")
class ApiAuthController {

    @Autowired
    lateinit var apiClientRepository: ApiClientRepository

    @Autowired
    lateinit var apiJwtRepository: ApiJwtRepository

    @Autowired
    lateinit var apiClientDao: ApiClientDao

    @Autowired
    lateinit var systemUtil: SystemUtilService

    @Autowired
    lateinit var jwtConfig: JwtConfig

    @GetMapping(value = ["/gettoken", "/gettoken/{clientid}"])
    fun loginInit(appid: String, @PathVariable clientid: String?): ResponseEntity<Any> {

        /*   tetantConfigDao.default?.also {
               if (it.tenantId != tetantId) {
                   throw TransactionCheckException(TradeErrorCode.INPUT_DATA_ERROR,
                           "租户ID错误")
               }
           } ?: throw TransactionCheckException(TradeErrorCode.BUSINESS_DEAL_ERROR,
                   "系统未配置租户信息")*/

        apiClientDao.findById(appid).run {
            if (!isPresent) {
                return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build()
            }
            if (get().status != TradeDict.STATUS_NORMAL) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.BUSINESS_DEAL_ERROR, "API状态错误"))
            }
            get()
        }.let { api ->
            val token = generateRandomToken()
            val now = systemUtil.sysdatetime.hostdatetime
            ApiClientRedis().also {
                it.id = if (clientid == null) appid else "$appid-$clientid"
                it.loginTimestamp = now
                it.roles = api.roles
                it.token = HmacUtil.HMACSHA256(token, api.secret)
            }.also {
                apiClientRepository.save(it)
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data("token", token)
                    .data("timestamp", now)
                    .success())
        }
    }

    private fun generateRandomToken(): String {
        val random = ByteArray(12) { 0x00 }
        Random(System.currentTimeMillis()).nextBytes(random)
        return Base64.getEncoder().encode(random).toString(Charsets.UTF_8)
    }

    private fun checkSecretToken(api: ApiClientRedis, secret: String): Boolean {
        return (api.token == secret)
    }

    @GetMapping(value = ["/authentication", "/authentication/{clientid}"])
    fun login(appid: String, sign: String, @PathVariable clientid: String?): ResponseEntity<Any> {
        val requestId = if (clientid == null) appid else "$appid-$clientid"
        return apiClientRepository.findById(requestId).let {
            if (it.isPresent && checkSecretToken(it.get(), sign)) {
                apiClientRepository.deleteById(requestId)
                val token = JwtTokenUtil(jwtConfig).generateToken(
                        mapOf(Constants.JWT_CLAIM_UID to appid,
                                "issuer" to "payapi",
                                "audience" to (clientid ?: appid),
                                Constants.JWT_CLAIM_AUTHORITIES to it.get().roles.split(";")))
                JwtRedis().apply {
                    jti = token.jti
                    uid = appid
                    status = TradeDict.JWT_STATUS_NORMAL
                    expiration = token.expiration.valueInMillis
                }.apply {
                    apiJwtRepository.save(this)
                }

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("jwt", token.jwtToken)
                        .data("appid", appid)
                        .data("expiredAt", DateUtil.getUTCTime(token.expiration.valueInMillis))
                        .success())
            } else {
                ResponseEntity.status(HttpStatus.UNAUTHORIZED).build()
            }
        }
    }


    @GetMapping("/refresh")
    fun refresh(request: HttpServletRequest): ResponseEntity<Any> {
        val auth = request.getHeader(jwtConfig.header) ?: ""
        if (!auth.startsWith(jwtConfig.tokenHeader)) {
            return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build()
        }
        val jwt = JwtTokenUtil(jwtConfig).verifyToken(auth.substring(jwtConfig.tokenHeader.length))
        val appid = jwt["uid"] as String
        return apiClientDao.findById(appid).let {
            if (it.isPresent && it.get().status == TradeDict.STATUS_NORMAL) {
                // 新证书
                val token = JwtTokenUtil(jwtConfig).generateToken(
                        mapOf(Constants.JWT_CLAIM_UID to appid,
                                "issuer" to "payapi",
                                "audience" to jwt["audience"],
                                Constants.JWT_CLAIM_AUTHORITIES to it.get().roles.split(";")))
                JwtRedis().apply {
                    jti = token.jti
                    uid = appid
                    status = TradeDict.JWT_STATUS_NORMAL
                    expiration = token.expiration.valueInMillis
                }.apply {
                    apiJwtRepository.save(this)
                }

                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("jwt", token.jwtToken)
                        .data("appid", appid)
                        .data("expiredAt", DateUtil.getUTCTime(token.expiration.valueInMillis))
                        .success())
            } else {
                ResponseEntity.status(HttpStatus.UNAUTHORIZED).build()
            }
        }
    }
}

@RestController
class ValidateCodeController {

    @Autowired
    private lateinit var captchaProducer: Producer

    @GetMapping("/code/image")
    fun createCode(request: HttpServletRequest, response: HttpServletResponse) {
        val imageCode = VerifyCode(60)
        HttpSessionSessionStrategy().setAttribute(ServletWebRequest(request), ImageCodeUtil.LOGIN_IMAGECODE_SESSIONKEY, imageCode)
        val session = request.session
        response.setDateHeader("Expires", 0)
        response.setHeader("Cache-Control", "no-store, no-cache, must-revalidate")
        response.addHeader("Cache-Control", "post-check=0, pre-check=0")
        response.setHeader("Pragma", "no-cache")
        response.contentType = "image/jpeg"
        //生成验证码
        val capText = captchaProducer.createText()
        session.setAttribute(com.google.code.kaptcha.Constants.KAPTCHA_SESSION_KEY, capText)
        //向客户端写出
        val bi = captchaProducer.createImage(capText);
        ImageIO.write(bi, "JPEG", response.outputStream)
        try {
            response.outputStream.flush()
        } catch (ex: Exception) {
            response.outputStream.close()
        }
    }
}

@RestController
class UserInforController {

    @RequestMapping("/api/userinfor")
    fun user(user: Principal): Principal {
        System.out.println(user)
        return user
    }
}


@Controller
class WebMainController {
    @Autowired
    lateinit var functionService: FunctionService
    @Autowired
    lateinit var commonService: CommonService

    private val logger = KotlinLogging.logger {}

    @GetMapping("/login")
    fun loginView() = "login"

    @RequestMapping("/third/logout")
    fun oauthLogout(request: HttpServletRequest, response: HttpServletResponse) {
        val back = request.getParameter("redirect_uri")
        SecurityContextLogoutHandler().logout(request, null, null);
        try {
            SecurityContextHolder.getContext().authentication = null
            if (back != null) {
                response.sendRedirect(back)
            } else {
                logger.debug { request.getHeader("referer") }
                response.sendRedirect(request.getHeader("referer"))
            }
        } catch (e: IOException) {
            e.printStackTrace()
        }
    }

    @GetMapping("/logout")
    fun logout(request: HttpServletRequest, response: HttpServletResponse): String {
        SecurityContextHolder.getContext().authentication?.also {
            logger.debug { "user logout!! " }
            SecurityContextLogoutHandler().logout(request, response, it)
        }
        return "redirect:/login?logout"
    }

    @GetMapping(value = ["/", "/index"])
    fun menuView(@AuthenticationPrincipal operUser: UserDetails, model: Model): String {
        model.addAttribute("loginOper", operUser as TOperator)
        val funclist = functionService.getFunctionsByOperid(operUser.operid)
        model.addAttribute("menus", functionService.getMenuTree(funclist, -1))
        model.addAttribute("waterVersion", commonService.getSystemVersion())
        return "index"
    }

}

@Controller
@RequestMapping("/home")
class HomeController {
    /**
     * 控制台
     */
    @GetMapping("/console")
    fun console(): String {
        return "home/console"
    }

    /**
     * 消息弹窗
     */
    @GetMapping("/message")
    fun message(): String {
        return "home/message"
    }

    /**
     * 修改密码弹窗
     */
    @GetMapping("/password")
    fun password(): String {
        return "home/password"
    }

    /**
     * 主题设置弹窗
     */
    @GetMapping("/theme")
    fun theme(): String {
        return "home/theme"
    }

    /**
     * 设置主题
     */
    @RequestMapping("/setTheme")
    fun setTheme(themeName: String?, request: HttpServletRequest): String {
        if (null == themeName) {
            request.session.removeAttribute("theme")
        } else {
            request.session.setAttribute("theme", themeName)
        }
        return "redirect:/"
    }

}
