package com.supwisdom.dlpay.water

import com.supwisdom.dlpay.exception.RequestParamCheckException
import com.supwisdom.dlpay.exception.TransactionCheckException
import com.supwisdom.dlpay.exception.TransactionException
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.service.CommonService
import com.supwisdom.dlpay.framework.util.APIRequestParam
import com.supwisdom.dlpay.framework.util.TradeErrorCode
import mu.KotlinLogging
import org.aspectj.lang.ProceedingJoinPoint
import org.aspectj.lang.annotation.Around
import org.aspectj.lang.annotation.Aspect
import org.aspectj.lang.annotation.Pointcut
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.http.ResponseEntity
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Component
import org.springframework.web.bind.annotation.ExceptionHandler
import org.springframework.web.bind.annotation.RestControllerAdvice
import java.lang.reflect.UndeclaredThrowableException
import javax.servlet.http.HttpServletRequest


@RestControllerAdvice("com.supwisdom.dlpay.water")
class RestControllerAdvice {
    val logger = KotlinLogging.logger { }

    @ExceptionHandler
    fun handleException(ex: Exception, request: HttpServletRequest): ResponseEntity<Any> {
        logger.error { "Request unhandler exception, url<${request.requestURI}>, ex<${ex.cause}>" }
        val undeclared = if (ex is UndeclaredThrowableException) ex.undeclaredThrowable else ex
        if (undeclared is RequestParamCheckException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .requestException(undeclared, "请求参数错误"))
        } else if (undeclared is TransactionException) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(undeclared.code(), undeclared, "业务处理错误"))
        }

        return ResponseEntity.ok().body(ResponseBodyBuilder.create()
                .exception(TradeErrorCode.BUSINESS_DEAL_ERROR, undeclared, "业务处理报错"))
    }
}

@Component
@Aspect
class RestControllerAspect {
    @Autowired
    lateinit var commonService: CommonService

    @Value("\${security.request.sign: true}")
    private var requestSignCheck: Boolean = true

    private val logger = KotlinLogging.logger { }

    @Pointcut("@within(org.springframework.web.bind.annotation.RestController)")
    fun restcontroller() {
    }

    @Pointcut("@annotation(org.springframework.web.bind.annotation.PostMapping)")
    fun postmapping() {
    }

    @Around("restcontroller() && postmapping() && args(@RequestBody body, ..)")
    @Throws(Throwable::class)
    fun logPostMethods(joinPoint: ProceedingJoinPoint, body: Any): Any {
        return try {
            if (body is APIRequestParam) {
                body.checkParam()

//                if (requestSignCheck &&
//                        !body.checkSign(commonService.getSecretByAppid(
//                                SecurityContextHolder.getContext().authentication.name))) {
//                    throw RequestParamCheckException(TradeErrorCode.REQUEST_SIGN_ERROR, "参数签名错误")
//                }
            } else {
                throw TransactionCheckException(TradeErrorCode.REQUEST_PARAM_EEROR, "请求参数实体位置错误")
            }

            val result = joinPoint.proceed()
            logger.debug(result.toString())
            result
        } catch (t: Throwable) {
            logger.error { "request error , ex<${t.message}>" }
            throw t
        }
    }
}
