package com.supwisdom.dlpay.water.controller

import com.supwisdom.dlpay.api.bean.QueryUserParam
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.service.BusinessparaService
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.paysdk.proxy.UserProxy
import com.supwisdom.dlpay.system.service.UserDataService
import com.supwisdom.dlpay.water.*
import com.supwisdom.dlpay.water.domain.TCollectdtl
import com.supwisdom.dlpay.water.pay.WaterPayAsyncTask
import com.supwisdom.dlpay.water.service.AreaService
import com.supwisdom.dlpay.water.service.DeviceService
import com.supwisdom.dlpay.water.service.CollectdtlService
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.*
import org.springframework.web.servlet.ModelAndView


@RestController
@RequestMapping("/api/device")
class WaterApiController {

    @Autowired
    private lateinit var systemUtilService: SystemUtilService

    @Autowired
    private lateinit var deviceService: DeviceService

    @Autowired
    private lateinit var userDataService: UserDataService

    @Autowired
    private lateinit var collectdtlService: CollectdtlService

    @Autowired
    private lateinit var shortURLUtil: ShortURLUtil

    @Autowired
    private lateinit var businessparaService: BusinessparaService

    @Autowired
    private lateinit var waterPayAsyncTask: WaterPayAsyncTask

    @Autowired
    private lateinit var areaService: AreaService

    @Autowired
    private lateinit var userProxy: UserProxy

    private val logger = KotlinLogging.logger { }

    @PostMapping("/devicelogin")
    fun deviceLogin(param: DeviceLoginParam): ResponseEntity<Any> {
        try {
            val result = deviceService.deviceLogin(param)
            if (result["flag"] == true) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data(WaterDeviceParam.devOfflineMaxHour, result[WaterDeviceParam.devOfflineMaxHour]!!)
                        .data(WaterDeviceParam.pulseInHML, result[WaterDeviceParam.pulseInHML]!!)
                        .data(WaterDeviceParam.systemDate, systemUtilService.sysdatetime.hostdate)
                        .data(WaterDeviceParam.systemTime, systemUtilService.sysdatetime.hosttime)
                        .success())
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(WaterErrorCode.DATA_NOTFOUND_ERROR, result["errorMsg"].toString()))
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "系统处理异常"))
        }
    }


    @RequestMapping("/linecheck", method = [RequestMethod.GET, RequestMethod.POST])
    fun deviceLineCheck(param: DeviceLineCheckParam): ResponseEntity<Any> {
        try {
            val device = deviceService.lineCheck(param)
                    ?: return ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(WaterErrorCode.DATA_NOTFOUND_ERROR, "没有编号为${param.deviceno}的设备"))
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data(WaterDeviceParam.systemDate, systemUtilService.sysdatetime.hostdate)
                    .data(WaterDeviceParam.systemTime, systemUtilService.sysdatetime.hosttime)
                    .data(WaterDeviceParam.deviceStatus, device.deviceStatus)
                    .success())
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, ex.message))
        }
    }

    @PostMapping("/card/purseinit")
    fun cardPurseInit(param: CardPayRequest): ResponseEntity<Any> {
        try {//1. 通过 citizenCardno 检查用户以及合法性
            val personIdentity = userDataService.getPersonIdentityByThirdUid(param.citizenCardno)
                    ?: return ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(WaterErrorCode.DATA_NOTFOUND_ERROR, "未查询到您的身份信息"))
            if (personIdentity.cardphyId == null || personIdentity.cardphyId != param.cardphyid) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.DATA_NOTFOUND_ERROR, "物理卡号不匹配"))
            }
            //2. 通过 deviceno 查询设备费率参数
            val deviceParam = deviceService.getParaMapByDeviceno(param.deviceno)
            //3. 创建 collectdtl 记录初始流水
            val trans = TCollectdtl().apply {
                mode = TradeDict.PAY_MODE_CARD
                transDate = param.termdate
                transTime = param.termtime
                deviceno = param.deviceno
                userid = personIdentity.person.userid
                citizenCardno = param.citizenCardno
                cardPhyId = param.cardphyid
                amount = 0.0
                waterSumHundredLitre = 0
                status = TradeDict.DTL_STATUS_INIT
                authStatus = true
                uploadStatus = false
            }
            val savedTrans = collectdtlService.createNewTransdtl(trans)
            //4. 将流水 cobillno 和费率信息返回给终端
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data(WaterDeviceParam.cobillNo, savedTrans.cobillno)
                    .data(WaterDeviceParam.feeAmount, deviceParam[WaterDeviceParam.feeAmount]!!)
                    .data(WaterDeviceParam.waterLimit, deviceParam[WaterDeviceParam.waterLimit]!!)
                    .data(WaterDeviceParam.feestart, deviceParam[WaterDeviceParam.feestart]!!)
                    .data(WaterDeviceParam.feeUnit, deviceParam[WaterDeviceParam.feeUnit]!!)
                    .success())
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "系统出错"))
        }
    }

    @PostMapping("/qrcode/init")
    fun qrcodePayInit(param: QrcodePayRequest): ResponseEntity<Any> {
        try {
            val deviceParam = deviceService.getParaMapByDeviceno(param.deviceno)
            //1. 创建并记录初始流水
            val trans = TCollectdtl().apply {
                mode = TradeDict.PAY_MODE_QRCODE
                transDate = param.termdate
                transTime = param.termtime
                deviceno = param.deviceno
                amount = 0.0
                waterSumHundredLitre = 0
                status = TradeDict.DTL_STATUS_INIT
                authStatus = false
                uploadStatus = false
            }
            val savedTrans = collectdtlService.createNewTransdtl(trans)
            //2.将流水号及认证地址返回给终端
            //将认证url转为短码
            val url = shortURLUtil.doGetShortUrl(
                    businessparaService.findByParakey(WaterBudinessConstants.waterAuthUrl).paraval + savedTrans.cobillno)
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data(WaterDeviceParam.cobillNo, savedTrans.cobillno)
                    .data(WaterDeviceParam.url, url)
                    .data(WaterDeviceParam.validTime, deviceParam[WaterDeviceParam.validTime]!!)
                    .success())
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "系统出错"))
        }
    }

    @PostMapping("/qrcode/query")
    fun qrcodeQuery(param: QrcodeQueryRequest): ResponseEntity<Any> {
        try {
            val deviceParam = deviceService.getParaMapByDeviceno(param.deviceno)
            val trans = collectdtlService.queryTrans(param.cobillno)
            var authStatus = 0
            if (trans.authStatus) {
                authStatus = 1
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data(WaterDeviceParam.cobillNo, trans.cobillno)
                    .data(WaterDeviceParam.authStatus, authStatus)
                    //2为代扣模式
                    .data(WaterDeviceParam.payStatus, 2)
                    .data(WaterDeviceParam.waterLimit, deviceParam[WaterDeviceParam.waterLimit]!!)
                    .data(WaterDeviceParam.feeAmount, deviceParam[WaterDeviceParam.feeAmount]!!)
                    .data(WaterDeviceParam.feestart, deviceParam[WaterDeviceParam.feestart]!!)
                    .data(WaterDeviceParam.feeUnit, deviceParam[WaterDeviceParam.feeUnit]!!)
                    //  如果订单为已支付状态，支付金额多少
                    .data(WaterDeviceParam.paidAmount, 0)
                    .success())
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, ex.message))
        }
    }

    /**
     * 通过userid查询用户卡片信息并记录在流水中
     */
    @GetMapping("/qrcode/pretend")
    fun confirmView(@RequestParam("cobillno") cobillno: Int,
                    @RequestParam("userid") userid: String): ModelAndView {
        try {
            val userInfo = userProxy.querybycardno(QueryUserParam().apply {
                this.userid = userid
            })
            if (userInfo.retcode != 0) {
                logger.error { "获取用户${userid}信息失败,错误信息:${userInfo.retmsg}" }
                return ModelAndView("/error/500").addObject("errorMsg", "用户信息异常")
            }
            if (userInfo.cardstatus != TradeDict.STATUS_NORMAL || userInfo.transstatus != TradeDict.STATUS_NORMAL) {
                return ModelAndView("/error/500").addObject("errorMsg", "用户消费状态异常")
            }
            val collectDtl = collectdtlService.pretendAsCard(userInfo, cobillno)
            if (collectDtl.status != TradeDict.DTL_STATUS_INIT) {
                return ModelAndView("/error/500").addObject("errorMsg", "流水状态异常")
            }
            val device = deviceService.queryDeviceByDeviceno(collectDtl.deviceno)
                    ?: return ModelAndView("/error/500").addObject("errorMsg", "设备信息异常")
            val area = areaService.findByAreaNo(device.areano)
                    ?: return ModelAndView("/error/500").addObject("errorMsg", "区域信息异常")
            return ModelAndView("/system/start/start").
                    addObject("devicename", device.devicename).
                    addObject("areaname",area.areaName)
        } catch (e: Exception) {
            logger.error { e.message }
            return ModelAndView("/error/500")
        }
    }

    /**
     * 手机控制开始出水
     */
    @PostMapping("/qrcode/start")
    @ResponseBody
    fun qrcodeStart(@RequestBody param: StartWaterRequest): ResponseEntity<Any> {
        return try {
            collectdtlService.startWater(param)
            ResponseEntity.ok(ResponseBodyBuilder.create().success())
        } catch (ex: Exception) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "出水失败，请稍后重试或联系管理员!"))
        }
    }


    @PostMapping("/uploadrecord")
    fun transdtlUpload(param: UploadRecordRequest): ResponseEntity<Any> {
        // 1. 根据 cobillno 查询 collectdtl , 并加锁
        val querycodtl = collectdtlService.queryTrans(param.cobillno)
                ?: return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.DATA_NOTFOUND_ERROR, "采集流水号" + param.cobillno + "未找到"))
        if (param.deviceno != querycodtl.deviceno) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(WaterErrorCode.REQUEST_DATA_ERROR, "设备号" + param.deviceno + "与采集流水号" + param.cobillno + "不匹配"))
        }
        //  流水是否重复上传
        if (!querycodtl.uploadStatus) {
            val dtl = collectdtlService.saveDeviceDtlData(param)
            userDataService.updateCoamount(dtl.citizenCardno, param.amount / 100.0)
            //  是否立即扣费
            if (param.amount >= deviceService.getParaMapByDeviceno(param.deviceno)[WaterDeviceParam.imdDecThreshold]!!.toInt()) {
                //  立即异步扣费
                waterPayAsyncTask.waterPay(querycodtl.cobillno)
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create().data(WaterDeviceParam.cobillNo, dtl.cobillno)
                    .success())
        }
        return ResponseEntity.ok(ResponseBodyBuilder.create().data(WaterDeviceParam.cobillNo, querycodtl.cobillno)
                .success())
    }
}