package com.supwisdom.dlpay.water.controller

import com.supwisdom.dlpay.api.bean.JsonResult
import com.supwisdom.dlpay.framework.ResponseBodyBuilder
import com.supwisdom.dlpay.framework.util.*
import com.supwisdom.dlpay.system.service.UserDataService
import com.supwisdom.dlpay.util.WebCheckException
import com.supwisdom.dlpay.water.bean.*
import com.supwisdom.dlpay.water.domain.*
import com.supwisdom.dlpay.water.pojo.*
import com.supwisdom.dlpay.water.service.*
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.ResponseEntity
import org.springframework.stereotype.Controller
import org.springframework.ui.Model
import org.springframework.web.bind.annotation.*

@Controller
class RegionController {

    @Autowired
    lateinit var areaService: AreaService

    @Autowired
    lateinit var deviceService: DeviceService

    @GetMapping("/region/index")
    fun indexView() = "system/region/index"

    @GetMapping("/region/loadadd")
    fun loadadd() = "system/region/form"

    @GetMapping("/region/all")
    @ResponseBody
    fun queryAllRegion(): ResponseEntity<Any> {
        try {
            areaService.queryAllAreas()?.let {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("regions", it)
                        .success())
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create().fail(WaterErrorCode.DATA_NOTFOUND_ERROR, "区域未找到"))
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @GetMapping("/region/list/{parentId}")
    @ResponseBody
    fun queryRegions(@PathVariable parentId: Int,
                     @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
                     @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TArea> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val searchBean = RegionSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
                this.parentId = parentId
            }
            areaService.queryAreasByParentId(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "区域未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }

    @GetMapping("/region/areano")
    @ResponseBody
    fun queryAreaByAreano(@RequestParam("areano") areano: Int): PageResult<TArea> {
        try {
            if (areano < 0) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            areaService.queryAreaByAreaNo(areano)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "区域未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }

    @GetMapping("/region/search")
    @ResponseBody
    fun queryRegions(@RequestParam(value = "searchkey", required = false) searchKey: String?,
                     @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
                     @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TArea> {

        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val searchBean = RegionSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
                this.searchKey = searchKey
            }
            areaService.queryAreasBySearchKey(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "区域未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }

    @GetMapping("/region/tree")
    @ResponseBody
    fun getRegionTree(): ResponseEntity<Any> {
        try {
            areaService.queryAllAreas()?.let { it ->
                areaService.getAreaTree(it, 0).let {
                    return ResponseEntity.ok(ResponseBodyBuilder.create()
                            .data("regions", it)
                            .success())
                }
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data("regions", "")
                    .success())
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @PostMapping("/region/add")
    @ResponseBody
    fun addRegion(@RequestBody area: TArea): ResponseEntity<Any> {
        try {
            if (null == area.areaName) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.REQUEST_DATA_ERROR, "区域名不能为空"))
            }
            if (!areaService.checkAreaName(area)) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.REQUEST_DATA_ERROR, "区域名已存在"))
            }

            if (null == area.areano) {
                return if (areaService.saveArea(area)) {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .data("msg", "新增成功")
                            .success())
                } else {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(WaterErrorCode.PROCESS_ERROR, "新增失败"))
                }
            } else {
                return if (areaService.updateArea(area)) {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .data("msg", "更新成功")
                            .success())
                } else {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(WaterErrorCode.PROCESS_ERROR, "更新失败"))
                }
            }
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @PostMapping("/region/del")
    @ResponseBody
    fun del(@RequestParam("areano") areano: Int): ResponseEntity<Any> {
        try {
            if (areaService.existSubArea(areano)) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.REQUEST_DATA_ERROR, "该区域存在子区域，删除失败"))
            }
            if (deviceService.existSubDevice(areano)) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.REQUEST_DATA_ERROR, "该区域有关联设备，删除失败"))
            }
            return if (areaService.deleteArea(areano)) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("msg", "删除成功")
                        .success())
            } else {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.PROCESS_ERROR, "删除失败"))
            }
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }
}

@Controller
class DeviceController {

    @Autowired
    lateinit var deviceService: DeviceService


    @GetMapping("/device/index")
    fun indexView() = "system/device/index"


    @GetMapping("/device/loadadd")
    fun loadadd() = "system/device/form"

    @GetMapping("/device/list")
    @ResponseBody
    fun queryDevices(
            @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
            @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TDevice> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val searchBean = DeviceSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
            }
            deviceService.queryDeviceList(searchBean)?.let {
                return it
            }
            return PageResult(404, "区域未找到")
        } catch (ex: Exception) {
            return PageResult(500, "服务器繁忙,请稍后再试")
        }
    }

    @PostMapping("/device/add")
    @ResponseBody
    fun addDevice(@RequestBody device: TDevice): ResponseEntity<Any> {
        try {
            val checkResult = deviceService.checkParams(device)
            if (checkResult["flag"] == false) {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(TradeErrorCode.INPUT_DATA_ERROR, checkResult["msg"].toString()))
            }
            if (checkResult["type"].toString() == "update") {
                return if (deviceService.updateDevice(device)) {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .data("msg", "更新成功")
                            .success())
                } else {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(TradeErrorCode.PROCESS_ERROR, "更新失败"))
                }
            } else {
                return if (deviceService.saveDevice(device)) {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .data("msg", "新增成功")
                            .success())
                } else {
                    ResponseEntity.ok(ResponseBodyBuilder.create()
                            .fail(WaterErrorCode.PROCESS_ERROR, "新增失败"))
                }
            }
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @GetMapping("/device/devino/{deviceid}")
    @ResponseBody
    fun checkDevino(@PathVariable deviceid: Int): ResponseEntity<Any> {
        try {
            deviceService.queryDeviceById(deviceid)?.let {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("flag", true)
                        .success())
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .data("flag", false)
                    .success())
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @PostMapping("/device/del")
    @ResponseBody
    fun del(@RequestParam deviceid: Int): ResponseEntity<Any> {
        return try {
            if (deviceService.deleteDevice(deviceid)) {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("msg", "删除成功")
                        .success())
            } else {
                ResponseEntity.ok(ResponseBodyBuilder.create()
                        .fail(WaterErrorCode.PROCESS_ERROR, "删除失败"))
            }
        } catch (ex: Exception) {
            ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @GetMapping("/device/status")
    @ResponseBody
    fun allStatus(): ResponseEntity<Any> {
        try {
            deviceService.groupStatus()?.let {
                return ResponseEntity.ok(ResponseBodyBuilder.create()
                        .data("status", it)
                        .success())
            }
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .fail(WaterErrorCode.PROCESS_ERROR, "查询失败"))
        } catch (ex: Exception) {
            return ResponseEntity.ok(ResponseBodyBuilder.create()
                    .exception(WaterErrorCode.PROCESS_ERROR, ex, "服务器繁忙,请稍后再试"))
        }
    }

    @GetMapping("/device/search")
    @ResponseBody
    fun queryRegions(@RequestParam("devicename", required = false) devicename: String,
                     @RequestParam("deviceno", required = false) deviceno: String,
                     @RequestParam("areano", required = false) areanoStr: String,
                     @RequestParam("deviceStatus", required = false) deviceStatus: String,
                     @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
                     @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TDevice> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val areano: Int?
            areano = if (StringUtil.isEmpty(areanoStr)) {
                null
            } else {
                areanoStr.toInt()
            }
            val searchBean = DeviceSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
                this.devicename = devicename
                this.deviceno = deviceno
                this.areano = areano
                this.deviceStatus = deviceStatus
            }
            deviceService.queryDeviceByParams(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "设备未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }
}

@Controller
class CollectdtlController {
    @Autowired
    private lateinit var collectdtlService: CollectdtlService

    @GetMapping("/collectdtl/index")
    fun dtlIndexView() = "system/collectdtl/index"

    @GetMapping("/collectdtl/list")
    @ResponseBody
    fun queryTransdtl(@RequestParam("devicename", required = false) devicename: String,
                      @RequestParam("deviceno", required = false) deviceno: String,
                      @RequestParam("areano", required = false) areanoStr: String,
                      @RequestParam("transdate", required = false) transdate: String,
                      @RequestParam("accdate", required = false) accdate: String,
                      @RequestParam("username", required = false) username: String,
                      @RequestParam("notacc", required = false) notacc: Boolean,
                      @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
                      @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TCollectdtlDTO> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val areano: Int?
            areano = if (StringUtil.isEmpty(areanoStr)) {
                null
            } else {
                areanoStr.toInt()
            }
            val searchBean = TransdtlSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
                this.devicename = devicename
                this.deviceno = deviceno
                this.areano = areano
                this.username = username
                this.transdate = transdate
                this.accdate = accdate
                this.notacc = notacc
            }
            collectdtlService.queryTransdtlDTOByParam(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "采集流水未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }

    @GetMapping("/dtlcount/index")
    fun dtlCountIndexView() = "system/dtlcount/index"

    @GetMapping("/dtlcount/list")
    @ResponseBody
    fun queryDtlCount(@RequestParam("devicename", required = false) devicename: String?,
                      @RequestParam("areano", required = false) areanoStr: String?,
                      @RequestParam("accdate", required = false) accdate: String?,
                      @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
                      @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TTransdtlCountVO> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val areano: Int?
            areano = if (StringUtil.isEmpty(areanoStr)) {
                null
            } else {
                areanoStr!!.toInt()
            }
            val searchBean = TransdtlCountSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
                this.devicename = devicename
                this.areano = areano
                this.accdate = accdate
            }
            collectdtlService.queryTransdtlCountByParam(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "流水统计未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "系统繁忙")
        }
    }
}

@Controller
class AreaparaController {

    @Autowired
    private lateinit var tAreaparaService: TAreaparaService

    @GetMapping("/areapara/index")
    fun areaparaIndexView() = "system/areapara/index"

    @GetMapping("/areapara/grouplist")
    @ResponseBody
    fun queryGroupList(@RequestParam("groupname", required = false) groupname: String?,
                       @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
                       @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int): PageResult<TAreaparaGroup> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val searchBean = AreaparaSearchBean().apply {
                this.groupname = groupname
                this.pageNo = pageNo
                this.pageSize = pageSize
            }
            tAreaparaService.queryParaGroupByParam(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "设备参数组未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }

    @GetMapping("/areapara/loadadd")
    fun editView(@RequestParam("groupid", required = false) groupid: Int?, model: Model): String {
        tAreaparaService.getAreaparaGroupByGroupid(groupid ?: -99)?.let {
            model.addAttribute("areaparaGroup", it)
            model.addAttribute("areaparaConfigList", tAreaparaService.getAreaparaInfo(it.groupid))
            return "system/areapara/form"
        }
        val group = TAreaparaGroup()
        group.groupid = 0
        model.addAttribute("areaparaGroup", group)
        model.addAttribute("areaparaConfigList", tAreaparaService.getAreaparaInfo(0))
        return "system/areapara/form"
    }

    @PostMapping("/areapara/editareapara")
    @ResponseBody
    fun editAreapara(@RequestBody param: HashMap<String, String>?): JsonResult {
        val groupidHtmlKey = "waterform_areapara_groupid"   //页面上传来groupid的KEY
        val groupnameHtmlKey = "waterform_areapara_groupname" //页面上传来groupname的KEY
        if (null == param || !NumberUtil.isDigits(param[groupidHtmlKey]) || StringUtil.isEmpty(groupnameHtmlKey)) {
            return JsonResult.error("参数传递错误")
        }
        try {
            val groupid = Integer.parseInt(param[groupidHtmlKey])
            val groupname = param[groupnameHtmlKey]?.trim { it <= ' ' }
            param.remove(groupidHtmlKey)
            param.remove(groupnameHtmlKey)
            val msg: String
            return if (tAreaparaService.saveOrUpdateAreapara(groupid, groupname, param)) {
                msg = if (groupid == 0) {
                    "新增成功"
                } else {
                    "修改成功"
                }
                JsonResult.ok(msg)
            } else {
                msg = if (groupid == 0) {
                    "新增失败"
                } else {
                    "修改失败"
                }
                JsonResult.ok(msg)
            }
        } catch (ex: WebCheckException) {
            return JsonResult.error(ex.message)
        } catch (ex: Exception) {
            ex.printStackTrace()
            return JsonResult.error("系统处理异常").put("exception", ex)!!
        }
    }

    @PostMapping("/areapara/deleteareapara")
    @ResponseBody
    fun deleteAreapara(@RequestParam("groupid") groupid: Int?): JsonResult {
        try {
            if (null == groupid) {
                return JsonResult.error("参数传递错误")
            }
            val group = tAreaparaService.getAreaparaGroupByGroupid(groupid) ?: return JsonResult.error("参数组不存在！")
            return if (tAreaparaService.deleteAreaparaGroup(group)) {
                JsonResult.ok("删除成功")
            } else {
                JsonResult.error("删除失败")
            }
        } catch (ex: WebCheckException) {
            return JsonResult.error(ex.message)
        } catch (ex: Exception) {
            ex.printStackTrace()
            return JsonResult.error("系统处理异常").put("exception", ex)!!
        }
    }

    @GetMapping("/areaparabind/index")
    fun parambindIndexView():String{
        return "system/areaparabind/index"
    }

    @GetMapping("/areapara/areaparabindlist")
    @ResponseBody
    fun getAreaparaBindData(
            @RequestParam("page", defaultValue = "1", required = false) pageNo: Int,
            @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int,
            @RequestParam(value = "searchkey", required = false) searchkey: String?): PageResult<TAreaparaBindDTO> {
        try {
            if (pageNo < 1 || pageSize < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            return tAreaparaService.getAreaparaBindInfos(searchkey, pageNo, pageSize)
        } catch (ex: Exception) {
            ex.printStackTrace()
            return PageResult(WaterErrorCode.PROCESS_ERROR, "系统繁忙，请稍后重试")
        }
    }

    @GetMapping("/areapara/loadbindadd")
    fun loadBindView(model: Model): String {
        model.addAttribute("grouplist", tAreaparaService.queryAllAreaparaGroups())
        return "system/areaparabind/form"
    }

    @PostMapping("/areapara/addbindareapara")
    @ResponseBody
    fun addAreaparaBind(@RequestParam("groupid") groupid: Int,
                        @RequestParam("areano[]") areanos: List<Int>): JsonResult? {
        return try {
            if (tAreaparaService.saveBindAreapara(groupid, areanos)) {
                JsonResult.ok("绑定成功")
            } else {
                JsonResult.ok("绑定失败")
            }
        } catch (ex: WebCheckException) {
            JsonResult.error(ex.message)
        } catch (ex: Exception) {
            ex.printStackTrace()
            JsonResult.error("系统处理异常").put("exception", ex)
        }
    }

    @PostMapping("/areapara/deleteareaparabind")
    @ResponseBody
    fun deleteAreaparaBind(@RequestParam("areano") areano: Int): JsonResult? {
        try {
            val bind = tAreaparaService.getAreaparaBindByAreano(areano) ?: return JsonResult.error("区域参数组绑定关系不存在")
            return if (tAreaparaService.deleteAreaparaBind(bind)) {
                JsonResult.ok("删除成功")
            } else {
                JsonResult.error("删除失败")
            }
        } catch (ex: Exception) {
            ex.printStackTrace()
            return JsonResult.error("系统处理异常").put("exception", ex)
        }
    }

    @GetMapping("/areapara/arealist")
    @ResponseBody
    fun getAreaList(@RequestParam("searchkey") searchKey: String?): PageResult<TArea> {
        return try {
            tAreaparaService.getAreaBySearch(searchKey)
        } catch (ex: Exception) {
            PageResult(WaterErrorCode.PROCESS_ERROR, "服务器繁忙,请稍后再试")
        }
    }
}

@Controller
class AccdtlController {

    @Autowired
    private lateinit var accdtlService: AccdtlService

    @GetMapping("/accdtl/index")
    fun dtlIndexView() = "system/accdtl/index"

    @GetMapping("/accdtl/list")
    @ResponseBody
    fun queryAccdtl(@RequestParam("devicename", required = false) devicename: String?,
                    @RequestParam("deviceno", required = false) deviceno: String?,
                    @RequestParam("areano", required = false) areano: Int?,
                    @RequestParam("transdate", required = false) transdate: String?,
                    @RequestParam("accdate", required = false) accdate: String?,
                    @RequestParam("username", required = false) username: String?,
                    @RequestParam("page", defaultValue = "1", required = false) pageNo: Int?,
                    @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int?): PageResult<TAccdtlDTO> {
        try {
            if (pageNo!! < 1 || pageSize!! < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val searchBean = TransdtlSearchBean().apply {
                this.pageNo = pageNo
                this.pageSize = pageSize
                this.devicename = devicename
                this.deviceno = deviceno
                this.areano = areano
                this.username = username
                this.transdate = transdate
                this.accdate = accdate
            }
            accdtlService.queryAccdtlByParam(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "采集流水未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "系统繁忙，请稍后重试")
        }
    }
}

@Controller
class UserAccountController {
    @Autowired
    private lateinit var userDataService: UserDataService

    @GetMapping("useracc/index")
    fun userAccIndexView() = "system/useraccount/index"

    @GetMapping("useracc/list")
    @ResponseBody
    fun queryUserAcc(@RequestParam("username", required = false) username: String?,
                     @RequestParam("cardno", required = false) cardno: String?,
                     @RequestParam("page", defaultValue = "1", required = false) pageNo: Int?,
                     @RequestParam("limit", defaultValue = "10", required = false) pageSize: Int?): PageResult<UserAccountDTO> {
        try {
            if (pageNo!! < 1 || pageSize!! < 1) {
                return PageResult(WaterErrorCode.REQUEST_DATA_ERROR, "请求参数错误")
            }
            val searchBean = UserAccountSearchBean().apply {
                this.username = username
                this.cardno = cardno
            }
            userDataService.queryUserAccByParam(searchBean)?.let {
                return it
            }
            return PageResult(WaterErrorCode.DATA_NOTFOUND_ERROR, "采集流水未找到")
        } catch (ex: Exception) {
            return PageResult(WaterErrorCode.PROCESS_ERROR, "系统繁忙，请稍后重试")
        }
    }
}
