package com.supwisdom.dlpay.water.service

import com.supwisdom.dlpay.api.bean.UserInforResponse
import com.supwisdom.dlpay.exception.TransactionProcessException
import com.supwisdom.dlpay.framework.service.SystemUtilService
import com.supwisdom.dlpay.framework.util.PageResult
import com.supwisdom.dlpay.framework.util.StringUtil
import com.supwisdom.dlpay.framework.util.TradeDict
import com.supwisdom.dlpay.framework.util.WaterErrorCode
import com.supwisdom.dlpay.water.StartWaterRequest
import com.supwisdom.dlpay.water.UploadRecordRequest
import com.supwisdom.dlpay.water.bean.TransdtlCountSearchBean
import com.supwisdom.dlpay.water.dao.CollectdtlDao
import com.supwisdom.dlpay.water.domain.TCollectdtl
import com.supwisdom.dlpay.water.pojo.TCollectdtlDTO
import com.supwisdom.dlpay.water.bean.TransdtlSearchBean
import com.supwisdom.dlpay.water.dao.DtlCountDateDao
import com.supwisdom.dlpay.water.dao.TransdtlCountDao
import com.supwisdom.dlpay.water.domain.TTransdtlCount
import com.supwisdom.dlpay.water.domain.TdtlCountDate
import com.supwisdom.dlpay.water.pojo.TTransdtlCountDTO
import com.supwisdom.dlpay.water.pojo.TTransdtlCountVO
import mu.KotlinLogging
import org.hibernate.query.internal.NativeQueryImpl
import org.hibernate.transform.Transformers
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service
import java.lang.IllegalStateException
import java.lang.RuntimeException
import java.math.BigInteger
import java.text.SimpleDateFormat
import java.util.*
import javax.persistence.EntityManager

@Service
class CollectdtlServiceImpl : CollectdtlService {

    private val logger = KotlinLogging.logger { }

    @Autowired
    private lateinit var collectdtlDao: CollectdtlDao

    @Autowired
    private lateinit var em: EntityManager

    @Autowired
    private lateinit var transdtlCountDao: TransdtlCountDao

    @Autowired
    private lateinit var dtlCountDateDao: DtlCountDateDao

    @Autowired
    private lateinit var systemUtilsService: SystemUtilService


    override fun createNewTransdtl(dtl: TCollectdtl): TCollectdtl {
        collectdtlDao.save(dtl)
        return dtl
    }

    override fun saveDeviceDtlData(record: UploadRecordRequest): TCollectdtl {
        val dtl = collectdtlDao.findByCobillnoForUpdate(record.cobillno)
                ?: throw TransactionProcessException(WaterErrorCode.DATA_NOTFOUND_ERROR,
                        "交易订单号不存在")
        if (record.transtatus == "2") {
            dtl.status = TradeDict.DTL_STATUS_FAIL
        } else if (record.transtatus == "1") {
            dtl.status = TradeDict.DTL_STATUS_WIP
        }
        dtl.amount = record.amount / 100.0
        dtl.waterSumHundredLitre = record.flowsensors
        dtl.uploadTime = record.transdate + record.transtime
        dtl.uploadStatus = true
        collectdtlDao.save(dtl)
        return dtl
    }

    override fun queryTrans(cobillno: Int?): TCollectdtl? {
        return try {
            collectdtlDao.findByCobillnoForUpdate(cobillno)
        } catch (ex: IllegalStateException) {
            null
        }
    }

    override fun startWater(param: StartWaterRequest): TCollectdtl {
        val dtl = collectdtlDao.findByCobillnoForUpdate(param.cobillno)
                ?: throw TransactionProcessException(WaterErrorCode.DATA_NOTFOUND_ERROR,
                        "交易订单号不存在")
        dtl.authStatus = true
        collectdtlDao.save(dtl)
        return dtl
    }

    override fun pretendAsCard(param: UserInforResponse, cobillno: Int?): TCollectdtl {
        val dtl = collectdtlDao.findByCobillnoForUpdate(cobillno)
                ?: throw TransactionProcessException(WaterErrorCode.DATA_NOTFOUND_ERROR,
                        "交易订单号不存在")
        dtl.userid = param.userid
        dtl.citizenCardno = param.cardno
        dtl.cardPhyId = param.cardphyid
        collectdtlDao.save(dtl)
        return dtl
    }

    override fun queryTransdtlDTOByParam(param: TransdtlSearchBean): PageResult<TCollectdtlDTO>? {
        val sql = StringBuffer("select t1.cobillno,t1.amount,t1.citizencardno,t1.cardphyid,t1.deviceno,t1.mode,t1.status,t1.transdate,t1.entryno,t1.transtime,t1.accdate,t1.water_in_100ml water_sum_hundred_litre,t1.name username,t2.devicename,t2.areaname,t2.areano  " +
                "from (select dtl.*,person.name from tb_collectdtl dtl left join tb_person person on dtl.userid = person.userid) t1," +
                "(select device.deviceno,device.devicename,area.areaname,area.areano from tb_device device,tb_area area where device.areano = area.areano) t2 " +
                "where t1.deviceno = t2.deviceno")
        val countSql = StringBuffer("select count(cobillno) " +
                "from (select dtl.*,person.name from tb_collectdtl dtl left join tb_person person on dtl.userid = person.userid) t1," +
                "(select device.deviceno,device.devicename,area.areaname,area.areano from tb_device device,tb_area area where device.areano = area.areano) t2 " +
                "where t1.deviceno = t2.deviceno")
        val map: HashMap<String, Int> = HashMap()
        var position = 1
        if (param.notacc) {
            sql.append(" and status='wip'")
            countSql.append(" and status='wip'")
        }
        if (!StringUtil.isEmpty(param.transdate)) {
            sql.append(" and transdate >=?")
            countSql.append(" and transdate >=?")

            sql.append(" and transdate <=?")
            countSql.append(" and transdate <=?")
            map["mintransdate"] = position++
            map["maxtransdate"] = position++
        }
        if (!StringUtil.isEmpty(param.accdate)) {
            sql.append(" and accdate >=?")
            countSql.append(" and accdate >=?")

            sql.append(" and accdate <=?")
            countSql.append(" and accdate <=?")
            map["minaccdate"] = position++
            map["maxaccdate"] = position++
        }
        if (!StringUtil.isEmpty(param.username)) {
            sql.append(" and name like ?")
            countSql.append(" and name like ?")
            map["username"] = position++
        }
        if (!StringUtil.isEmpty(param.deviceno)) {
            sql.append(" and t1.deviceno like ?")
            countSql.append(" and t1.deviceno like ?")
            map["deviceno"] = position++
        }
        if (!StringUtil.isEmpty(param.devicename)) {
            sql.append(" and devicename like ?")
            countSql.append(" and devicename like ?")
            map["devicename"] = position++
        }
        param.areano?.let {
            sql.append(" and areano  =?")
            countSql.append(" and areano  =?")
            map.put("areano", position++)
        }
        sql.append(" order by cobillno desc limit " + param.pageSize + " offset " + (param.pageNo - 1) * param.pageSize)
        val query = em.createNativeQuery(sql.toString(), TCollectdtlDTO::class.java)
        val countQuery = em.createNativeQuery(countSql.toString())
        map["devicename"]?.let {
            query.setParameter(it, "%" + param.devicename + "%")
            countQuery.setParameter(it, "%" + param.devicename + "%")
        }
        map["deviceno"]?.let {
            query.setParameter(it, "%" + param.deviceno + "%")
            countQuery.setParameter(it, "%" + param.deviceno + "%")
        }
        map["areano"]?.let {
            query.setParameter(it, param.areano)
            countQuery.setParameter(it, param.areano)
        }
        map["username"]?.let {
            query.setParameter(it, "%" + param.username + "%")
            countQuery.setParameter(it, "%" + param.username + "%")
        }
        map["mintransdate"]?.let {

            val timerange = param.transdate.replace("-", "").split("  ")
            query.setParameter(it, timerange[0])
            query.setParameter(map["maxtransdate"]!!, timerange[1])
            countQuery.setParameter(it, timerange[0])
            countQuery.setParameter(map["maxtransdate"]!!, timerange[1])
        }
        map["minaccdate"]?.let {
            val timerange = param.accdate.replace("-", "").split("  ")
            query.setParameter(it, timerange[0])
            query.setParameter(map["maxaccdate"]!!, timerange[1])
            countQuery.setParameter(it, timerange[0])
            countQuery.setParameter(map["maxaccdate"]!!, timerange[1])
        }
        val collectdtlDTO: List<TCollectdtlDTO> = query.resultList as List<TCollectdtlDTO>
        val count: Int = countQuery.singleResult.toString().toInt()
        val result: PageResult<TCollectdtlDTO> = PageResult()
        result.apply {
            this.data = collectdtlDTO
            this.count = count.toLong()
            this.code = 0
            this.msg = "成功"
        }
        return result
    }

    override fun queryTransdtlCountByParam(param: TransdtlCountSearchBean): PageResult<TTransdtlCountVO> {
        val querySql = StringBuffer("select t1.*,t2.devicename,t2.areaname from (select * from tb_transdtl_count) t1 left join " +
                "(select device.deviceno,device.devicename,area.areaname from tb_device device left join tb_area area on device.areano = area.areano) t2 " +
                "on t1.deviceno=t2.deviceno where 1=1")
        val countSql = StringBuffer("select count(id) from (select * from tb_transdtl_count) t1 left join " +
                "(select device.deviceno,device.devicename,area.areaname from tb_device device left join tb_area area on device.areano = area.areano) t2 " +
                "on t1.deviceno=t2.deviceno where 1=1")
        if (!StringUtil.isEmpty(param.accdate)) {
            querySql.append(" and accdate>:minaccdate")
            querySql.append(" and accdate<:maxaccdate")
            countSql.append(" and accdate>:minaccdate")
            countSql.append(" and accdate<:maxaccdate")
        }
        if (!StringUtil.isEmpty(param.devicename)) {
            querySql.append(" and devicename like :devicename")
            countSql.append(" and devicename like :devicename")
        }
        param.areano?.let {
            querySql.append(" and areano=:areano")
            countSql.append(" and areano=:areano")

        }
        querySql.append(" order by id desc")
        val query = em.createNativeQuery(querySql.toString())
        val countQuery = em.createNativeQuery(countSql.toString())
        if (!StringUtil.isEmpty(param.accdate)) {
            val timerange = param.accdate.replace("-", "").split("  ")
            query.setParameter("minaccdate", timerange[0])
            query.setParameter("maxaccdate", timerange[1])
            countQuery.setParameter("minaccdate", timerange[0])
            countQuery.setParameter("maxaccdate", timerange[1])
        }
        if (!StringUtil.isEmpty(param.devicename)) {
            query.setParameter("devicename", "%" + param.devicename.trim() + "%")
            countQuery.setParameter("devicename", "%" + param.devicename.trim() + "%")
        }
        param.areano?.let {
            query.setParameter("areano", param.areano)
            countQuery.setParameter("areano", param.areano)
        }
        query.firstResult = (param.pageNo - 1) * param.pageSize
        query.maxResults = param.pageSize
        query.unwrap(NativeQueryImpl::class.java).setResultTransformer(Transformers.aliasToBean(TTransdtlCountVO::class.java))
        val list = query.resultList as List<TTransdtlCountVO>
        val count = countQuery.singleResult as BigInteger
        return PageResult(count.longValueExact(), list)
    }

    override fun queryTransdtlNotEntry(): List<TCollectdtl>? {
        return collectdtlDao.findWipCollectdtl(PageRequest.of(0, 10))
    }

    override fun findByCobillnoForUpdate(cobillno: Int?): TCollectdtl {
        return collectdtlDao.findByCobillnoForUpdate(cobillno) ?: throw RuntimeException("采集流水号:" + cobillno + "未找到")
    }

    override fun generateCountdtl() {
        val dtlcountDate = dtlCountDateDao.findCountDateForUpdate()
        val currentDate = systemUtilsService.sysdatetime.hostdate
        if (dtlcountDate.countdate.toInt() >= currentDate.toInt()) {
            logger.error { "统计流水日期<${dtlcountDate.countdate}>应小于当前日期<$currentDate>" }
            return
        }
        val count = daysBetween(dtlcountDate.countdate, currentDate)
        var accdate = dtlcountDate.countdate
        val querySql = "select t3.deviceno,t3.areano,coalesce(t4.accdate,:accdate) accdate,t4.mode,coalesce(t4.amount,0) amount,coalesce(t4.water,0) water,coalesce(t4.count,0) " +
                "count from tb_device t3 LEFT JOIN (select t1.deviceno,t1.accdate,t1.mode,t1.amount,t1.water,t1.count,t2.areano from (select deviceno,accdate,mode,sum(cast(amount as DECIMAL(18,2))) amount,sum(water_in_100ml) water,count(cobillno) count " +
                "from tb_collectdtl where accdate=:accdate and status = 'success' group by deviceno,mode,accdate) t1 LEFT JOIN " +
                "(select deviceno,areano from tb_device) t2 on t1.deviceno=t2.deviceno) t4 on t3.deviceno=t4.deviceno"
        for (i in 1..count) {
            if (accdate.toInt() >= currentDate.toInt()) {
                logger.error { "统计流水日期<$accdate>应小于当前日期<$currentDate>" }
                return
            }
            if (transdtlCountDao.countByAccdate(accdate)>0) {
                logger.error { accdate + "的统计流水已生成!" }
                return
            }
            val list = ArrayList<TTransdtlCount>(300)
            val query = em.createNativeQuery(querySql)
            query.setParameter("accdate", accdate)
            query.unwrap(NativeQueryImpl::class.java).setResultTransformer(Transformers.aliasToBean(TTransdtlCountDTO::class.java))
            val countDTOList = query.resultList as List<TTransdtlCountDTO>
            countDTOList.forEach {
                val transCount = TTransdtlCount()
                transCount.accdate = it.accdate
                transCount.amount = it.amount.toDouble()
                transCount.water = it.water.toInt()
                transCount.count = it.count.toInt()
                transCount.mode = it.mode
                transCount.areano = it.areano
                transCount.deviceno = it.deviceno
                list.add(transCount)
            }
            transdtlCountDao.saveAll(list)
            accdate = nextDay(accdate)
        }
        dtlcountDate.countdate=accdate
        dtlCountDateDao.save(dtlcountDate)
    }

    private fun daysBetween(date1: String, date2: String): Int {
        val sdf = SimpleDateFormat("yyyyMMdd")
        val cal = Calendar.getInstance()
        cal.time = sdf.parse(date1)
        val time1 = cal.timeInMillis
        cal.time = sdf.parse(date2)
        val time2 = cal.timeInMillis
        val between = (time2 - time1) / (1000 * 3600 * 24)
        return Integer.parseInt(between.toString())
    }

    private fun nextDay(date: String): String {
        val sdf = SimpleDateFormat("yyyyMMdd")
        val cal = Calendar.getInstance()
        cal.time = sdf.parse(date)
        cal.add(Calendar.DATE, 1)
        return sdf.format(cal.time)
    }

    override fun initDtlCountDate() {
        val dtlCountDate = dtlCountDateDao.findCountDateForUpdate()
        if (dtlCountDate == null) {
            val countDate = TdtlCountDate()
            countDate.id = 1
            countDate.countdate = systemUtilsService.sysdatetime.hostdate
            dtlCountDateDao.save(countDate)
        }
    }
}