package com.supwisdom.institute.backend.base.api.v1.controller.admin;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ConfigCreateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ConfigQueryRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ConfigUpdateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ConfigCreateResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ConfigLoadResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ConfigQueryResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ConfigUpdateResponseData;
import com.supwisdom.institute.backend.base.domain.entity.Config;
import com.supwisdom.institute.backend.base.domain.exception.ConfigException;
import com.supwisdom.institute.backend.base.domain.service.ConfigService;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;

@Api(value = "BaseAdminConfig", tags = { "BaseAdminConfig" }, description = "配置项的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/configs")
public class AdminConfigController {
  
  @Autowired
  private ConfigService configService;


  /**
   * @param configQueryRequest
   * @return
   */
  @ApiOperation(value = "查询配置列表", notes = "查询配置列表", nickname = "systemAdminConfigQuery")
  @ApiImplicitParams({
    @ApiImplicitParam(name = "loadAll", value = "是否加载全部", required = true, dataType = "boolean", paramType = "query", defaultValue = "false"),
    @ApiImplicitParam(name = "pageIndex", value = "分页 - 页码", required = true, dataType = "int", paramType = "query", defaultValue = "0", example = "0"),
    @ApiImplicitParam(name = "pageSize", value = "分页 - 每页记录数", required = true, dataType = "int", paramType = "query", defaultValue = "20", example = "20"),
    @ApiImplicitParam(name = "mapBean[deleted]", value = "查询条件 - 删除状态 (精确)", required = false, dataType = "boolean", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[categoryCode]", value = "查询条件 - 分类代码 (精确)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[categoryName]", value = "查询条件 - 分类名称 (模糊)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[name]", value = "查询条件 - 名称 (模糊)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[description]", value = "查询条件 - 描述 (模糊)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[configKey]", value = "查询条件 - 配置Key (精确)", required = false, dataType = "string", paramType = "query"),
  })
  @RequestMapping(method = RequestMethod.GET, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ConfigQueryResponseData> query(ConfigQueryRequest configQueryRequest) {
    
    Page<Config> page = configService.selectPageList(
        configQueryRequest.isLoadAll(),
        configQueryRequest.getPageIndex(),
        configQueryRequest.getPageSize(),
        configQueryRequest.getMapBean(),
        configQueryRequest.getOrderBy());

    ConfigQueryResponseData resp = ConfigQueryResponseData.of(configQueryRequest).build(page);
    
    return new DefaultApiResponse<ConfigQueryResponseData>(resp);
  }

  /**
   * @param id
   * @return
   */
  @ApiOperation(value = "根据ID获取配置项", notes = "根据ID获取配置项", nickname="systemAdminConfigLoad")
  @RequestMapping(method = RequestMethod.GET, path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ConfigLoadResponseData> load(
      @PathVariable("id") String id) {
    
    if (id == null || id.length() == 0) {
      throw new ConfigException().newInstance("exception.get.id.must.not.empty");
    }
    
    Config config = configService.selectById(id);
    
    if (config == null) {
      throw new ConfigException().newInstance("exception.get.domain.not.exist");
    }

    ConfigLoadResponseData resp = ConfigLoadResponseData.build(config);
    
    return new DefaultApiResponse<ConfigLoadResponseData>(resp);
  }
  
  /**
   * @param configCreateRequest
   * @return
   */
  @ApiOperation(value = "创建配置项", notes = "创建配置项", nickname = "systemAdminConfigCreate")
  @RequestMapping(method = RequestMethod.POST, consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.CREATED)
  @ResponseBody
  public DefaultApiResponse<ConfigCreateResponseData> create(
      @RequestBody ConfigCreateRequest configCreateRequest) {

    // FIXME: 验证数据有效性

    Config entity = configCreateRequest.getEntity();

    Config ret = configService.insert(entity);

    ConfigCreateResponseData resp = ConfigCreateResponseData.build(ret);

    return new DefaultApiResponse<ConfigCreateResponseData>(resp);
  }

  @ApiOperation(value = "更新配置项", notes = "更新配置项", nickname = "systemAdminConfigUpdate")
  @RequestMapping(method = RequestMethod.PUT, consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ConfigUpdateResponseData> update(
      @PathVariable("id") String id,
      @RequestBody ConfigUpdateRequest configUpdateRequest) {

    if (id == null || id.length() == 0) {
      throw new ConfigException().newInstance("exception.update.id.must.not.empty");
    }

    Config tmp = configService.selectById(id);
    if (tmp == null) {
      throw new ConfigException().newInstance("exception.update.domain.not.exist");
    }

    if (!tmp.getEditable().booleanValue()) {
      throw new ConfigException().newInstance("exception.editable.can.not.update");
    }

    Config entity = configUpdateRequest.getEntity();
    entity.setId(id);

    entity = EntityUtils.merge(tmp, entity);
    
//    if (tmp.getEditable().booleanValue() != entity.getEditable().booleanValue()) {
//      throw new ConfigException().newInstance("exception.editable.can.not.update");
//    }

    entity.setEditable(true);  // 防止 可修改记录的 editable 被置为false

    Config ret = configService.update(entity);

    ConfigUpdateResponseData resp = ConfigUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<ConfigUpdateResponseData>(resp);
  }

  
  /**
   * @param categoryCode
   * @param configKey
   * @return
   */
  @ApiOperation(value = "根据 categoryCode、configKey 获取配置项", notes = "根据 categoryCode、configKey 获取配置项", nickname = "systemAdminConfigLoadByCategoryKey")
  @RequestMapping(method = RequestMethod.GET, path = "/loadByCategoryKey", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ConfigLoadResponseData> loadByCategoryKey(
      @RequestParam("categoryCode") String categoryCode, 
      @RequestParam("configKey") String configKey
      ) {
    
    if (categoryCode == null || categoryCode.length() == 0) {
      throw new ConfigException().newInstance("exception.load.params.must.not.empty");
    }
    if (configKey == null || configKey.length() == 0) {
      throw new ConfigException().newInstance("exception.load.params.must.not.empty");
    }
    
    Config config = configService.selectByCategoryKey(categoryCode, configKey);
    
    if (config == null) {
      throw new ConfigException().newInstance("exception.load.domain.not.exist");
    }

    ConfigLoadResponseData resp = ConfigLoadResponseData.build(config);
    
    return new DefaultApiResponse<ConfigLoadResponseData>(resp);
  }

}
