package com.supwisdom.institute.backend.common.framework.repo.resultTransformer;

import com.google.common.collect.Lists;
import com.supwisdom.institute.backend.common.util.ReflectUtils;

import org.hibernate.HibernateException;
import org.hibernate.transform.ResultTransformer;

import java.lang.reflect.Field;
import java.util.List;

/**
 * 修正hibernate返回自定义pojo类型时找不到属性的BUG
 * 主要发生在使用oracle或高版本的mysql时，查询返回的字段默认是大写的(除非SQL中指定了别名)，这导致返回自定义pojo类型时会报找不到属性的错误，该类用于修正此BUG。
 * 使用该类时SQL返回的字段名大小写或者带"_"都会被忽略，如数据库字段为 USER_NAME，自定义pojo的属性名为username就可以使用  
 * 
 * @author feng
 */
public class IgnoreCaseResultTransformer implements ResultTransformer {
  private static final long serialVersionUID = -3779317531110592988L;
  private final Class<?> resultClass;
  private Field[] fields;
  private List<Class<?>> types = Lists.newArrayList();

  public IgnoreCaseResultTransformer(final Class<?> resultClass) {
    this.resultClass = resultClass;
    List<Field> list = Lists.newArrayList();
    for (Class<?> superClass = resultClass; superClass != Object.class; superClass = superClass.getSuperclass()) {
      Field[] fs = superClass.getDeclaredFields();
      List<Field> newFs = Lists.newArrayList();
      for (int i = 0; i < fs.length; i++) {
        if (fs[i].getName().equals("serialVersionUID")) {
          continue;
        }
        types.add(fs[i].getType());
        ReflectUtils.makeAccessible(fs[i]);
        newFs.add(fs[i]);
      }
      list.addAll(newFs);
    }
    this.fields = list.toArray(new Field[list.size()]);
  }

  /**
   * aliases为每条记录的数据库字段名,ORACLE字段名默认为大写
   * tupe为与aliases对应的字段的值      
   */
  @Override
  public Object transformTuple(final Object[] tuple, final String[] aliases) {
    Object result;
    try {
      result = this.resultClass.newInstance();
      for (int i = 0; i < aliases.length; i++) {
        for (int j = 0; j < this.fields.length; j++) {
          String fieldName = this.fields[j].getName();
          // 数据库字段带下划线的时候也能保证使用，如数据库字段为 USER_NAME，自定义pojo的属性名为username就可以使用
          if (fieldName.equalsIgnoreCase(aliases[i].replaceAll("_", ""))) {
            ReflectUtils.invokeSetter(result, fieldName, tuple[i], this.types.get(j));
            // beanUtilsBean.setProperty(result, fieldName, tuple[i]);
            break;
          }
        }
      }
    } catch (Exception e) {
      throw new HibernateException("Could not instantiate resultclass: " + this.resultClass.getName(), e);
    }
    return result;
  }

  @Override
  @SuppressWarnings("rawtypes")
  public List transformList(final List collection) {
    return collection;
  }
}
