package com.supwisdom.institute.backend.thirdparty.poa;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.BoundSetOperations;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;

public class RedisUtils<T> {
  
  private RedisUtils() {
    this.redisTemplate = null;
  }
  
  private RedisUtils(RedisTemplate<String, T> redisTemplate) {
    this.redisTemplate = redisTemplate;
  }
  
  private final RedisTemplate<String, T> redisTemplate;
  
  public static <T> RedisUtils<T> redisTemplate(RedisTemplate<String, T> redisTemplate) {
    RedisUtils<T> redisUtils = new RedisUtils<T>(redisTemplate);
    
    return redisUtils;
  }
  


  /**
   * Set the value of a key
   * @param redisKey
   * @param expireTime, time in seconds
   * @param t
   */
  public void setValue(String redisKey, Long expireTime, T t) {
    BoundValueOperations<String, T> boundValue = this.redisTemplate.boundValueOps(redisKey);
    boundValue.set(t);
    
    if (expireTime != null) {
      boundValue.expire(expireTime, TimeUnit.SECONDS);
    }
  }
  
  /**
   * Get the value of a key
   * @param redisKey
   * @return
   */
  public T getValue(String redisKey) {
    BoundValueOperations<String, T> boundValue = this.redisTemplate.boundValueOps(redisKey);
    if (boundValue == null) {
      return null;
    }
    
    return boundValue.get();
  }
  
  /**
   * Expire the value of a key
   * @param redisKey
   */
  public void expireValue(String redisKey) {
    BoundValueOperations<String, T> boundValue = this.redisTemplate.boundValueOps(redisKey);
    if (boundValue == null) {
      return;
    }
    
    boundValue.expire(-1L, TimeUnit.SECONDS);
  }
  
  public List<T> getAllValue(String patternRedisKey) {
    return this.redisTemplate.keys(patternRedisKey)
        .stream()
        .map(redisKey -> this.redisTemplate.boundValueOps(redisKey).get())
        .filter(Objects::nonNull).collect(Collectors.toList());
  }
  
  
  /**
   * SADD key member
   * Add a member ({@link T}) to a set
   * @param redisKey
   * @param expireTime
   * @param member
   */
  public void addToSet(String redisKey, Long expireTime, T member) {
    BoundSetOperations<String, T> boundSet = this.redisTemplate.boundSetOps(redisKey);
    boundSet.add(member);
    
    if (expireTime != null) {
      boundSet.expire(expireTime, TimeUnit.SECONDS);
    }
  }
  
  /**
   * SREM key member
   * Remove a member ({@link T}) from a set
   * @param redisKey
   * @param member
   */
  public void remFromSet(String redisKey, T member) {
    BoundSetOperations<String, T> boundSet = this.redisTemplate.boundSetOps(redisKey);
    if (boundSet == null) {
      return;
    }
    
    boundSet.remove(member);
    
    if (boundSet.size() == 0) {
      boundSet.expire(-1L, TimeUnit.SECONDS);
    }
  }
  
  public void expireSet(String redisKey) {
    BoundSetOperations<String, T> boundSet = this.redisTemplate.boundSetOps(redisKey);
    if (boundSet == null) {
      return;
    }
    
    boundSet.expire(-1L, TimeUnit.SECONDS);
  }
  
  
  /**
   * HSET key field value
   * Set the string value of a hash field
   * @param redisKey
   * @param expireTime
   * @param field
   * @param value
   */
  public void setToHash(String redisKey, Long expireTime, String field, T value) {
    BoundHashOperations<String, String, T> boundHash = this.redisTemplate.boundHashOps(redisKey);
    boundHash.put(field, value);
    
    if (expireTime != null) {
      boundHash.expire(expireTime, TimeUnit.SECONDS);
    }
  }
  
  /**
   * HGETALL key
   * Get all the fields and values in a hash
   * @param redisKey
   * @return
   */
  public Map<String, T> getAllFromHash(String redisKey) {
    BoundHashOperations<String, String, T> boundHash = this.redisTemplate.boundHashOps(redisKey);
    if (boundHash == null) {
      return null;
    }
    
    return boundHash.entries();
  }
  
  /**
   * HGET key field
   * summary: Get the value of a hash field
   * @param redisKey
   * @param field
   * @return
   */
  public T getFromHash(String redisKey, String field) {
    BoundHashOperations<String, String, T> boundHash = this.redisTemplate.boundHashOps(redisKey);
    if (boundHash == null) {
      return null;
    }
    
    return boundHash.get(field);
  }
  
  /**
   * HDEL key field
   * summary: Delete a hash field
   * @param redisKey
   * @param field
   */
  public void delFromHash(String redisKey, String field) {
    BoundHashOperations<String, String, T> boundHash = this.redisTemplate.boundHashOps(redisKey);
    if (boundHash == null) {
      return;
    }
    
    boundHash.delete(field);
    
    if (boundHash.size() == 0) {
      boundHash.expire(-1L, TimeUnit.SECONDS);
    }
  }
  
  public void expireHash(String redisKey) {
    BoundHashOperations<String, String, T> boundHash = this.redisTemplate.boundHashOps(redisKey);
    if (boundHash == null) {
      return;
    }
    
    boundHash.expire(-1L, TimeUnit.SECONDS);
  }
  
}
