package com.supwisdom.institute.backend.base.api.v1.controller.admin;

import io.swagger.annotations.Api;

import java.util.HashMap;

import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.base.api.v1.vo.request.PermissionCreateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.request.PermissionQueryRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.request.PermissionUpdateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.response.PermissionCreateResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.PermissionLoadResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.PermissionQueryResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.PermissionRemoveResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.PermissionUpdateResponseData;
import com.supwisdom.institute.backend.base.domain.entity.Permission;
import com.supwisdom.institute.backend.base.domain.service.PermissionService;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;

@Api(value = "BaseAdminOperation", tags = { "BaseAdminOperation" }, description = "操作的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/operations")
public class AdminOperationController {

  @Autowired
  private PermissionService permissionService;

  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<PermissionQueryResponseData> query(PermissionQueryRequest queryRequest) {
    
    if (queryRequest.getMapBean() == null) {
      queryRequest.setMapBean(new HashMap<String, Object>());
    }
    queryRequest.getMapBean().put("type", Permission.TYPE_OPERATION);

    Page<Permission> page = permissionService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    PermissionQueryResponseData data = PermissionQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<PermissionQueryResponseData>(data);
  }
  
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<PermissionLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Permission permission = permissionService.selectById(id);

    if (permission == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    if (!Permission.TYPE_OPERATION.equals(permission.getType())) {
      throw new RuntimeException("exception.get.domain.type.error");
    }
    
    PermissionLoadResponseData data = PermissionLoadResponseData.of(permission);

    return new DefaultApiResponse<PermissionLoadResponseData>(data);
  }

  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<PermissionCreateResponseData> create(
      @RequestBody PermissionCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Permission permission = createRequest.getEntity();
    permission.setType(Permission.TYPE_OPERATION);
    
    Permission ret = permissionService.insert(permission);
    
    PermissionCreateResponseData data = PermissionCreateResponseData.build(ret);

    return new DefaultApiResponse<PermissionCreateResponseData>(data);
  }
  
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<PermissionUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody PermissionUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Permission tmp = permissionService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }
    if (!Permission.TYPE_OPERATION.equals(tmp.getType())) {
      throw new RuntimeException("exception.update.domain.type.error");
    }

    Permission permission = updateRequest.getEntity();
    permission.setId(id);

    permission = EntityUtils.merge(tmp, permission);

    Permission ret = permissionService.update(permission);

    PermissionUpdateResponseData data = PermissionUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<PermissionUpdateResponseData>(data);
  }

  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<PermissionRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Permission tmp = permissionService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }
    if (!Permission.TYPE_OPERATION.equals(tmp.getType())) {
      throw new RuntimeException("exception.delete.domain.type.error");
    }

    permissionService.deleteById(id);

    PermissionRemoveResponseData data = PermissionRemoveResponseData.build(tmp);
    return new DefaultApiResponse<PermissionRemoveResponseData>(data);
  }

}
