package com.supwisdom.institute.backend.base.api.v1.controller.admin;

import java.util.List;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.base.api.v1.vo.request.RouteCreateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.request.RouteDeleteBatchRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.request.RouteQueryRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.request.RouteUpdateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.response.RouteCreateResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.RouteDeleteBatchResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.RouteLoadResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.RouteQueryResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.RouteRemoveResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.response.RouteUpdateResponseData;
import com.supwisdom.institute.backend.base.domain.entity.Route;
import com.supwisdom.institute.backend.base.domain.service.RouteService;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;

@Api(value = "BaseAdminRoute", tags = { "BaseAdminRoute" }, description = "路由信息的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/routes")
public class AdminRouteController {

  @Autowired
  private RouteService routeService;
  

  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RouteQueryResponseData> query(RouteQueryRequest queryRequest) {

    Page<Route> page = routeService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    RouteQueryResponseData data = RouteQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<RouteQueryResponseData>(data);
  }
  
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RouteLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Route route = routeService.selectById(id);

    if (route == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    
    RouteLoadResponseData data = RouteLoadResponseData.of(route);

    return new DefaultApiResponse<RouteLoadResponseData>(data);
  }

  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RouteCreateResponseData> create(
      @RequestBody RouteCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Route route = createRequest.getEntity();
    
    Route ret = routeService.insert(route);
    
    RouteCreateResponseData data = RouteCreateResponseData.build(ret);

    return new DefaultApiResponse<RouteCreateResponseData>(data);
  }
  
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RouteUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody RouteUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Route tmp = routeService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }

    Route route = updateRequest.getEntity();
    route.setId(id);

    route = EntityUtils.merge(tmp, route);

    Route ret = routeService.update(route);

    RouteUpdateResponseData data = RouteUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<RouteUpdateResponseData>(data);
  }

  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RouteRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Route tmp = routeService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }

    routeService.deleteById(id);

    RouteRemoveResponseData data = RouteRemoveResponseData.build(tmp);
    return new DefaultApiResponse<RouteRemoveResponseData>(data);
  }

  @DeleteMapping(path = "/batch", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RouteDeleteBatchResponseData> deleteBatch(
      @RequestBody RouteDeleteBatchRequest deleteBatchRequest) {
    
    List<String> ids = deleteBatchRequest.getIds();
    
    routeService.deleteBatch(ids);
    
    RouteDeleteBatchResponseData data = RouteDeleteBatchResponseData.build(ids);
    return new DefaultApiResponse<RouteDeleteBatchResponseData>(data);
  }

}
